
{*******************************************************}
{                                                       }
{       SSPI based components implementing              }
{       basic SSPI functionality and                    }
{       specifically connection based protocols         }
{       like NTLM, Kerberos, Negotiate                  }
{                                                       }
{       Copyright (c) 2000-2001, Eventree Systems       }
{                                                       }
{*******************************************************}

unit xsspi;

{$DEFINE SET_ENCRYPT_IN_FT_WITH_GETPROCADDRESS_FUDGE}
{$DEFINE USE_SEAL_INSTEAD_OF_SET_ENCRYPT_IN_FT_WITH_GETPROCADDRESS_FUDGE}

interface

uses

  Windows, classes, SysUtils,
  winerror, sspi, XWindows;

type

  { ESSPIException }

  ESSPIException = class(ECustomSysError)

  protected
    function GetSysErrorTypeStr: String; override;
  public
    class function GetErrorMessageByNo(aErrorNo: LongWord): String; override;

  public
    constructor CreateError(
      aFailedFuncName: String; anErrorNo: Longint = SEC_E_OK);

  end;

  ESSPIInterfaceInitFailed = class(ESSPIException)

  end;

  { TRefreshableSSPIObject }

  TRefreshableSSPIObject = class(TObject)

  private
    fUpdatePending, fRefreshing: Boolean;

  protected
    procedure RefreshIfUpdatePending;
    procedure SetUpdatePending; virtual;
    property UpdatePending: Boolean read fUpdatePending;
    property Refreshing: Boolean read fRefreshing;

  protected
    procedure Update; virtual;

  public
    constructor Create;

  end;

  TRefreshableSSPIObjectList = class(TRefreshableSSPIObject)

  private
    fList: TStringList;
    function getCount: Integer;
    function getObjects(idx: Integer): TObject;
  protected
    procedure Clear; virtual;
    procedure AddObject(o: TObject);
    property Objects[idx: Integer]: TObject read getObjects;
    property Count: Integer read getCount;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  { TSSPIInterface }

  TSSPIInterface = class(TObject)

  private
    fLoadPending, fIsAvailable: Boolean;
    fPFunctionTable: PSecurityFunctionTableA;
    dll: TDLLLoader;
    procedure releaseFunctionTable;
    procedure checkAvailable;
    function getFunctionTable: SecurityFunctionTableA;
    function getVersion: ULONG;
  public
    class procedure RaiseIfError(
      aStatus: SECURITY_STATUS; aFunctionName: String);
    function IsAvailable: Boolean;
    property FunctionTable: SecurityFunctionTableA read getFunctionTable;
    property Version: ULONG read getVersion;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  function SSPIInterface: TSSPIInterface;

type

  { TSSPIPackages }

  TSSPIPackage = class(TObject)

  private
    fPSecPkginfo: PSecPkgInfo;
    function getPSecPkgInfo: PSecPkgInfo;
    function getCapabilities: ULONG;
    function getVersion: Word;
    function getRPCID: Word;
    function getMaxToken: ULONG;
    function getName: String;
    function getComment: String;
  public
    property Capabilities: ULONG read getCapabilities;
    property Version: Word read getVersion;
    property RPCID: Word read getRPCID;
    property MaxToken: ULONG read getMaxToken;
    property Name: String read getName;
    property Comment: String read getComment;

  public
    constructor Create(aPSecPkginfo: PSecPkgInfo);

  end;

  TSSPIPackages = class(TRefreshableSSPIObjectList)

  private
    fPSecPkgInfoList: PSecPkgInfo;
    function loadItems: ULONG;
    procedure releaseItems;
  protected
    procedure Update; override;

  private
    function getItems(idx: Integer): TSSPIPackage;
  public
    function IndexOf(aName: String): Integer;
    property Items[idx: Integer]: TSSPIPackage read getItems; default;
    property Count;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  TCustomSSPIPackage = class(TSSPIPackage)

  private
    fInfo: PSecPkgInfo;
  public
    constructor Create(aPkgName: String);
    destructor Destroy; override;

  end;

  TSSPINTLMPackage = class(TCustomSSPIPackage)

  public
    constructor Create;

  end;

  TSSPIKerberosPackage = class(TCustomSSPIPackage)

  public
    constructor Create;

  end;

  TSSPINegotiatePackage = class(TCustomSSPIPackage)

  public
    constructor Create;

  end;

  { TCustomSSPIAttribute }

  TCustomSSPIAttribute = class(TRefreshableSSPIObject)

  private
    fReleaseSSPIOwnedMemoryPending: Boolean;
    procedure releaseBuffers;
  protected
    procedure Update; override;
    function GetPBuffer: PVOID; virtual; abstract;
    procedure DoQuery(aPBuffer: PVOID); virtual; abstract;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); virtual;
    procedure ReleaseContextBuffer(aPBuffer: PVOID);
    function AsPVOID: PVOID;
  public
    class function Tag: ULONG; virtual; abstract;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  { TSSPICredentials }

  TSSPICredentials = class;

  TCustomSSPICredentialsAttribute = class(TCustomSSPIAttribute)

  protected
    procedure DoQuery(aPBuffer: PVOID); override;

  private
    fParent: TSSPICredentials;
  public
    constructor Create(aParent: TSSPICredentials);

  end;

  TSSPICredentialsAttributeNames = class(TCustomSSPICredentialsAttribute)

  private
    fBuffer: SecPkgCredentials_Names;
    function getUserName: String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property UserName: String read getUserName;

  end;

  TSSPICredentialsAttributes = class(TObject)

  private
    fNames: TSSPICredentialsAttributeNames;
  public
    procedure Clear; virtual;
    property Names: TSSPICredentialsAttributeNames read fNames;

  public
    constructor Create(aParent: TSSPICredentials);
    destructor Destroy; override;

  end;

  TSSPICredentialsUse = (
    scuInBound, scuOutBound, scuBoth
  );

  TSSPICredentials = class(TObject)

  private
    fPackage: TSSPIPackage;
    fHandle: CredHandle;
    fUse: TSSPICredentialsUse;
    fAcquired: Boolean;
    fExpiry: TimeStamp;
    function getHandle: PCredHandle;
    procedure setUse(aValue: TSSPICredentialsUse);
    function getExpiry: TimeStamp;
  protected
    procedure CheckAcquired;
    procedure CheckNotAcquired;
    procedure DoAcquire(pszPrincipal: PSEC_CHAR; pvLogonId, pAuthData: PVOID);
    procedure DoRelease; virtual;
  public
    procedure Release;
    property Package: TSSPIPackage read fPackage;
    property Handle: PCredHandle read getHandle;
    property Use: TSSPICredentialsUse read fUse write setUse;
    property Acquired: Boolean read fAcquired;
    property Expiry: TimeStamp read getExpiry;

  public
    constructor Create(aPackage: TSSPIPackage);
    destructor Destroy; override;

  end;

  { TSSPIWinNTCredentials }

  TSSPIWinNTCredentials = class(TSSPICredentials)

  protected
    procedure DoRelease; override;

  public
    procedure Acquire(
      aUse: TSSPICredentialsUse); overload;
    procedure Acquire(
      aUse: TSSPICredentialsUse; aDomain,
      aUserName, aPassword: String); overload;

  private
    fAttributes: TSSPICredentialsAttributes;
  public
    property Attributes: TSSPICredentialsAttributes read fAttributes;

  public
    constructor Create(aPackage: TSSPIPackage);
    destructor Destroy; override;

  end;

  { TSSPIContextRequirements }

  TSSPIContextRequirementsValue = (
    scrDelegation, scrMutualAuth, scrReplayDetect, scrSequenceDetect,
    scrConfidentiality, scrUseSessionKey,
    scrPromptForCreds, scrUseSuppliedCreds,
    scrAllocateMemory, scrUseDCEStyle,
    scrDatagram, scrConnection, scrStream,
    scrExtednedError, scrIntegrity, scrIdentify,
    scrManualCredValidation
  );
  TSSPIContextRequirementsValues = set of TSSPIContextRequirementsValue;

  TSSPIContextRequirement = class(TObject)

  protected
    class function IsServer: Boolean; virtual; abstract;
    class function IsEstablished: Boolean; virtual; abstract;
    class function ULONGToValue(aULONG: ULONG): TSSPIContextRequirementsValues;

  private
    fULONG: ULONG;
  protected
    function GetValue: TSSPIContextRequirementsValues;
    function GetAsULONG: ULONG; virtual;
    procedure SetAsULONG(aValue: ULONG);
  public
    property AsULONG: ULONG read GetAsULONG;

  end;

  TSSPIContextRequirementEstablished = class(TSSPIContextRequirement)

  protected
    class function IsEstablished: Boolean; override;

  public
    property Value: TSSPIContextRequirementsValues read GetValue;

  end;

  TSSPIContext = class;

  TSSPIContextRequirementRequested = class(TSSPIContextRequirement)

  private
    fMinimumReq: ULONG;
  protected
    class function IsEstablished: Boolean; override;
    function GetAsULONG: ULONG; override;

  private
    procedure setValue(aValue: TSSPIContextRequirementsValues);
  protected
    procedure SetValueWhileContextIsActive(
      aValue: TSSPIContextRequirementsValues);
  public
    property Value: TSSPIContextRequirementsValues read GetValue write setValue;

  private
    fContext: TSSPIContext;
  public
    constructor Create(aContext: TSSPIContext; aMinimumReq: ULONG);

  end;

  TSSPIClientContextRequirementRequested
      = class(TSSPIContextRequirementRequested)

  protected
    class function IsServer: Boolean; override;

  end;

  TSSPIClientContextRequirementEstablished
      = class(TSSPIContextRequirementEstablished)

  protected
    class function IsServer: Boolean; override;

  end;

  TSSPIServerContextRequirementRequested
      = class(TSSPIContextRequirementRequested)

  protected
    class function IsServer: Boolean; override;

  end;

  TSSPIServerContextRequirementEstablished
      = class(TSSPIContextRequirementEstablished)

  protected
    class function IsServer: Boolean; override;

  end;

  TSSPICustomContextRequirements = class(TObject)

  protected
    procedure SetEstablishedAsULong(aValue: ULONG); virtual; abstract;
    property EstablishedAsULong: ULONG write SetEstablishedAsULong;

  protected
    procedure SetRequestedValueWhileContextIsActive(
      aValue: TSSPIContextRequirementsValues); virtual; abstract;
    property RequestedValueWhileContextIsActive: TSSPIContextRequirementsValues
      write SetRequestedValueWhileContextIsActive;

  end;

  TSSPIClientContextRequirements = class(TSSPICustomContextRequirements)

  private
    fRequested: TSSPIClientContextRequirementRequested;
    fEstablished: TSSPIClientContextRequirementEstablished;
  protected
    procedure SetEstablishedAsULong(aValue: ULONG); override;
    procedure SetRequestedValueWhileContextIsActive(
      aValue: TSSPIContextRequirementsValues); override;
  public
    property Established: TSSPIClientContextRequirementEstablished
      read fEstablished;
    property Requested: TSSPIClientContextRequirementRequested read fRequested;

  public
    constructor Create(aContext: TSSPIContext; aMinimumReq: ULONG);
    destructor Destroy; override;

  end;

  TSSPIServerContextRequirements = class(TSSPICustomContextRequirements)

  private
    fRequested: TSSPIServerContextRequirementRequested;
    fEstablished: TSSPIServerContextRequirementEstablished;
  protected
    procedure SetEstablishedAsULong(aValue: ULONG); override;
    procedure SetRequestedValueWhileContextIsActive(
      aValue: TSSPIContextRequirementsValues); override;
  public
    property Established: TSSPIServerContextRequirementEstablished
      read fEstablished;
    property Requested: TSSPIServerContextRequirementRequested read fRequested;

  public
    constructor Create(aContext: TSSPIContext; aMinimumReq: ULONG);
    destructor Destroy; override;

  end;

  { TSSPIContextAttributes }

  TCustomSSPIContextAttribute = class(TCustomSSPIAttribute)

  protected
    procedure DoQuery(aPBuffer: PVOID); override;

  private
    fParent: TSSPIContext;
  public
    constructor Create(aParent: TSSPIContext);

  end;

  TSSPIContextAttributeSizes = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_Sizes;
    function getULONG(idx: Integer): ULONG;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property MaxToken: ULONG index 1 read getULONG;
    property MaxSignature: ULONG index 2 read getULONG;
    property BlockSize: ULONG index 3 read getULONG;
    property SecurityTrailer: ULONG index 4 read getULONG;

  end;

  TSSPIContextAttributeNames = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_Names;
    function getUserName: String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property UserName: String read getUserName;

  end;

  TSSPIContextAttributeLifespan = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_Lifespan;
    function getTimeStamp(idx: Integer): TimeStamp;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property Start: TimeStamp index 1 read getTimeStamp;
    property Expiry: TimeStamp index 2 read getTimeStamp;

  end;

  TSSPIContextAttributeDCEInfo = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_DceInfo;
    function getAuthzSvc: ULONG;
    function getPac: PVOID;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property AuthzSvc: ULONG read getAuthzSvc;
    property Pac: PVOID read getPac;

  end;

  TSSPIContextAttributeStreamSizes = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_StreamSizes;
    function getULONG(idx: Integer): ULONG;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property Header: ULONG index 1 read getULONG;
    property Trailer: ULONG index 2 read getULONG;
    property MaximumMessage: ULONG index 3 read getULONG;
    property Buffers: ULONG index 4 read getULONG;
    property BlockSize: ULONG index 5 read getULONG;

  end;

  TSSPIContextAttributeKeyInfo = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_KeyInfo;
    function getULONG(idx: Integer): ULONG;
    function getString(idx: Integer): String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property SignatureAlgorithmName: String index 1 read getString;
    property EncryptAlgorithmName: String index 2 read getString;
    property KeySize: ULONG index 1 read getULONG;
    property SignatureAlgorithm: ULONG index 2 read getULONG;
    property EncryptAlgorithm: ULONG index 3 read getULONG;

  end;

  TSSPIContextAttributeAuthority = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_Authority;
    function getAuthorityName: String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property AuthorityName: String read getAuthorityName;

  end;

  TSSPIContextAttributeProtoInfo = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_ProtoInfo;
    function getULONG(idx: Integer): ULONG;
    function getProtocolName: String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property ProtocolName: String read getProtocolName;
    property MajorVersion: ULONG index 1 read getULONG;
    property MinorVersion: ULONG index 2 read getULONG;

  end;

  TSSPIContextAttributePasswordExpiry = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_PasswordExpiry;
    function getPasswordExpires: TimeStamp;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property PasswordExpires: TimeStamp read getPasswordExpires;

  end;

  TSSPIContextAttributeSessionKey = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_SessionKey;
    function getSessionKeyLength: ULONG;
    function getSessionKey: PUCHAR;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property SessionKeyLength: ULONG read getSessionKeyLength;
    property SessionKey: PUCHAR read getSessionKey;

  end;

  TSSPIContextAttributePackageInfo = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_PackageInfo;
    function getPackageInfo: PSecPkgInfoA;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property PackageInfo: PSecPkgInfoA read getPackageInfo;

  end;

  TSSPIContextAttributeUserFlags = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_UserFlags;
    function getUserFlags: ULONG;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property UserFlags: ULONG read getUserFlags;

  end;

  TSSPIContextAttributeNegotiationInfo = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_NegotiationInfo;
    function getPackageInfo: PSecPkgInfoA;
    function getNegotiationState: ULONG;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property PackageInfo: PSecPkgInfoA read getPackageInfo;
    property NegotiationState: ULONG read getNegotiationState;

  end;

  TSSPIContextAttributeNativeNames = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_NativeNames;
    function getString(idx: Integer): String;
  protected
    function GetPBuffer: PVOID; override;
    procedure ReleaseSSPIOwnedMemory(aPBuffer: PVOID); override;
  public
    class function Tag: ULONG; override;
    property ClientName: String index 1 read getString;
    property ServerName: String index 2 read getString;

  end;

  TSSPIContextAttributeFlags = class(TCustomSSPIContextAttribute)

  private
    fBuffer: SecPkgContext_Flags;
    function getFlags: ULONG;
  protected
    function GetPBuffer: PVOID; override;
  public
    class function Tag: ULONG; override;
    property Flags: ULONG read getFlags;

  end;

  TSSPIContextAttributes = class(TObject)

  private
    fSizes: TSSPIContextAttributeSizes;
    fNames: TSSPIContextAttributeNames;
    fLifespan: TSSPIContextAttributeLifespan;
    fDCEInfo: TSSPIContextAttributeDCEInfo;
    fStreamSizes: TSSPIContextAttributeStreamSizes;
    fKeyInfo: TSSPIContextAttributeKeyInfo;
    fAuthority: TSSPIContextAttributeAuthority;
    fProtoInfo: TSSPIContextAttributeProtoInfo;
    fPasswordExpiry: TSSPIContextAttributePasswordExpiry;
    fSessionKey: TSSPIContextAttributeSessionKey;
    fPackageInfo: TSSPIContextAttributePackageInfo;
    fUserFlags: TSSPIContextAttributeUserFlags;
    fNegotiationInfo: TSSPIContextAttributeNegotiationInfo;
    fNativeNames: TSSPIContextAttributeNativeNames;
    fFlags: TSSPIContextAttributeFlags;
  public
    procedure Clear; virtual;
    property Sizes: TSSPIContextAttributeSizes read fSizes;
    property Names: TSSPIContextAttributeNames read fNames;
    property Lifespan: TSSPIContextAttributeLifespan read fLifespan;
    property DCEInfo: TSSPIContextAttributeDCEInfo read fDCEInfo;
    property StreamSizes: TSSPIContextAttributeStreamSizes read fStreamSizes;
    property KeyInfo: TSSPIContextAttributeKeyInfo read fKeyInfo;
    property Authority: TSSPIContextAttributeAuthority read fAuthority;
    property ProtoInfo: TSSPIContextAttributeProtoInfo read fProtoInfo;
    property PasswordExpiry: TSSPIContextAttributePasswordExpiry
      read fPasswordExpiry;
    property SessionKey: TSSPIContextAttributeSessionKey read fSessionKey;
    property PackageInfo: TSSPIContextAttributePackageInfo read fPackageInfo;
    property UserFlags: TSSPIContextAttributeUserFlags read fUserFlags;
    property NegotiationInfo: TSSPIContextAttributeNegotiationInfo
      read fNegotiationInfo;
    property NativeNames: TSSPIContextAttributeNativeNames read fNativeNames;
    property Flags: TSSPIContextAttributeFlags read fFlags;

  public
    constructor Create(aParent: TSSPIContext);
    destructor Destroy; override;

  end;

  { TSSPIContext }

  TSSPIContext = class(TObject)

  private
    fCredentials: TSSPICredentials;
    fHandle: CtxtHandle;
    fHasHandle: Boolean;
    fExpiry: TimeStamp;
    function getHandle: PCtxtHandle;
    function getExpiry: TimeStamp;
    procedure updateHasContextAndCheckForError(
      const aFuncResult: SECURITY_STATUS; const aFuncName: String;
      const aErrorsToIgnore: array of SECURITY_STATUS);
  protected
    procedure CheckHasHandle;
    procedure CheckCredentials;
    procedure SetCredentials(aCredentials: TSSPICredentials);
    function DoInitialize(
      aTokenSourceName: PChar;
      var aIn, aOut: SecBufferDesc;
      const errorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
    function DoAccept(
      var aIn, aOut: SecBufferDesc;
      const errorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
    procedure DoRelease; virtual;
    function GetRequestedFlags: ULONG; virtual; abstract;
    procedure SetEstablishedFlags(aFlags: ULONG); virtual; abstract;
    function GetAuthenticated: Boolean; virtual; abstract;
    property HasHandle: Boolean read fHasHandle;
  public
    procedure Release;
    property Credentials: TSSPICredentials read fCredentials;
    property Handle: PCtxtHandle read getHandle;
    property Authenticated: Boolean read GetAuthenticated;
    property Expiry: TimeStamp read getExpiry;

  protected
    function DoEncrypt(
      var aMsg: SecBufferDesc; aMsgNo, aQOP: ULONG;
      const aErrorsToIgnore: array of SECURITY_STATUS
    ): SECURITY_STATUS;
    function DoDecrypt(
      var aMsg: SecBufferDesc; aMsgNo: ULONG; var aQOP: ULONG;
      const aErrorsToIgnore: array of SECURITY_STATUS
    ): SECURITY_STATUS;
(*
  { TODO : implement SignStr and VerifyStr versions }
  public
    procedure Sign(
      const aMsgBuffer: PByte; const aMsgCount: Longint;
      var aSignatureBuffer: PByte; var aSignatureCount: Longint);
//    function SignStr(const aMsg: String): String;
    function Verify(
      const aMsgBuffer: PByte; const aMsgCount: Longint;
      const aSignatureBuffer: PByte; const aSignatureCount: Longint): Boolean;
//    function VerifyStr(const aMsg, aSignature: String): Boolean;
*)

  { TODO: maybe we should call RevertToSelf in Destroy if Impersonating }
  { TODO: implement QuerySecurityContextToken }
  private
    fImpersonating: Boolean;
    function getImpersonating: Boolean;
  public
    procedure Impersonate;
    procedure RevertToSelf;
    property Impersonating: Boolean read getImpersonating;

  public
    constructor Create(aCredentials: TSSPICredentials);
    destructor Destroy; override;

  end;

  { TSSPIConnectionContext }

  TCustomSSPIConnectionContext = class(TSSPIContext)

  private
    fStatus: SECURITY_STATUS;
    fOutBuffDesc, fInBuffDesc: SecBufferDesc;
    fInBuff: SecBuffer;
    fAttributes: TSSPIContextAttributes;
  protected
    procedure DoRelease; override;
    function GetAuthenticated: Boolean; override;
    function DoUpdateAndGenerateReply(
      var aIn, aOut: SecBufferDesc;
      const aErrorsToIgnore: array of SECURITY_STATUS
    ): SECURITY_STATUS; virtual; abstract;
  public
    function UpdateAndGenerateReply(
      const aFromPeerToken: String; var aToPeerToken: String): Boolean;
    property Attributes: TSSPIContextAttributes read fAttributes;

  public
    constructor Create(aCredentials: TSSPICredentials);
    destructor Destroy; override;

  end;

  TSSPIClientConnectionContext = class(TCustomSSPIConnectionContext)

  private
    fTargetName: String;
    fRequirements: TSSPIClientContextRequirements;
    function getTargetName: String;
  protected
    function GetRequestedFlags: ULONG; override;
    procedure SetEstablishedFlags(aFlags: ULONG); override;
    function DoUpdateAndGenerateReply(
      var aIn, aOut: SecBufferDesc;
      const aErrorsToIgnore: array of SECURITY_STATUS
    ): SECURITY_STATUS; override;
  public
    function GenerateInitialChalenge(
      const aTargetName: String; var aToPeerToken: String): Boolean;
    property TargetName: String read getTargetName;
    property Requirements: TSSPIClientContextRequirements read fRequirements;

  public
    constructor Create(aCredentials: TSSPICredentials);
    destructor Destroy; override;

  end;

  TSSPIServerConnectionContext = class(TCustomSSPIConnectionContext)

  private
    fRequirements: TSSPIServerContextRequirements;
  protected
    function GetRequestedFlags: ULONG; override;
    procedure SetEstablishedFlags(aFlags: ULONG); override;
    function DoUpdateAndGenerateReply(
      var aIn, aOut: SecBufferDesc;
      const aErrorsToIgnore: array of SECURITY_STATUS
    ): SECURITY_STATUS; override;
  public
    property Requirements: TSSPIServerContextRequirements read fRequirements;

  public
    constructor Create(aCredentials: TSSPICredentials);
    destructor Destroy; override;

  end;

implementation

var

  g: TSSPIInterface;

{----------------------------------------------------------------------------
  ESSPIException
 ----------------------------------------------------------------------------}

function ESSPIException.GetSysErrorTypeStr: String;
begin
  Result := 'SSPI';
end;

class function ESSPIException.GetErrorMessageByNo
  (aErrorNo: LongWord): String;
begin
  case HRESULT(aErrorNo) of
  SEC_E_OK: Result := 'Successfull API call';
  SEC_E_INSUFFICIENT_MEMORY:
    Result := 'Not enough memory is available to complete this request';
  SEC_E_INVALID_HANDLE:
    Result := 'The handle specified is invalid';
  SEC_E_UNSUPPORTED_FUNCTION:
    Result := 'The function requested is not supported';
  SEC_E_TARGET_UNKNOWN:
    Result := 'The specified target is unknown or unreachable';
  SEC_E_INTERNAL_ERROR:
    Result := 'The Local Security Authority cannot be contacted';
  SEC_E_SECPKG_NOT_FOUND:
    Result := 'The requested security package does not exist';
  SEC_E_NOT_OWNER:
    Result := 'The caller is not the owner of the desired credentials';
  SEC_E_CANNOT_INSTALL:
    Result := 'The security package failed to initialize, and cannot be installed';
  SEC_E_INVALID_TOKEN:
    Result := 'The token supplied to the function is invalid';
  SEC_E_CANNOT_PACK:
    Result := 'The security package is not able to marshall the logon buffer, so the logon attempt has failed';
  SEC_E_QOP_NOT_SUPPORTED:
    Result := 'The per-message Quality of Protection is not supported by the security package';
  SEC_E_NO_IMPERSONATION:
    Result := 'The security context does not allow impersonation of the client';
  SEC_E_LOGON_DENIED:
    Result := 'The logon attempt failed';
  SEC_E_UNKNOWN_CREDENTIALS:
    Result := 'The credentials supplied to the package were not recognized';
  SEC_E_NO_CREDENTIALS:
    Result := 'No credentials are available in the security package';
  SEC_E_MESSAGE_ALTERED:
    Result := 'The message or signature supplied for verification has been altered';
  SEC_E_OUT_OF_SEQUENCE:
    Result := 'The message supplied for verification is out of sequence';
  SEC_E_NO_AUTHENTICATING_AUTHORITY:
    Result := 'No authority could be contacted for authentication.';
  SEC_I_CONTINUE_NEEDED:
    Result := 'The function completed successfully, but must be called again to complete the context';
  SEC_I_COMPLETE_NEEDED:
    Result := 'The function completed successfully, but CompleteToken must be called';
  SEC_I_COMPLETE_AND_CONTINUE:
    Result := 'The function completed successfully, but both CompleteToken and this function must be called to complete the context';
  SEC_I_LOCAL_LOGON:
    Result := 'The logon was completed, but no network authority was available. The logon was made using locally known information';
  SEC_E_BAD_PKGID:
    Result := 'The requested security package does not exist';
  SEC_E_CONTEXT_EXPIRED:
    Result := 'The context has expired and can no longer be used.';
  SEC_E_INCOMPLETE_MESSAGE:
    Result := 'The supplied message is incomplete.  The signature was not verified.';
  SEC_E_INCOMPLETE_CREDENTIALS:
    Result := 'The credentials supplied were not complete, and could not be verified. The context could not be initialized.';
  SEC_E_BUFFER_TOO_SMALL:
    Result := 'The buffers supplied to a function was too small.';
  SEC_I_INCOMPLETE_CREDENTIALS:
    Result := 'The credentials supplied were not complete, and could not be verified. Additional information can be returned from the context.';
  SEC_I_RENEGOTIATE:
    Result := 'The context data must be renegotiated with the peer.';
  SEC_E_WRONG_PRINCIPAL:
    Result := 'The target principal name is incorrect.';
  SEC_I_NO_LSA_CONTEXT:
    Result := 'There is no LSA mode context associated with this context.';
  SEC_E_TIME_SKEW:
    Result := 'The clocks on the client and server machines are skewed.';
  SEC_E_UNTRUSTED_ROOT:
    Result := 'The certificate chain was issued by an untrusted authority.';
  SEC_E_ILLEGAL_MESSAGE:
    Result := 'The message received was unexpected or badly formatted.';
  SEC_E_CERT_UNKNOWN:
    Result := 'An unknown error occurred while processing the certificate.';
  SEC_E_CERT_EXPIRED:
    Result := 'The received certificate has expired.';
  SEC_E_ENCRYPT_FAILURE:
    Result := 'The specified data could not be encrypted.';
  SEC_E_DECRYPT_FAILURE:
    Result := 'The specified data could not be decrypted.';
  SEC_E_ALGORITHM_MISMATCH:
    Result := 'The client and server cannot communicate, because they do not possess a common algorithm.';
  SEC_E_SECURITY_QOS_FAILED:
    Result := 'The security context could not be established due to a failure in the requested quality of service (e.g. mutual authentication or delegation).';
  else
    Result := 'Unknown error';
  end;
end;

constructor ESSPIException.CreateError
  (aFailedFuncName: String; anErrorNo: Longint = SEC_E_OK);
begin
  if anErrorNo = SEC_E_OK then
    inherited CreateError(aFailedFuncName)
  else
    inherited CreateError(aFailedFuncName, anErrorNo);
end;

{----------------------------------------------------------------------------
  TRefreshableSSPIObject
 ----------------------------------------------------------------------------}

procedure TRefreshableSSPIObject.RefreshIfUpdatePending;
begin
  if UpdatePending and not Refreshing then begin
    fRefreshing := True;
    try
      Update;
      fUpdatePending := False;
    finally
      fRefreshing := False;
    end;
  end;
end;

procedure TRefreshableSSPIObject.SetUpdatePending;
begin
  fUpdatePending := True;
end;

constructor TRefreshableSSPIObject.Create;
begin
  inherited Create;
  fUpdatePending := True;
  fRefreshing := False;
end;

procedure TRefreshableSSPIObject.Update;
begin
  { does nothing -> to be overriden }
end;

{----------------------------------------------------------------------------
  TRefreshableSSPIObjectList
 ----------------------------------------------------------------------------}

function TRefreshableSSPIObjectList.getCount: Integer;
begin
  RefreshIfUpdatePending;
  if not Assigned(fList) then
    Result := 0
  else
    Result := fList.Count;
end;

function TRefreshableSSPIObjectList.getObjects(idx: Integer): TObject;
begin
  RefreshIfUpdatePending;
  if not Assigned(fList) then
    raise ESSPIException.CreateFmt('Object at index=%d does not exists', [idx]);
  Result := fList.Objects[idx];
end;

procedure TRefreshableSSPIObjectList.Clear;
var
  i: Integer;
begin
  if Assigned(fList) then
    with fList do begin
      for i := 0 to Count - 1 do
        Objects[i].Free;
      Clear;
    end;
end;

procedure TRefreshableSSPIObjectList.AddObject(o: TObject);
begin
  if not Assigned(fList) then
    fList := TStringList.Create;
  fList.AddObject('', o);
end;

{ ------------------------------------------------------------------------}

constructor TRefreshableSSPIObjectList.Create;
begin
  inherited Create;
  fList := nil;
end;

destructor TRefreshableSSPIObjectList.Destroy;
begin
  Clear;
  fList.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIInterface
 ----------------------------------------------------------------------------}

procedure TSSPIInterface.releaseFunctionTable;
begin
  if fPFunctionTable <> nil then begin
{ TODO: it looks like I don't have to free the memory
    ....FreeContextBuffer(fPFunctionTable);
}
    fPFunctionTable := nil;
  end;
end;

procedure TSSPIInterface.checkAvailable;
begin
  if not IsAvailable then
    raise ESSPIInterfaceInitFailed.Create(
      'SSPI interface has failed to initialise properly');
end;

function TSSPIInterface.getFunctionTable: SecurityFunctionTableA;
begin
  checkAvailable;
  Result := fPFunctionTable^;
end;

function TSSPIInterface.getVersion: ULONG;
begin
  Result := FunctionTable.dwVersion;
end;

{ ------------------------------------------------------------------------}

class procedure TSSPIInterface.RaiseIfError
  (aStatus: SECURITY_STATUS; aFunctionName: String);
begin
  { TODO : complete RaiseIfError }
  if not SEC_SUCCESS(aStatus) then
    raise ESSPIException.CreateError(aFunctionName, aStatus);
end;

function TSSPIInterface.IsAvailable: Boolean;

  procedure loadDLL;
  const
    SECURITY_DLL_NT = 'security.dll';
    SECURITY_DLL_95 = 'secur32.dll';
    ENCRYPT_MESSAGE = 'EncryptMessage';
    DECRYPT_MESSAGE = 'DecryptMessage';
  var
    dllName: String;
    entrypoint: INIT_SECURITY_INTERFACE_A;
  begin
    fIsAvailable := False;
    if Win32Platform = VER_PLATFORM_WIN32_WINDOWS then
      { Windows95 SSPI dll }
      dllName := SECURITY_DLL_95
    else
      { WindowsNT & Windows2000 SSPI dll }
      dllName := SECURITY_DLL_NT;
    try
      { load SSPI dll }
      dll.Link(dllName);
      { get InitSecurityInterface entry point
        and call it to fetch SPPI function table}
      entrypoint := dll.GetProcedure(SECURITY_ENTRYPOINTA);
      fPFunctionTable := entrypoint;
      { let's see what SSPI functions are available
        and if we can continue on with the set }
      with fPFunctionTable^ do begin
        fIsAvailable :=
          Assigned(QuerySecurityPackageInfoA) and
          Assigned(FreeContextBuffer) and
          Assigned(DeleteSecurityContext) and
          Assigned(FreeCredentialHandle) and
          Assigned(AcquireCredentialsHandleA) and
          Assigned(InitializeSecurityContextA) and
          { TODO: not required sometimes
          Assigned(CompleteAuthToken) and
          }
          Assigned(AcceptSecurityContext) and
          Assigned(ImpersonateSecurityContext) and
          Assigned(RevertSecurityContext) and
          Assigned(QueryContextAttributesA) and
          Assigned(MakeSignature) and
          Assigned(VerifySignature);
{$IFDEF SET_ENCRYPT_IN_FT_WITH_GETPROCADDRESS_FUDGE}
        { fudge for Encrypt/DecryptMessage }
        if (not Assigned(EncryptMessage)) and dll.Exists(ENCRYPT_MESSAGE) then
          EncryptMessage := dll.GetProcedure(ENCRYPT_MESSAGE);
        if (not Assigned(DecryptMessage)) and dll.Exists(DECRYPT_MESSAGE) then
          DecryptMessage := dll.GetProcedure(DECRYPT_MESSAGE);
{$ENDIF}
       end;
    except
      on e: EDLLLoader do begin
        { get out quitly }
      end else
        raise;
    end;
  end;

begin
  if fIsAvailable then
    Result := True
  else begin
    if fLoadPending then begin
      releaseFunctionTable;
      loadDLL;
      fLoadPending := False;
    end;
    Result := fIsAvailable;
  end;
end;

{ ------------------------------------------------------------------------}

constructor TSSPIInterface.Create;
begin
  inherited Create;
  dll := TDLLLoader.Create;
  fLoadPending := True;
  fIsAvailable := False;
  fPFunctionTable := nil;
end;

destructor TSSPIInterface.Destroy;
begin
  releaseFunctionTable;
  dll.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  SSPIInterface
 ----------------------------------------------------------------------------}

function SSPIInterface: TSSPIInterface;
begin
  Result := g;
end;

{----------------------------------------------------------------------------
  TSSPIPackage
 ----------------------------------------------------------------------------}

function TSSPIPackage.getPSecPkgInfo: PSecPkgInfo;
begin
  if fPSecPkginfo = nil then
    raise ESSPIException.Create('No PSecPkgInfo specified');
  Result := fPSecPkginfo;
end;

function TSSPIPackage.getCapabilities: ULONG;
begin
  Result := getPSecPkgInfo^.fCapabilities;
end;

function TSSPIPackage.getComment: String;
begin
  Result := StrPas(getPSecPkgInfo^.Comment);
end;

function TSSPIPackage.getMaxToken: ULONG;
begin
  Result := getPSecPkgInfo^.cbMaxToken;
end;

function TSSPIPackage.getName: String;
begin
  Result := StrPas(getPSecPkgInfo^.Name);
end;

function TSSPIPackage.getRPCID: Word;
begin
  Result := getPSecPkgInfo^.wRPCID;
end;

function TSSPIPackage.getVersion: Word;
begin
  Result := getPSecPkgInfo^.wVersion;
end;

{ ------------------------------------------------------------------------}

constructor TSSPIPackage.Create(aPSecPkginfo: PSecPkgInfo);
begin
  inherited Create;
  fPSecPkginfo := aPSecPkginfo;
end;

{----------------------------------------------------------------------------
  TSSPIPackages
 ----------------------------------------------------------------------------}

function TSSPIPackages.loadItems: ULONG;
begin
  releaseItems;
  g.RaiseIfError(
    g.FunctionTable.EnumerateSecurityPackagesA(@Result, @fPSecPkgInfoList),
    'EnumerateSecurityPackages');
end;

procedure TSSPIPackages.releaseItems;
begin
  if fPSecPkgInfoList <> nil then begin
    g.RaiseIfError(
      g.FunctionTable.FreeContextBuffer(fPSecPkgInfoList), 'FreeContextBuffer');
    fPSecPkgInfoList := nil;
  end;
end;

procedure TSSPIPackages.Update;
var
  qty: ULONG;
  i: Integer;
  p: PSecPkgInfo;
begin
  Clear;
  inherited Update;
  if g.IsAvailable then begin
    qty := loadItems;
    p := fPSecPkgInfoList;
    for i := 0 to qty - 1 do begin
      AddObject(TSSPIPackage.Create(p));
      Inc(p);
    end;
  end;
end;

{ ------------------------------------------------------------------------}

function TSSPIPackages.getItems(idx: Integer): TSSPIPackage;
begin
  Result := Objects[idx] as TSSPIPackage;
end;

function TSSPIPackages.IndexOf(aName: String): Integer;
var
  i: Integer;
begin
  Result := -1;
  for i := 0 to Count - 1 do
    if Items[i].Name = aName then begin
      Result := i;
      exit;
    end;
end;

{ ------------------------------------------------------------------------}

constructor TSSPIPackages.Create;
begin
  inherited Create;
  fPSecPkgInfoList := nil;
end;

destructor TSSPIPackages.Destroy;
begin
  releaseItems;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomSSPIPackage
 ----------------------------------------------------------------------------}

constructor TCustomSSPIPackage.Create(aPkgName: String);
begin
  g.RaiseIfError(
    g.FunctionTable.QuerySecurityPackageInfoA(PChar(aPkgName), @fInfo),
    'QuerySecurityPackageInfoA');
  inherited Create(fInfo);
end;

destructor TCustomSSPIPackage.Destroy;
begin
  if fInfo <> nil then
    g.RaiseIfError(
      g.FunctionTable.FreeContextBuffer(fInfo), 'FreeContextBuffer');
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPINTLMPackage
 ----------------------------------------------------------------------------}

constructor TSSPINTLMPackage.Create;
begin
  inherited Create(NTLMSP_NAME);
end;

{----------------------------------------------------------------------------
  TSSPIKerberosPackage
 ----------------------------------------------------------------------------}

constructor TSSPIKerberosPackage.Create;
begin
  inherited Create(MICROSOFT_KERBEROS_NAME);
end;

{----------------------------------------------------------------------------
  TSSPINegotiatePackage
 ----------------------------------------------------------------------------}

constructor TSSPINegotiatePackage.Create;
begin
  inherited Create(NEGOSSP_NAME);
end;

{----------------------------------------------------------------------------
  TCustomSSPIAttribute
 ----------------------------------------------------------------------------}

procedure TCustomSSPIAttribute.releaseBuffers;
begin
  if fReleaseSSPIOwnedMemoryPending then begin
    ReleaseSSPIOwnedMemory(GetPBuffer);
    fReleaseSSPIOwnedMemoryPending := False;
  end;
end;

procedure TCustomSSPIAttribute.ReleaseSSPIOwnedMemory(aPBuffer: PVOID);
begin
  { do nothing - to be overriden if needed }
end;

procedure TCustomSSPIAttribute.ReleaseContextBuffer(aPBuffer: PVOID);
begin
  if aPBuffer <> nil then
    g.RaiseIfError(
      g.FunctionTable.FreeContextBuffer(aPBuffer), 'FreeContextBuffer');
end;

procedure TCustomSSPIAttribute.Update;
begin
  inherited Update;
  releaseBuffers;
  DoQuery(GetPBuffer);
  fReleaseSSPIOwnedMemoryPending := True;
end;

function TCustomSSPIAttribute.AsPVOID: PVOID;
begin
  RefreshIfUpdatePending;
  Result := GetPBuffer;
end;

{ ------------------------------------------------------------------------}

constructor TCustomSSPIAttribute.Create;
begin
  inherited Create;
  fReleaseSSPIOwnedMemoryPending := False;
end;

destructor TCustomSSPIAttribute.Destroy;
begin
  releaseBuffers;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomSSPICredentialsAttribute
 ----------------------------------------------------------------------------}

procedure TCustomSSPICredentialsAttribute.DoQuery(aPBuffer: PVOID);
begin
  g.RaiseIfError(
    g.FunctionTable.QueryCredentialsAttributesA(fParent.Handle, Tag, aPBuffer),
    'QueryCredentialsAttributesA');
end;

{ ------------------------------------------------------------------------}

constructor TCustomSSPICredentialsAttribute.Create(aParent: TSSPICredentials);
begin
  inherited Create;
  fParent := aParent;
end;

{----------------------------------------------------------------------------
  TSSPICredentialsAttributeNames
 ----------------------------------------------------------------------------}

function TSSPICredentialsAttributeNames.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

procedure TSSPICredentialsAttributeNames.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgCredentials_Names(aPBuffer)^.sUserName);
end;

function TSSPICredentialsAttributeNames.getUserName: String;
begin
  Result := PSecPkgCredentials_Names(AsPVOID)^.sUserName;
end;

class function TSSPICredentialsAttributeNames.Tag: ULONG;
begin
  Result := SECPKG_CRED_ATTR_NAMES;
end;

{----------------------------------------------------------------------------
  TSSPICredentialsAttributes
 ----------------------------------------------------------------------------}

procedure TSSPICredentialsAttributes.Clear;
begin
  fNames.SetUpdatePending;
end;

constructor TSSPICredentialsAttributes.Create
  (aParent: TSSPICredentials);
begin
  inherited Create;
  fNames := TSSPICredentialsAttributeNames.Create(aParent);
end;

destructor TSSPICredentialsAttributes.Destroy;
begin
  fNames.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPICredentials
 ----------------------------------------------------------------------------}

procedure TSSPICredentials.CheckAcquired;
begin
  if not fAcquired then
    raise ESSPIException.Create('No credential handle acquired');
end;

procedure TSSPICredentials.CheckNotAcquired;
begin
  if fAcquired then
    raise ESSPIException.Create(
      'Can not change credentials after handle aquired. Use Release first');
end;

procedure TSSPICredentials.DoAcquire
  (pszPrincipal: PSEC_CHAR; pvLogonId, pAuthData: PVOID);
var
  cu: ULONG;
begin
  Release;
  case Use of
  scuInBound:
    cu := SECPKG_CRED_INBOUND;
  scuOutBound:
    cu := SECPKG_CRED_OUTBOUND;
  scuBoth:
    cu := SECPKG_CRED_BOTH;
  else
    raise ESSPIException.Create('Unknown credentials use');
  end;
  g.RaiseIfError(
    g.FunctionTable.AcquireCredentialsHandleA(
      pszPrincipal, PSEC_CHAR(Package.Name), cu, pvLogonId, pAuthData, nil, nil,
      @fHandle, @fExpiry),
    'AcquireCredentialsHandleA');
  fAcquired := True;
end;

procedure TSSPICredentials.DoRelease;
begin
  g.RaiseIfError(
    g.FunctionTable.FreeCredentialHandle(@fHandle),
    'FreeCredentialHandle');
  SecInvalidateHandle(@fHandle);
end;

procedure TSSPICredentials.Release;
begin
  if fAcquired then begin
    DoRelease;
    fAcquired := False;
  end;
end;

{ ------------------------------------------------------------------------}

function TSSPICredentials.getHandle: PCredHandle;
begin
  CheckAcquired;
  Result := @fHandle;
end;

procedure TSSPICredentials.setUse(aValue: TSSPICredentialsUse);
begin
  if fUse <> aValue then begin
    CheckNotAcquired;
    fUse := aValue;
  end;
end;

function TSSPICredentials.getExpiry: TimeStamp;
begin
  CheckAcquired;
  Result := fExpiry;
end;

{ ------------------------------------------------------------------------}

constructor TSSPICredentials.Create(aPackage: TSSPIPackage);
begin
  inherited Create;
  fPackage := aPackage;
  fUse := scuOutBound;
  fAcquired := False;
end;

destructor TSSPICredentials.Destroy;
begin
  Release;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIWinNTCredentials
 ----------------------------------------------------------------------------}

procedure TSSPIWinNTCredentials.DoRelease;
begin
  inherited DoRelease;
  if Assigned(Attributes) then
    Attributes.Clear;
end;

procedure TSSPIWinNTCredentials.Acquire(aUse: TSSPICredentialsUse);
begin
  Acquire(aUse, '', '', '');
end;

procedure TSSPIWinNTCredentials.Acquire
  (aUse: TSSPICredentialsUse; aDomain, aUserName, aPassword: String);
var
  ai: SEC_WINNT_AUTH_IDENTITY;
  pai: PVOID;
begin
  Use := aUse;
  if (Length(aDomain) > 0) and (Length(aUserName) > 0) then begin
    with ai do begin
      User := PChar(aUserName);
      UserLength := Length(aUserName);
      Domain := PChar(aDomain);
      DomainLength := Length(aDomain);
      Password := PChar(aPassword);
      PasswordLength := Length(aPassword);
      Flags := SEC_WINNT_AUTH_IDENTITY_ANSI;
    end;
    pai := @ai;
  end else
    pai := nil;
  DoAcquire(nil, nil, pai);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIWinNTCredentials.Create(aPackage: TSSPIPackage);
begin
  inherited Create(aPackage);
  fAttributes := TSSPICredentialsAttributes.Create(self);
end;

destructor TSSPIWinNTCredentials.Destroy;
begin
  fAttributes.Free;
  fAttributes := nil;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIContextRequirement
 ----------------------------------------------------------------------------}

const

  REQUIREMENT_VALUES:
    array [Boolean] of
      array [Boolean] of
        array [TSSPIContextRequirementsValue] of ULONG = (
    (  // client
      (  // requested
        ISC_REQ_DELEGATE, ISC_REQ_MUTUAL_AUTH, ISC_REQ_REPLAY_DETECT,
        ISC_REQ_SEQUENCE_DETECT, ISC_REQ_CONFIDENTIALITY,
        ISC_REQ_USE_SESSION_KEY, ISC_REQ_PROMPT_FOR_CREDS,
        ISC_REQ_USE_SUPPLIED_CREDS, ISC_REQ_ALLOCATE_MEMORY,
        ISC_REQ_USE_DCE_STYLE, ISC_REQ_DATAGRAM, ISC_REQ_CONNECTION,
        ISC_REQ_STREAM, ISC_REQ_EXTENDED_ERROR, ISC_REQ_INTEGRITY,
        ISC_REQ_IDENTIFY, ISC_REQ_MANUAL_CRED_VALIDATION
      ),
      (  // returned
        ISC_RET_DELEGATE, ISC_RET_MUTUAL_AUTH, ISC_RET_REPLAY_DETECT,
        ISC_RET_SEQUENCE_DETECT, ISC_RET_CONFIDENTIALITY,
        ISC_RET_USE_SESSION_KEY, ISC_RET_USED_COLLECTED_CREDS,
        ISC_RET_USED_SUPPLIED_CREDS, ISC_RET_ALLOCATED_MEMORY,
        ISC_RET_USED_DCE_STYLE, ISC_RET_DATAGRAM, ISC_RET_CONNECTION,
        ISC_RET_STREAM, ISC_RET_EXTENDED_ERROR, ISC_RET_INTEGRITY,
        ISC_RET_IDENTIFY, ISC_RET_MANUAL_CRED_VALIDATION
      )
    ),
    (  // server
      (  // requested
        ASC_REQ_DELEGATE, ASC_REQ_MUTUAL_AUTH, ASC_REQ_REPLAY_DETECT,
        ASC_REQ_SEQUENCE_DETECT, ASC_REQ_CONFIDENTIALITY,
        ASC_REQ_USE_SESSION_KEY,
        0, 0, ASC_REQ_ALLOCATE_MEMORY,
        ASC_REQ_USE_DCE_STYLE, ASC_REQ_DATAGRAM, ASC_REQ_CONNECTION,
        ASC_REQ_STREAM, ASC_REQ_EXTENDED_ERROR, ASC_REQ_INTEGRITY,
        ASC_REQ_IDENTIFY, 0
      ),
      (  // returned
        ASC_RET_DELEGATE, ASC_RET_MUTUAL_AUTH, ASC_RET_REPLAY_DETECT,
        ASC_RET_SEQUENCE_DETECT, ASC_RET_CONFIDENTIALITY,
        ASC_RET_USE_SESSION_KEY,
        0, 0, ASC_RET_ALLOCATED_MEMORY,
        ASC_RET_USED_DCE_STYLE, ASC_RET_DATAGRAM, ASC_RET_CONNECTION,
        ASC_RET_STREAM, ASC_RET_EXTENDED_ERROR, ASC_RET_INTEGRITY,
        ASC_RET_IDENTIFY, 0
      )
    )
  );

class function TSSPIContextRequirement.ULONGToValue
  (aULONG: ULONG): TSSPIContextRequirementsValues;
var
  i: TSSPIContextRequirementsValue;
begin
  Result := [];
  for i := Low(i) to High(i) do
    if (aULONG and REQUIREMENT_VALUES[IsServer, IsEstablished, i]) <> 0 then
      Include(Result, i);
end;

function TSSPIContextRequirement.GetValue: TSSPIContextRequirementsValues;
begin
  Result := ULONGToValue(AsULONG);
end;

function TSSPIContextRequirement.GetAsULONG: ULONG;
begin
  Result := fULONG;
end;

procedure TSSPIContextRequirement.SetAsULONG(aValue: ULONG);
begin
  fULONG := aValue;
end;

{----------------------------------------------------------------------------
  TSSPIContextRequirementEstablished
 ----------------------------------------------------------------------------}

class function TSSPIContextRequirementEstablished.IsEstablished: Boolean;
begin
  Result := True;
end;

{----------------------------------------------------------------------------
  TSSPIContextRequirementRequested
 ----------------------------------------------------------------------------}

class function TSSPIContextRequirementRequested.IsEstablished: Boolean;
begin
  Result := False;
end;

function TSSPIContextRequirementRequested.GetAsULONG: ULONG;
begin
  Result := inherited GetAsULONG + fMinimumReq;
end;

procedure TSSPIContextRequirementRequested.SetValueWhileContextIsActive
  (aValue: TSSPIContextRequirementsValues);
var
  i: TSSPIContextRequirementsValue;
  v: ULONG;
begin
  v := 0;
  for i := Low(i) to High(i) do
    if i in aValue then
      v := v or REQUIREMENT_VALUES[IsServer, IsEstablished, i];
  SetAsULONG(v);
end;

procedure TSSPIContextRequirementRequested.setValue
  (aValue: TSSPIContextRequirementsValues);
begin
  if fContext.HasHandle then
    raise ESSPIException.Create(
      'Can not change requested requirement after handle aquired');
  SetValueWhileContextIsActive(aValue);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIContextRequirementRequested.Create
  (aContext: TSSPIContext; aMinimumReq: ULONG);
begin
  inherited Create;
  fContext := aContext;
  fMinimumReq := aMinimumReq;
end;

{----------------------------------------------------------------------------
  TSSPI...ContextRequirement...
 ----------------------------------------------------------------------------}

class function TSSPIClientContextRequirementRequested.IsServer: Boolean;
begin
  Result := False;
end;

class function TSSPIClientContextRequirementEstablished.IsServer: Boolean;
begin
  Result := False;
end;

class function TSSPIServerContextRequirementRequested.IsServer: Boolean;
begin
  Result := True;
end;

class function TSSPIServerContextRequirementEstablished.IsServer: Boolean;
begin
  Result := True;
end;

{----------------------------------------------------------------------------
  TSSPIClientContextRequirements
 ----------------------------------------------------------------------------}

procedure TSSPIClientContextRequirements.SetEstablishedAsULong(aValue: ULONG);
begin
  Established.SetAsULONG(aValue);
end;

procedure TSSPIClientContextRequirements.SetRequestedValueWhileContextIsActive
  (aValue: TSSPIContextRequirementsValues);
begin
  Requested.SetValueWhileContextIsActive(aValue);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIClientContextRequirements.Create
  (aContext: TSSPIContext; aMinimumReq: ULONG);
begin
  inherited Create;
  fRequested :=
    TSSPIClientContextRequirementRequested.Create(aContext, aMinimumReq);
  fEstablished := TSSPIClientContextRequirementEstablished.Create;
end;

destructor TSSPIClientContextRequirements.Destroy;
begin
  fEstablished.Free;
  fRequested.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIServerContextRequirements
 ----------------------------------------------------------------------------}

procedure TSSPIServerContextRequirements.SetEstablishedAsULong(aValue: ULONG);
begin
  Established.SetAsULONG(aValue);
end;

procedure TSSPIServerContextRequirements.SetRequestedValueWhileContextIsActive
  (aValue: TSSPIContextRequirementsValues);
begin
  Requested.SetValueWhileContextIsActive(aValue);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIServerContextRequirements.Create
  (aContext: TSSPIContext; aMinimumReq: ULONG);
begin
  inherited Create;
  fRequested :=
    TSSPIServerContextRequirementRequested.Create(aContext, aMinimumReq);
  fEstablished := TSSPIServerContextRequirementEstablished.Create;
end;

destructor TSSPIServerContextRequirements.Destroy;
begin
  fEstablished.Free;
  fRequested.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomSSPIContextAttribute
 ----------------------------------------------------------------------------}

procedure TCustomSSPIContextAttribute.DoQuery(aPBuffer: PVOID);
begin
  g.RaiseIfError(
    g.FunctionTable.QueryContextAttributesA(fParent.Handle, Tag, aPBuffer),
    'QueryContextAttributes');
end;

{ ------------------------------------------------------------------------}

constructor TCustomSSPIContextAttribute.Create(aParent: TSSPIContext);
begin
  inherited Create;
  fParent := aParent;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeSizes
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeSizes.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeSizes.getULONG(idx: Integer): ULONG;
begin
  with PSecPkgContext_Sizes(AsPVOID)^ do
    case idx of
    1: Result := cbMaxToken;
    2: Result := cbMaxSignature;
    3: Result := cbBlockSize;
    4: Result := cbSecurityTrailer;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

class function TSSPIContextAttributeSizes.Tag: ULONG;
begin
  Result := SECPKG_ATTR_SIZES;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeNames
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeNames.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

procedure TSSPIContextAttributeNames.ReleaseSSPIOwnedMemory(aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgContext_Names(aPBuffer)^.sUserName);
end;

function TSSPIContextAttributeNames.getUserName: String;
begin
  Result := PSecPkgContext_Names(AsPVOID)^.sUserName;
end;

class function TSSPIContextAttributeNames.Tag: ULONG;
begin
  Result := SECPKG_ATTR_NAMES;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeLifespan
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeLifespan.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeLifespan.getTimeStamp(idx: Integer): TimeStamp;
begin
  with PSecPkgContext_Lifespan(AsPVOID)^ do
    case idx of
    1: Result := tsStart;
    2: Result := tsExpiry;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

class function TSSPIContextAttributeLifespan.Tag: ULONG;
begin
  Result := SECPKG_ATTR_LIFESPAN;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeDCEInfo
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeDCEInfo.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

procedure TSSPIContextAttributeDCEInfo.ReleaseSSPIOwnedMemory(aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgContext_DceInfo(aPBuffer)^.pPac);
end;

function TSSPIContextAttributeDCEInfo.getAuthzSvc: ULONG;
begin
  Result := PSecPkgContext_DceInfo(AsPVOID)^.AuthzSvc;
end;

function TSSPIContextAttributeDCEInfo.getPac: PVOID;
begin
  Result := PSecPkgContext_DceInfo(AsPVOID)^.pPac;
end;

class function TSSPIContextAttributeDCEInfo.Tag: ULONG;
begin
  Result := SECPKG_ATTR_DCE_INFO;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeStreamSizes
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeStreamSizes.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeStreamSizes.getULONG(idx: Integer): ULONG;
begin
  with PSecPkgContext_StreamSizes(AsPVOID)^ do
    case idx of
    1: Result := cbHeader;
    2: Result := cbTrailer;
    3: Result := cbMaximumMessage;
    4: Result := cBuffers;
    5: Result := cbBlockSize;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

class function TSSPIContextAttributeStreamSizes.Tag: ULONG;
begin
  Result := SECPKG_ATTR_STREAM_SIZES;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeKeyInfo
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeKeyInfo.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeKeyInfo.getString(idx: Integer): String;
begin
  with PSecPkgContext_KeyInfo(AsPVOID)^ do
    case idx of
    1: Result := sSignatureAlgorithmName;
    2: Result := sEncryptAlgorithmName;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

function TSSPIContextAttributeKeyInfo.getULONG(idx: Integer): ULONG;
begin
  with PSecPkgContext_KeyInfo(AsPVOID)^ do
    case idx of
    1: Result := KeySize;
    2: Result := SignatureAlgorithm;
    3: Result := EncryptAlgorithm;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

procedure TSSPIContextAttributeKeyInfo.ReleaseSSPIOwnedMemory(aPBuffer: PVOID);
begin
  ReleaseContextBuffer(
    PSecPkgContext_KeyInfo(aPBuffer)^.sSignatureAlgorithmName);
  ReleaseContextBuffer(
    PSecPkgContext_KeyInfo(aPBuffer)^.sEncryptAlgorithmName);
end;

class function TSSPIContextAttributeKeyInfo.Tag: ULONG;
begin
  Result := SECPKG_ATTR_KEY_INFO;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeAuthority
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeAuthority.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

procedure TSSPIContextAttributeAuthority.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgContext_Authority(aPBuffer)^.sAuthorityName);
end;

function TSSPIContextAttributeAuthority.getAuthorityName: String;
begin
  Result := PSecPkgContext_Authority(AsPVOID)^.sAuthorityName;
end;

class function TSSPIContextAttributeAuthority.Tag: ULONG;
begin
  Result := SECPKG_ATTR_AUTHORITY;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeProtoInfo
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeProtoInfo.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

procedure TSSPIContextAttributeProtoInfo.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgContext_ProtoInfo(aPBuffer)^.sProtocolName);
end;

function TSSPIContextAttributeProtoInfo.getProtocolName: String;
begin
  Result := PSecPkgContext_ProtoInfo(AsPVOID)^.sProtocolName;
end;

function TSSPIContextAttributeProtoInfo.getULONG(idx: Integer): ULONG;
begin
  with PSecPkgContext_ProtoInfo(AsPVOID)^ do
    case idx of
    1: Result := majorVersion;
    2: Result := minorVersion;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

class function TSSPIContextAttributeProtoInfo.Tag: ULONG;
begin
  Result := SECPKG_ATTR_PROTO_INFO;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributePasswordExpiry
 ----------------------------------------------------------------------------}

function TSSPIContextAttributePasswordExpiry.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributePasswordExpiry.getPasswordExpires: TimeStamp;
begin
  Result := PSecPkgContext_PasswordExpiry(AsPVOID)^.tsPasswordExpires;
end;

class function TSSPIContextAttributePasswordExpiry.Tag: ULONG;
begin
  Result := SECPKG_ATTR_PASSWORD_EXPIRY;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeSessionKey
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeSessionKey.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeSessionKey.getSessionKey: PUCHAR;
begin
  Result := PSecPkgContext_SessionKey(AsPVOID)^.SessionKey;
end;

function TSSPIContextAttributeSessionKey.getSessionKeyLength: ULONG;
begin
  Result := PSecPkgContext_SessionKey(AsPVOID)^.SessionKeyLength;
end;

procedure TSSPIContextAttributeSessionKey.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  ReleaseContextBuffer(PSecPkgContext_SessionKey(aPBuffer)^.SessionKey);
end;

class function TSSPIContextAttributeSessionKey.Tag: ULONG;
begin
  Result := SECPKG_ATTR_SESSION_KEY;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributePackageInfo
 ----------------------------------------------------------------------------}

function TSSPIContextAttributePackageInfo.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributePackageInfo.getPackageInfo: PSecPkgInfoA;
begin
  Result := PSecPkgContext_PackageInfo(AsPVOID)^.PackageInfo;
end;

procedure TSSPIContextAttributePackageInfo.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  with PSecPkgContext_PackageInfo(aPBuffer)^ do begin
    ReleaseContextBuffer(PackageInfo.Name);
    ReleaseContextBuffer(PackageInfo.Comment);
    ReleaseContextBuffer(PackageInfo);
  end;
end;

class function TSSPIContextAttributePackageInfo.Tag: ULONG;
begin
  Result := SECPKG_ATTR_PACKAGE_INFO;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeUserFlags
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeUserFlags.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeUserFlags.getUserFlags: ULONG;
begin
  Result := PSecPkgContext_UserFlags(AsPVOID)^.UserFlags;
end;

class function TSSPIContextAttributeUserFlags.Tag: ULONG;
begin
  Result := SECPKG_ATTR_USER_FLAGS;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeNegotiationInfo
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeNegotiationInfo.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeNegotiationInfo.getNegotiationState: ULONG;
begin
  Result := PSecPkgContext_NegotiationInfo(AsPVOID)^.NegotiationState;
end;

function TSSPIContextAttributeNegotiationInfo.getPackageInfo: PSecPkgInfoA;
begin
  Result := PSecPkgContext_NegotiationInfo(AsPVOID)^.PackageInfo;
end;

procedure TSSPIContextAttributeNegotiationInfo.ReleaseSSPIOwnedMemory
  (aPBuffer: PVOID);
begin
  with PSecPkgContext_NegotiationInfo(aPBuffer)^ do begin
    ReleaseContextBuffer(PackageInfo.Name);
    ReleaseContextBuffer(PackageInfo.Comment);
    ReleaseContextBuffer(PackageInfo);
  end;
end;

class function TSSPIContextAttributeNegotiationInfo.Tag: ULONG;
begin
  Result := SECPKG_ATTR_NEGOTIATION_INFO;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeNativeNames
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeNativeNames.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeNativeNames.getString(idx: Integer): String;
begin
  with PSecPkgContext_NativeNames(AsPVOID)^ do
    case idx of
    1: Result := sClientName;
    2: Result := sServerName;
    else
      raise ESSPIException.CreateFmt(
        'Invalid index=%d for attribute=%d', [idx, Tag]);
    end;
end;

procedure TSSPIContextAttributeNativeNames.ReleaseSSPIOwnedMemory(
  aPBuffer: PVOID);
begin
  with PSecPkgContext_NativeNames(aPBuffer)^ do begin
    ReleaseContextBuffer(sClientName);
    ReleaseContextBuffer(sServerName);
  end;
end;

class function TSSPIContextAttributeNativeNames.Tag: ULONG;
begin
  Result := SECPKG_ATTR_NATIVE_NAMES;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributeFlags
 ----------------------------------------------------------------------------}

function TSSPIContextAttributeFlags.GetPBuffer: PVOID;
begin
  Result := @fBuffer;
end;

function TSSPIContextAttributeFlags.getFlags: ULONG;
begin
  Result := PSecPkgContext_Flags(AsPVOID)^.Flags;
end;

class function TSSPIContextAttributeFlags.Tag: ULONG;
begin
  Result := SECPKG_ATTR_FLAGS;
end;

{----------------------------------------------------------------------------
  TSSPIContextAttributes
 ----------------------------------------------------------------------------}

procedure TSSPIContextAttributes.Clear;
begin
  Sizes.SetUpdatePending;
  Names.SetUpdatePending;
  Lifespan.SetUpdatePending;
  DCEInfo.SetUpdatePending;
  StreamSizes.SetUpdatePending;
  KeyInfo.SetUpdatePending;
  Authority.SetUpdatePending;
  ProtoInfo.SetUpdatePending;
  PasswordExpiry.SetUpdatePending;
  SessionKey.SetUpdatePending;
  PackageInfo.SetUpdatePending;
  UserFlags.SetUpdatePending;
  NegotiationInfo.SetUpdatePending;
  NativeNames.SetUpdatePending;
  Flags.SetUpdatePending;
end;

{ ------------------------------------------------------------------------}

constructor TSSPIContextAttributes.Create(aParent: TSSPIContext);
begin
  inherited Create;
  fSizes := TSSPIContextAttributeSizes.Create(aParent);
  fNames := TSSPIContextAttributeNames.Create(aParent);
  fLifespan := TSSPIContextAttributeLifespan.Create(aParent);
  fDCEInfo := TSSPIContextAttributeDCEInfo.Create(aParent);
  fStreamSizes := TSSPIContextAttributeStreamSizes.Create(aParent);
  fKeyInfo := TSSPIContextAttributeKeyInfo.Create(aParent);
  fAuthority := TSSPIContextAttributeAuthority.Create(aParent);
  fProtoInfo := TSSPIContextAttributeProtoInfo.Create(aParent);
  fPasswordExpiry := TSSPIContextAttributePasswordExpiry.Create(aParent);
  fSessionKey := TSSPIContextAttributeSessionKey.Create(aParent);
  fPackageInfo := TSSPIContextAttributePackageInfo.Create(aParent);
  fUserFlags := TSSPIContextAttributeUserFlags.Create(aParent);
  fNegotiationInfo := TSSPIContextAttributeNegotiationInfo.Create(aParent);
  fNativeNames := TSSPIContextAttributeNativeNames.Create(aParent);
  fFlags := TSSPIContextAttributeFlags.Create(aParent);
end;

destructor TSSPIContextAttributes.Destroy;
begin
  fFlags.Free;
  fNativeNames.Free;
  fNegotiationInfo.Free;
  fUserFlags.Free;
  fPackageInfo.Free;
  fSessionKey.Free;
  fPasswordExpiry.Free;
  fProtoInfo.Free;
  fAuthority.Free;
  fKeyInfo.Free;
  fStreamSizes.Free;
  fDCEInfo.Free;;
  fLifespan.Free;
  fNames.Free;
  fSizes.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIContext
 ----------------------------------------------------------------------------}

procedure TSSPIContext.updateHasContextAndCheckForError(
  const aFuncResult: SECURITY_STATUS; const aFuncName: String;
  const aErrorsToIgnore: array of SECURITY_STATUS);
var
  doRaise: Boolean;
  i: Integer;
begin
  doRaise := not SEC_SUCCESS(aFuncResult);
  if doRaise then
    for i := Low(aErrorsToIgnore) to High(aErrorsToIgnore) do
      if aFuncResult = aErrorsToIgnore[i] then begin
        doRaise := False;
        break;
      end;
  if doRaise then
    raise ESSPIException.CreateError(aFuncName, aFuncResult);
  if not fHasHandle then
    fHasHandle := True;
end;

function TSSPIContext.DoAccept
  (var aIn, aOut: SecBufferDesc;
   const errorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
var
  tmp: PCtxtHandle;
  r: ULONG;
begin
  if fHasHandle then
    tmp := @fHandle
  else
    tmp := nil;
  Result :=
  { TODO : make SECURITY_NATIVE_DREP optional }
    g.FunctionTable.AcceptSecurityContext(
      Credentials.Handle, tmp, @aIn,
      GetRequestedFlags, SECURITY_NATIVE_DREP,
      @fHandle, @aOut, @r, @fExpiry
    );
  updateHasContextAndCheckForError(
    Result, 'AcceptSecurityContextA', errorsToIgnore);
  SetEstablishedFlags(r);
end;

function TSSPIContext.DoInitialize
  (aTokenSourceName: PChar;
   var aIn, aOut: SecBufferDesc;
   const errorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
var
  tmp: PCtxtHandle;
  tmp2: PSecBufferDesc;
  r: ULONG;
begin
  if fHasHandle then begin
    tmp := @fHandle;
    tmp2 := @aIn;
  end else begin
    tmp := nil;
    tmp2 := nil;
  end;
  Result :=
    g.FunctionTable.InitializeSecurityContextA(
      Credentials.Handle, tmp, aTokenSourceName,
      GetRequestedFlags, 0, SECURITY_NATIVE_DREP, tmp2, 0,
      @fHandle, @aOut, @r, @fExpiry
    );
  updateHasContextAndCheckForError(
    Result, 'InitializeSecurityContextA', errorsToIgnore);
  SetEstablishedFlags(r);
end;

procedure TSSPIContext.DoRelease;
begin
  { TODO : clear attributes in child classes }
  g.RaiseIfError(
    g.FunctionTable.DeleteSecurityContext(@fHandle), 'DeleteSecurityContext');
end;

procedure TSSPIContext.Release;
begin
  if HasHandle then begin
    DoRelease;
    fHasHandle := False;
  end;
end;

{ ------------------------------------------------------------------------}

procedure TSSPIContext.CheckHasHandle;
begin
  if not HasHandle then
    raise ESSPIException.Create('No security context handle acquired');
end;

procedure TSSPIContext.CheckCredentials;
begin
  if (not Assigned(Credentials)) or (not Credentials.Acquired) then
    raise ESSPIException.Create('Do AcquireCredentialsHandle first');
end;

procedure TSSPIContext.SetCredentials(aCredentials: TSSPICredentials);
begin
  if Credentials <> aCredentials then
    fCredentials := aCredentials;
end;

function TSSPIContext.getExpiry: TimeStamp;
begin
  CheckHasHandle;
  Result := fExpiry;
end;

function TSSPIContext.getHandle: PCtxtHandle;
begin
  CheckHasHandle;
  Result := @fHandle;
end;

{ ------------------------------------------------------------------------}

const

  cnstNoEncryptionSupported = 'The SSPI does not support encryption';

function TSSPIContext.DoEncrypt
  (var aMsg: SecBufferDesc; aMsgNo, aQOP: ULONG;
   const aErrorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
var
  em: ENCRYPT_MESSAGE_FN;
  emName: String;
begin
  if Assigned(g.FunctionTable.EncryptMessage) then begin
    em := g.FunctionTable.EncryptMessage;
    emName := 'EncryptMessage';
{$IFDEF USE_SEAL_INSTEAD_OF_SET_ENCRYPT_IN_FT_WITH_GETPROCADDRESS_FUDGE}
  end else if Assigned(g.FunctionTable.Reserved3) then begin
    em := g.FunctionTable.Reserved3;
    emName := 'Reserved3';
{$ENDIF}
  end else
    raise ESSPIException.Create(cnstNoEncryptionSupported);
  Result := em(Handle, aQOP, @aMsg, aMsgNo);
  updateHasContextAndCheckForError(Result, emName, aErrorsToIgnore);
end;

function TSSPIContext.DoDecrypt
  (var aMsg: SecBufferDesc; aMsgNo: ULONG; var aQOP: ULONG;
   const aErrorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
var
  dm: DECRYPT_MESSAGE_FN;
  dmName: String;
begin
  if Assigned(g.FunctionTable.DecryptMessage) then begin
    dm := g.FunctionTable.DecryptMessage;
    dmName := 'DecryptMessage';
{$IFDEF USE_SEAL_INSTEAD_OF_SET_ENCRYPT_IN_FT_WITH_GETPROCADDRESS_FUDGE}
  end else if Assigned(g.FunctionTable.Reserved4) then begin
    dm := g.FunctionTable.Reserved4;
    dmName := 'Reserved4';
{$ENDIF}
  end else
    raise ESSPIException.Create(cnstNoEncryptionSupported);
  Result := dm(Handle, @aMsg, aMsgNo, @aQOP);
  updateHasContextAndCheckForError(Result, dmName, aErrorsToIgnore);
end;

{ ------------------------------------------------------------------------}

function TSSPIContext.getImpersonating: Boolean;
begin
  CheckHasHandle;
  Result := fImpersonating;
end;

procedure TSSPIContext.Impersonate;
begin
  g.RaiseIfError(
    g.FunctionTable.ImpersonateSecurityContext(@fHandle),
    'ImpersonateSecurityContext');
  fImpersonating := True;
end;

procedure TSSPIContext.RevertToSelf;
begin
  g.RaiseIfError(
    g.FunctionTable.RevertSecurityContext(@fHandle), 'RevertSecurityContext');
  fImpersonating := False;
end;

{ ------------------------------------------------------------------------}

constructor TSSPIContext.Create(aCredentials: TSSPICredentials);
begin
  inherited Create;
  fCredentials := aCredentials;
  fHasHandle := False;
end;

destructor TSSPIContext.Destroy;
begin
  Release;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomSSPIConnectionContext
 ----------------------------------------------------------------------------}

procedure TCustomSSPIConnectionContext.DoRelease;
begin
  inherited DoRelease;
  fStatus := SEC_E_INVALID_HANDLE; // just to put something other then SEC_E_OK
  if Assigned(Attributes) then
    Attributes.Clear;
end;

function TCustomSSPIConnectionContext.GetAuthenticated: Boolean;
begin
  CheckHasHandle;
  Result := fStatus = SEC_E_OK;
end;

function TCustomSSPIConnectionContext.UpdateAndGenerateReply
  (const aFromPeerToken: String; var aToPeerToken: String): Boolean;
var
  fOutBuff: SecBuffer;
begin
  Result := False;
  { check credentials }
  CheckCredentials;
  { prepare input buffer }
  with fInBuff do begin
    cbBuffer := Length(aFromPeerToken);
    pvBuffer := @(aFromPeerToken[1]);
  end;
  { prepare output buffer }
  with fOutBuff do begin
    BufferType := SECBUFFER_TOKEN;
    cbBuffer := Credentials.Package.MaxToken;
    pvBuffer := AllocMem(cbBuffer);
  end;
  with fOutBuffDesc do begin
    ulVersion := SECBUFFER_VERSION;
    cBuffers := 1;
    pBuffers := @fOutBuff;
  end;
  try
    { do processing }
    fStatus := DoUpdateAndGenerateReply(fInBuffDesc, fOutBuffDesc, []);
    { complete token if applicable }
    case fStatus of
    SEC_I_COMPLETE_NEEDED,
    SEC_I_COMPLETE_AND_CONTINUE:
      begin
        if not Assigned(g.FunctionTable.CompleteAuthToken) then
          raise ESSPIException.Create('CompleteAuthToken is not supported');
        fStatus := g.FunctionTable.CompleteAuthToken(Handle, @fOutBuffDesc);
        g.RaiseIfError(fStatus, 'CompleteAuthToken');
      end;
    end;
    Result :=
      (fStatus = SEC_I_CONTINUE_NEEDED) or
      (fStatus = SEC_I_COMPLETE_AND_CONTINUE) or
      (fOutBuff.cbBuffer > 0);
    if Result then
      with fOutBuff do
        SetString(aToPeerToken, PChar(pvBuffer), cbBuffer);
  finally
    FreeMem(fOutBuff.pvBuffer);
  end;
end;

{ ------------------------------------------------------------------------}

constructor TCustomSSPIConnectionContext.Create(aCredentials: TSSPICredentials);
begin
  inherited Create(aCredentials);
  fAttributes := TSSPIContextAttributes.Create(self);
  with fInBuff do begin
    BufferType := SECBUFFER_TOKEN;
  end;
  with fInBuffDesc do begin
    ulVersion := SECBUFFER_VERSION;
    cBuffers := 1;
    pBuffers := @fInBuff;
  end;
  with fOutBuffDesc do begin
    ulVersion := SECBUFFER_VERSION;
    cBuffers := 1;
  end;
end;

destructor TCustomSSPIConnectionContext.Destroy;
begin
  fAttributes.Free;
  fAttributes := nil;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIClientConnectionContext
 ----------------------------------------------------------------------------}

function TSSPIClientConnectionContext.GetRequestedFlags: ULONG;
begin
  Result := Requirements.Requested.AsULONG;
end;

procedure TSSPIClientConnectionContext.SetEstablishedFlags(aFlags: ULONG);
begin
  Requirements.Established.SetAsULONG(aFlags);
end;

function TSSPIClientConnectionContext.DoUpdateAndGenerateReply
  (var aIn, aOut: SecBufferDesc;
   const aErrorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
begin
  Result := DoInitialize(PChar(fTargetName), aIn, aOut, []);
end;

{ ------------------------------------------------------------------------}

function TSSPIClientConnectionContext.getTargetName: String;
begin
  CheckHasHandle;
  Result := fTargetName;
end;

function TSSPIClientConnectionContext.GenerateInitialChalenge
  (const aTargetName: String; var aToPeerToken: String): Boolean;
begin
  Release;
  fTargetName := aTargetName;
  Result := UpdateAndGenerateReply('', aToPeerToken);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIClientConnectionContext.Create(aCredentials: TSSPICredentials);
begin
  inherited Create(aCredentials);
  fRequirements :=
    TSSPIClientContextRequirements.Create(self, ISC_REQ_CONNECTION);
  fTargetName := '';
end;

destructor TSSPIClientConnectionContext.Destroy;
begin
  fRequirements.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TSSPIServerConnectionContext
 ----------------------------------------------------------------------------}

function TSSPIServerConnectionContext.GetRequestedFlags: ULONG;
begin
  Result := Requirements.Requested.AsULONG;
end;

procedure TSSPIServerConnectionContext.SetEstablishedFlags(aFlags: ULONG);
begin
  Requirements.Established.SetAsULONG(aFlags);
end;

function TSSPIServerConnectionContext.DoUpdateAndGenerateReply
  (var aIn, aOut: SecBufferDesc;
   const aErrorsToIgnore: array of SECURITY_STATUS): SECURITY_STATUS;
begin
  Result := DoAccept(aIn, aOut, []);
end;

{ ------------------------------------------------------------------------}

constructor TSSPIServerConnectionContext.Create(
  aCredentials: TSSPICredentials);
begin
  inherited Create(aCredentials);
  fRequirements :=
    TSSPIServerContextRequirements.Create(self, ASC_REQ_CONNECTION);
end;

destructor TSSPIServerConnectionContext.Destroy;
begin
  fRequirements.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------

 ----------------------------------------------------------------------------}

initialization
  g := TSSPIInterface.Create;
finalization
  g.Free;
end.
