
{*******************************************************}
{                                                       }
{       Misc Win32 API Interface Unit                   }
{                                                       }
{       Copyright (c) 2000, Eventree Systems            }
{                                                       }
{*******************************************************}

unit XWindows;

interface

uses

  Windows, Classes, SysUtils;

const

  NO_ERROR_SPECIFIED = High(Longword);

type

  { ECustomSysError }

  ECustomSysError = class(Exception)

  private
    fFailedFuncName: String;
    fErrorNo: Longword;
    function getErrorMessage: String;
  protected
    function GetSysErrorTypeStr: String; virtual;
  public
    class function GetErrorMessageByNo(
      aErrorNo: Longword): String; virtual; abstract;
    property ErrorMessage: String read getErrorMessage;
    property FailedFuncName: String read fFailedFuncName;
    property ErrorNo: Longword read fErrorNo;

  public
    constructor Create(aMessage: String);
    constructor CreateError(
      aFailedFuncName: String; anErrorNo: Longword = NO_ERROR_SPECIFIED);

  end;

  { EWinOSError }

  EWinOSError = class(ECustomSysError)

  public
    class function GetLastError: Longword;

  protected
    function GetSysErrorTypeStr: String; override;
  public
    class function GetErrorMessageByNo(aErrorNo: Longword): String; override;

  public
    constructor CreateError(
      aFailedFuncName: String; anErrorNo: Longword = NO_ERROR_SPECIFIED);

  end;

  { TProcess }

  TProcess = class(TObject)

  private
    fProcessInfo: TProcessInformation;
    fExitCode: DWord;
    fCmdLine, fCurDir: String;
    procedure setCmdLine(newValue: String);
    procedure setCurDir(newValue: String);
    procedure clearProcessInfo;
    function getRunning: Boolean;
    procedure setRunning(newValue: Boolean);
  protected
    procedure CheckIsNotRunning;
    procedure DoCreateProcess(
      aCmdLine, aCurDir: PChar; var aProcInfo: TProcessInformation); virtual;
  public
    procedure Start;
    procedure Stop;
    property Running: Boolean read getRunning write setRunning;
    property CmdLine: String read fCmdLine write setCmdLine;
    property CurDir: String read fCurDir write setCurDir;

  public
    constructor Create;

  end;

  { TCustomFileProcessor }

  TCustomFileProcessor = class(TObject)

  private
    fDirPattern: String;
  protected
    procedure ProcessFile
      (aFileName: String; var abortProcessing: Boolean); virtual; abstract;
  public
    function Process: Integer;
    property DirPattern: String read fDirPattern write fDirPattern;

  end;

  { TDLLLoader }

  EDLLLoader = class(EWinOSError)

  end;

  TDLLLoader = class(TObject)

  private
    fDLLHandle: THandle;
    fDLLName: String;
    function getLinked: Boolean;
  public
    procedure Link(aDLLName: String); virtual;
    procedure UnLink; virtual;
    property DLLName: String read fDLLName;
    property DLLHandle: THandle read fDLLHandle;
    property Linked: Boolean read getLinked;

  protected
    function FetchProcAddress(
      lpProcName: LPCSTR; var aProcAddress: FARPROC): Boolean;
  public
    function GetProcedure(aProcName: String): FARPROC; overload;
    function GetProcedure(aProcIndex: Integer): FARPROC; overload;
    function Exists(aProcName: String): Boolean;

  public
    constructor Create(aDLLName: String = '');
    destructor Destroy; override;

  end;

  { TWinHandleObject }

  TWinHandleObject = class(TObject)

  public
    class function WaitForOne(
      hs: array of THandle; aTimeOut: LongWord = INFINITE): Integer;

  protected
    function GetHandle: THandle; virtual; abstract;
  public
    property Handle: THandle read GetHandle;

  end;

  { TWinEvent }

  TWinEvent = class(TWinHandleObject)

  private
    fHandle: THandle;
  protected
    function GetHandle: THandle; override;
  public
    procedure SetEvent;
    procedure ResetEvent;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  { TCustomWinOverlappedIO }

  TCustomWinOverlappedIO = class(TWinHandleObject)

  private
    fOverlapped: TOverlapped;
    fIOEvent: TWinEvent;
  protected
    property Overlapped: TOverlapped read fOverlapped;
  public
    procedure Reset; virtual;
    function GetResult: Longint; virtual;
    property IOEvent: TWinEvent read fIOEvent;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  TCustomWinOverlappedReader = class(TCustomWinOverlappedIO)

  public
    procedure Submit(var Buffer; Count: Integer);

  end;

  TCustomWinOverlappedWriter = class(TCustomWinOverlappedIO)

  public
    procedure Submit(const Buffer; Count: Integer);

  end;

{ windows system info }

function GetWinVolumeInfo(aRootPath: String; var aVolName, aVolType: String;
  var aVolSerialNumber: DWord; var anError: Integer): Boolean;

{ temp file create }

function GenerateTempFileName
  (aPath: String = ''; aFilePrefix: String = ''): String;

{ windows NT shutdown }

procedure InitiateWinNTShutdown
  (aComputerName, aMsgToDisplay: String;
   aDspMsgTimeoutInSec: DWord; bForceAppsClosed, bRebootAfterShutdown: Boolean);
procedure AbortWinNTShutdown(aComputerName: String);

{ wait for sync objects }

const

  WAITTIMEDOUT = -1;

function WaitForAny
  (hs: array of THandle; aTimeOut: LongWord = INFINITE): Integer;
function WaitForAll
  (hs: array of THandle; aTimeOut: LongWord = INFINITE): Boolean;

{ some Windows date and time to TDateTime and back conversion functions }

function FileTimeToDateTime(const aFileTime: TFileTime): TDateTime;
function Int64ToDateTime(const aTimeStamp: Int64): TDateTime;
function DateTimeToFileTime(const aDateTime: TDateTime): TFileTime;
function DateTimeToInt64(const aDateTime: TDateTime): Int64;
function LocalToUTC(const aDateTime: TDateTime): TDateTime;
function UTCToLocal(const aDateTime: TDateTime): TDateTime;

implementation

function GetWinVolumeInfo(aRootPath: String; var aVolName, aVolType: String;
  var aVolSerialNumber: DWord; var anError: Integer): Boolean;
var
  MaximumComponentLength: DWord;
  FileSystemFlags: DWord;
  volNameBuffer, sysNameBuffer: array [0..255] of Char;
begin
  aRootPath := aRootPath + #0;
  Result :=GetVolumeInformation(
    PChar(aRootPath),
    volNameBuffer, SizeOf(volNameBuffer),
    @aVolSerialNumber,
    MaximumComponentLength,
    FileSystemFlags,
    sysNameBuffer, SizeOf(sysNameBuffer));
  if Result then begin
    aVolName := StrPas(volNameBuffer);
    aVolType := StrPas(sysNameBuffer);
  end else
    anError := GetLastError;
end;

{----------------------------------------------------------------------------
  ECustomSysError
 ----------------------------------------------------------------------------}

function ECustomSysError.getErrorMessage: String;
begin
  if fErrorNo = NO_ERROR_SPECIFIED then
    Result := 'unknown error'
  else
    Result := GetErrorMessageByNo(fErrorNo);
end;

function ECustomSysError.GetSysErrorTypeStr: String;
begin
  Result := 'System';
end;

{----------------------------------------------------------------------------}

constructor ECustomSysError.Create(aMessage: String);
begin
  fFailedFuncName := 'unknown';
  fErrorNo := NO_ERROR_SPECIFIED;
  inherited Create(aMessage);
end;

constructor ECustomSysError.CreateError
  (aFailedFuncName: String; anErrorNo: Longword = NO_ERROR_SPECIFIED);
begin
  fFailedFuncName := aFailedFuncName;
  fErrorNo := anErrorNo;
  inherited CreateFmt(
    '%s %s returns error #%d(0x%x): %s',
    [GetSysErrorTypeStr, FailedFuncName, ErrorNo, ErrorNo, ErrorMessage]);
end;

{----------------------------------------------------------------------------
  EWinOSError
 ----------------------------------------------------------------------------}

class function EWinOSError.GetLastError: Longword;
begin
  Result := Windows.GetLastError;
end;

{----------------------------------------------------------------------------}

class function EWinOSError.GetErrorMessageByNo(aErrorNo: Longword): String;
var
  buf: array [0..511] of Char;
begin
  FormatMessage(
    FORMAT_MESSAGE_FROM_SYSTEM or FORMAT_MESSAGE_ARGUMENT_ARRAY,
    nil, aErrorNo, LANG_NEUTRAL, buf, SizeOf(buf)-2, nil);
  buf[StrLen(buf) - 2] := #0;  //remove cr and newline character
  Result := StrPas(@buf);
end;

function EWinOSError.GetSysErrorTypeStr: String;
begin
  Result := 'Win API';
end;

{----------------------------------------------------------------------------}

constructor EWinOSError.CreateError
  (aFailedFuncName: String; anErrorNo: Longword = NO_ERROR_SPECIFIED);
begin
  if anErrorNo = NO_ERROR_SPECIFIED then
    anErrorNo := GetLastError;
  inherited CreateError(aFailedFuncName, anErrorNo);
end;

{----------------------------------------------------------------------------
  GenerateTempFileName
 ----------------------------------------------------------------------------}

function GenerateTempFileName
  (aPath: String = ''; aFilePrefix: String = ''): String;
var
  pathNameBuffer, fileNameBuffer,
  filePrefix: array [0..MAX_PATH] of Char;
begin
  { get temp directory name }
  if Length(aPath) > 0 then
    StrPCopy(pathNameBuffer, aPath)
  else
    if GetTempPath(MAX_PATH, pathNameBuffer) = 0 then
      raise EWinOSError.CreateError(
        Format('GetTempPath(%d, buffer)', [MAX_PATH]));
  { get temp file name }
  if Length(aFilePrefix) > 0 then
    StrPCopy(filePrefix, aFilePrefix)
  else
    StrPCopy(filePrefix, 'TMP');
  if GetTempFileName(pathNameBuffer, filePrefix, 0, fileNameBuffer) = 0 then
    raise EWinOSError.CreateError(
      Format('GetTempFileName("%s", "%s", 0, buffer)',
        [pathNameBuffer, aFilePrefix]));
  Result := StrPas(fileNameBuffer);
end;

{----------------------------------------------------------------------------
  TProcess
 ----------------------------------------------------------------------------}

const

  cnstErrChgNotAllowed    = 'Can not change property while process is running';
  cnstErrCmdLineNotSpeced = 'Command line is not specified';

procedure TProcess.clearProcessInfo;
begin
  with fProcessInfo do begin
    hProcess := INVALID_HANDLE_VALUE;
    hThread := INVALID_HANDLE_VALUE;
    dwProcessId := 0;
    dwThreadId := 0;
  end;
end;

function TProcess.getRunning: Boolean;
begin
  with fProcessInfo do begin
    if hProcess <> INVALID_HANDLE_VALUE then begin
      if not GetExitCodeProcess(hProcess, fExitCode) then
        raise EWinOSError.CreateError('GetExitCodeProcess');
      if fExitCode <> STILL_ACTIVE then
        clearProcessInfo;
    end;
    Result := hProcess <> INVALID_HANDLE_VALUE;
  end;
end;

procedure TProcess.setRunning(newValue: Boolean);
begin
  if Running <> newValue then begin
    if newValue then
      Start
    else
      Stop;
  end;
end;

procedure TProcess.CheckIsNotRunning;
begin
  if Running then
    raise Exception.Create(cnstErrChgNotAllowed);
end;

procedure TProcess.setCmdLine(newValue: String);
begin
  if fCmdLine <> newValue then begin
    CheckIsNotRunning;
    fCmdLine := newValue;
  end;
end;

procedure TProcess.setCurDir(newValue: String);
begin
  if fCurDir <> newValue then begin
    CheckIsNotRunning;
    fCurDir := newValue;
  end;
end;

procedure TProcess.DoCreateProcess
  (aCmdLine, aCurDir: PChar; var aProcInfo: TProcessInformation);
var
  si: TStartupInfo;
begin
  FillChar(si, SizeOf(si), 0);
  if not CreateProcess(nil, aCmdLine, nil, nil, True,
      NORMAL_PRIORITY_CLASS, nil, aCurDir, si, aProcInfo) then
    raise EWinOSError.CreateError('CreateProcess');
end;

procedure TProcess.Start;
var
  pCmdLine, pCurDir: PChar;
begin
  if Running then
    exit;
  if Length(fCmdLine) = 0 then
    raise Exception.Create(cnstErrCmdLineNotSpeced)
  else
    pCmdLine := PChar(fCmdLine + #0);
  if Length(fCurDir) > 0 then
    pCurDir := PChar(fCurDir + #0)
  else
    pCurDir := nil;
  fExitCode := 0;
  clearProcessInfo;
  DoCreateProcess(pCmdLine, pCurDir, fProcessInfo);
end;

procedure TProcess.Stop;
begin
  if not Running then
    exit;
  { ??? ... ??? }
end;

constructor TProcess.Create;
begin
  inherited Create;
  fCmdLine := '';
  fCurDir  := '';
  clearProcessInfo;
  fExitCode := 0;
end;

{----------------------------------------------------------------------------
  TCustomFileProcessor
 ----------------------------------------------------------------------------}

function TCustomFileProcessor.Process: Integer;
var
  SearchRec: TSearchRec;
  doAbort: Boolean;
begin
  Result := 0;
  if SysUtils.FindFirst(
      fDirPattern, faAnyFile - faDirectory, SearchRec) = 0 then
    try
      doAbort := False;
      repeat
        ProcessFile(SearchRec.Name, doAbort);
        if doAbort then
          break;
        Inc(Result);
      until SysUtils.FindNext(SearchRec) <> 0;
    finally
      SysUtils.FindClose(SearchRec);
    end;
end;

{----------------------------------------------------------------------------
  TDLLLoader
 ----------------------------------------------------------------------------}

const

 INVALID_DLL_HANDLE = 0;

function TDLLLoader.getLinked: Boolean;
begin
  Result := DLLHandle <> INVALID_DLL_HANDLE;
end;

procedure TDLLLoader.Link(aDLLName: String);
begin
  if fDLLName <> aDLLName then begin
    UnLink;
    fDLLHandle := LoadLibrary(PChar(aDLLName));
    if fDLLHandle = INVALID_DLL_HANDLE then
      raise EDLLLoader.CreateError('LoadLibrary');
    fDLLName := aDLLName;
  end;
end;

procedure TDLLLoader.UnLink;
begin
  if fDLLHandle <> INVALID_DLL_HANDLE then begin
    FreeLibrary(fDLLHandle);
    fDLLHandle := INVALID_DLL_HANDLE;
  end;
  fDLLName := '';
end;

{----------------------------------------------------------------------------}

function TDLLLoader.FetchProcAddress
  (lpProcName: LPCSTR; var aProcAddress: FARPROC): Boolean;
begin
  if DLLHandle = INVALID_DLL_HANDLE then
    raise EDLLLoader.Create('DLL is not loaded. Use Link(aDLLName) first.');
  aProcAddress := GetProcAddress(DLLHandle, lpProcName);
  Result := aProcAddress <> nil;
end;

function TDLLLoader.GetProcedure(aProcName: String): FARPROC;
begin
  if not FetchProcAddress(PChar(aProcName), Result) then
    raise EDLLLoader.CreateFmt(
      'GetProcAddress has failed to find [%s] in [%s]',
      [aProcName, DLLName]);
end;

function TDLLLoader.GetProcedure(aProcIndex: Integer): FARPROC;
var
  w: DWord;
begin
  w := aProcIndex;
  if HiWord(w) <> 0 then
    raise EDLLLoader.CreateFmt(
      'Procedure index [%d] to GetProcAddress for [%s] is invalid',
      [aProcIndex, DLLName]);
  if not FetchProcAddress(PChar(@w), Result) then
    raise EDLLLoader.CreateFmt(
      'GetProcAddress has failed to find procedure with index=%d in [%s]',
      [aProcIndex, DLLName]);
end;

function TDLLLoader.Exists(aProcName: String): Boolean;
var
  not_used: FARPROC;
begin
  Result := FetchProcAddress(PChar(aProcName), not_used);
end;

{----------------------------------------------------------------------------}

constructor TDLLLoader.Create(aDLLName: String = '');
begin
  inherited Create;
  fDLLName := '';
  fDLLHandle := INVALID_DLL_HANDLE;
  Link(aDLLName);
end;

destructor TDLLLoader.Destroy;
begin
  UnLink;
  inherited Destroy;
end;


{----------------------------------------------------------------------------
  TWinHandleObject
 ----------------------------------------------------------------------------}

const

  WAIT_TIMEOUT_RESULT = -1;

class function TWinHandleObject.WaitForOne
  (hs: array of THandle; aTimeOut: LongWord = INFINITE): Integer;
var
  r, qty: DWord;
begin
  Result := WAIT_TIMEOUT_RESULT;
  qty := Length(hs);
  r := WaitForMultipleObjects(qty, @hs, False, aTimeOut);
  if r = WAIT_TIMEOUT then
    Result := WAIT_TIMEOUT_RESULT
  else if {$WARNINGS OFF} (WAIT_OBJECT_0 <= r) {$WARNINGS ON} and
      (r < (WAIT_OBJECT_0 + qty)) then
    Result := Low(hs) + Integer(r) - WAIT_OBJECT_0
  else if r = WAIT_FAILED then
    EWinOSError.CreateError('WaitForMultipleObjects')
  else if (WAIT_ABANDONED_0 <= r) and (r < WAIT_ABANDONED_0 + qty) then
    EWinOSError.CreateFmt(
      'Abandoned handle in position %d signalled', [r]);
end;

{----------------------------------------------------------------------------
  TWinEvent
 ----------------------------------------------------------------------------}

function TWinEvent.GetHandle: THandle;
begin
  Result := fHandle;
end;

procedure TWinEvent.ResetEvent;
begin
  if not Windows.ResetEvent(Handle) then
    EWinOSError.CreateError('ResetEvent');
end;

procedure TWinEvent.SetEvent;
begin
  if not Windows.SetEvent(Handle) then
    EWinOSError.CreateError('SetEvent');
end;

{----------------------------------------------------------------------------}

constructor TWinEvent.Create;
begin
  inherited Create;
  fHandle := Windows.CreateEvent(nil, True, False, nil);
end;

destructor TWinEvent.Destroy;
begin
  if not Windows.CloseHandle(Handle) then
    EWinOSError.CreateError('CloseHandle');
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomWinOverlappedIO
 ----------------------------------------------------------------------------}

function TCustomWinOverlappedIO.GetResult: Longint;
begin
  if not GetOverlappedResult(Handle, fOverlapped, LongWord(Result), False) then
    raise EWinOSError.CreateError('GetOverlappedResult');
  IOEvent.ResetEvent;
end;

procedure TCustomWinOverlappedIO.Reset;
begin
  IOEvent.ResetEvent;
end;

{----------------------------------------------------------------------------}

constructor TCustomWinOverlappedIO.Create;
begin
  inherited Create;
  fIOEvent := TWinEvent.Create;
  FillChar(fOverlapped, SizeOf(fOverlapped), 0);
  fOverlapped.hEvent := IOEvent.Handle;
end;

destructor TCustomWinOverlappedIO.Destroy;
begin
  fIOEvent.Free;
  inherited Destroy;
end;

{----------------------------------------------------------------------------
  TCustomWinOverlappedReader
 ----------------------------------------------------------------------------}

procedure TCustomWinOverlappedReader.Submit(var Buffer; Count: Integer);
var
  aResultToDiscard: Longword;
begin
  if (not ReadFile(Handle, Buffer, Count, aResultToDiscard, @Overlapped))
      and (GetLastError <> ERROR_IO_PENDING) then
    raise EWinOSError.CreateError('ReadFile');
end;

{----------------------------------------------------------------------------
  TCustomWinOverlappedWriter
 ----------------------------------------------------------------------------}

procedure TCustomWinOverlappedWriter.Submit(const Buffer; Count: Integer);
var
  aResultToDiscard: Longword;
begin
  if (not WriteFile(Handle, Buffer, Count, aResultToDiscard, @fOverlapped))
      and (GetLastError <> ERROR_IO_PENDING) then
    raise EWinOSError.CreateError('WriteFile');
end;

{----------------------------------------------------------------------------
  Windows NT shutdown
 ----------------------------------------------------------------------------}

procedure InitiateWinNTShutdown
  (aComputerName, aMsgToDisplay: String;
   aDspMsgTimeoutInSec: DWord; bForceAppsClosed, bRebootAfterShutdown: Boolean);
var
  pMsg: PChar;
begin
  if Length(aMsgToDisplay) > 0 then
    pMsg := PChar(aMsgToDisplay)
  else
    pMsg := nil;
  if not InitiateSystemShutdown(
      PChar(aComputerName), pMsg,
      aDspMsgTimeoutInSec, bForceAppsClosed, bRebootAfterShutdown) then
    raise EWinOSError.CreateError('InitiateSystemShutdown');
end;

procedure AbortWinNTShutdown(aComputerName: String);
begin
  if not AbortSystemShutdown(PChar(aComputerName)) then
    raise EWinOSError.CreateError('AbortSystemShutdown');
end;

{----------------------------------------------------------------------------
  WaitFor*
 ----------------------------------------------------------------------------}

function waitForIDontKnow
  (hs: array of THandle; bWaitAll: Boolean; aTimeOut: LongWord): Integer;
var
  r, qty: DWord;
begin
  Result := WAITTIMEDOUT;
  qty := Length(hs);
  r := WaitForMultipleObjects(qty, @hs, bWaitAll, aTimeOut);
  if r = WAIT_TIMEOUT then
    Result := WAITTIMEDOUT
  else if {(WAIT_OBJECT_0 <= r) and  without the comment it generates warning}
      (r < (WAIT_OBJECT_0 + qty)) then
    Result := Low(hs) + Integer(r) - WAIT_OBJECT_0
  else if r = WAIT_FAILED then
    RaiseLastOSError
  else if (WAIT_ABANDONED_0 <= r) and (r < WAIT_ABANDONED_0 + qty) then begin
    if bWaitAll then
      raise Exception.Create(
        'All objects signalled, but at least one has been abandoned')
    else
      raise Exception.CreateFmt(
        'Abandoned mutex in position %d signalled', [r]);
  end;
end;

function WaitForAny
  (hs: array of THandle; aTimeOut: LongWord = INFINITE): Integer;
begin
  Result := waitForIDontKnow(hs, False, aTimeOut);
end;

function WaitForAll
  (hs: array of THandle; aTimeOut: LongWord = INFINITE): Boolean;
begin
  Result := waitForIDontKnow(hs, True, aTimeOut) <> WAIT_TIMEOUT;
end;

{----------------------------------------------------------------------------
  FileTimeToDateTime and Int64ToDateTime
 ----------------------------------------------------------------------------}

function FileTimeToDateTime(const aFileTime: TFileTime): TDateTime;
var
  st: TSystemTime;
begin
  if not FileTimeToSystemTime(aFileTime, st) then
    raise EWinOSError.CreateError('FileTimeToSystemTime');
  Result := SystemTimeToDateTime(st);
end;

function Int64ToDateTime(const aTimeStamp: Int64): TDateTime;
var
  ft: TFileTime;
begin
  with ft do begin
    dwHighDateTime := (aTimeStamp shr 32) and $ffffffff;
    dwLowDateTime := aTimeStamp and $ffffffff;
  end;
  Result := FileTimeToDateTime(ft);
end;

function DateTimeToFileTime(const aDateTime: TDateTime): TFileTime;
var
  st: TSystemTime;
begin
  DateTimeToSystemTime(aDateTime, st);
  if not SystemTimeToFileTime(st, Result) then
    raise EWinOSError.CreateError('SystemTimeToFileTime');
end;

function DateTimeToInt64(const aDateTime: TDateTime): Int64;
var
  ft: TFileTime;
begin
  ft := DateTimeToFileTime(aDateTime);
  Result :=
    (ft.dwHighDateTime shl 32) and $ffffffff00000000 +
    ft.dwLowDateTime and $ffffffff;
end;

function GetLocalBias: TDateTime;
var
  TZ: TTimeZoneInformation;
  bias: Longint;
begin
  case GetTimeZoneInformation(TZ) of
  TIME_ZONE_ID_STANDARD:
    bias := TZ.Bias + TZ.StandardBias;
  TIME_ZONE_ID_DAYLIGHT:
    bias := TZ.Bias + TZ.DaylightBias;
  else
    bias := TZ.Bias;
  end;
  Result := EncodeTime(Abs(bias) div 60, Abs(bias) mod 60, 0, 0);
  if bias < 0 then
    Result := - Result;
end;

function LocalToUTC(const aDateTime: TDateTime): TDateTime;
begin
  Result := aDateTime + GetLocalBias;
end;

function UTCToLocal(const aDateTime: TDateTime): TDateTime;
begin
  Result := aDateTime - GetLocalBias;
end;

end.
