unit IdThreadMgr;

(*
Changes

02 Oct 2001 - Allen O'Neill - Added support for thread priority - new property Threadpriority, new line added to OnCreate
*)

interface

uses
  Classes,
  IdException, IdBaseComponent, IdGlobal, IdThread,
  SyncObjs;

type
  TIdThreadMgr = class(TIdBaseComponent)
  protected
    FActiveThreads: TThreadList;
    FThreadClass: TIdThreadClass;
    FThreadPriority: TIdThreadPriority;
  public
    constructor Create(AOwner: TComponent); override;
    function CreateNewThread: TIdThread; virtual;
    destructor Destroy; override;
    function GetThread: TIdThread; virtual; abstract;
    procedure ReleaseThread(AThread: TIdThread); virtual; abstract;
    procedure TerminateThreads(TerminateWaitTime: integer); virtual;
    //
    property ActiveThreads: TThreadList read FActiveThreads;
    property ThreadClass: TIdThreadClass read FThreadClass write FThreadClass;
    property ThreadPriority: TIdThreadPriority read FThreadPriority
     write FThreadPriority default tpNormal;
  end;

  EIdThreadMgrError = class(EIdException);
  EIdThreadClassNotSpecified = class(EIdThreadMgrError);

implementation

uses
  IdResourceStrings, IdTCPServer,
  SysUtils;

{ TIdThreadMgr }

constructor TIdThreadMgr.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FActiveThreads := TThreadList.Create;
end;

function TIdThreadMgr.CreateNewThread: TIdThread;
begin
  if ThreadClass = nil then begin
    raise EIdThreadClassNotSpecified.create(RSThreadClassNotSpecified);
  end;
  Result := TIdThreadClass(ThreadClass).Create;
  SetThreadPriority(Result, ThreadPriority);
end;

destructor TIdThreadMgr.Destroy;
begin
  FreeAndNil(FActiveThreads);
  inherited Destroy;
end;

procedure TIdThreadMgr.TerminateThreads(TerminateWaitTime: integer);
Var
  LTimedOut: Boolean;
  i: Integer;
  LThreads: TList;
const
  LSleepTime: integer = 250;
begin
  LTimedOut := True;
  for i := 1 to (TerminateWaitTime div LSleepTime) do begin
    Sleep(LSleepTime);
    LThreads := ActiveThreads.LockList; try
      if LThreads.Count = 0 then begin
        LTimedOut := False;
        Break;
      end;
     finally ActiveThreads.UnlockList; end;
  end;
  if LTimedOut then begin
    raise EIdTerminateThreadTimeout.Create(RSTerminateThreadTimeout);
  end;
end;

end.
