unit IdThread;
{$I IdCompilerDefines.inc}
interface

uses
  Classes,
  IdGlobal,
  SysUtils, SyncObjs;

type
  TIdThreadStopMode = (smTerminate, smSuspend);
  TIdExceptionEvent = procedure(ASender: TObject; AException: Exception) of object;
  TIdThread = class;
  TIdNotifyThreadEvent = procedure(AThread: TIdThread) of object;

  TIdThread = class(TThread)
  protected
    FData: TObject;
    FLock: TCriticalSection;
    FStopMode: TIdThreadStopMode;
    FStopped: Boolean;
    FTerminatingException: string;
    FOnException: TIdExceptionEvent;
    FOnStopped: TIdNotifyThreadEvent;
    //
    procedure AfterRun; virtual; // Not abstract - otherwise it is required
    procedure BeforeRun; virtual; // Not abstract - otherwise it is required
    procedure DoStopped; virtual;
    procedure Execute; override;
    function GetStopped: Boolean;
    procedure Run; virtual; abstract;
    procedure Cleanup; virtual;
  public
    constructor Create(ACreateSuspended: Boolean = True); virtual;
    destructor Destroy; override;
    procedure Start; virtual;
    procedure Stop; virtual;
    // Synchronize -  Here to expose it
    procedure Synchronize(Method: TThreadMethod);
    // Here to make virtual
    procedure Terminate; virtual;
    procedure TerminateAndWaitFor; virtual;
    //
    property Data: TObject read FData write FData;
    property StopMode: TIdThreadStopMode read FStopMode write FStopMode;
    property Stopped: Boolean read GetStopped;
    property OnException: TIdExceptionEvent read FOnException
     write FOnException;
    property OnStopped: TIdNotifyThreadEvent read FOnStopped write FOnStopped;
    // Terminated exposed
    property Terminated;
    property TerminatingException: string read FTerminatingException;
  end;

  TIdThreadClass = class of TIdThread;

{$IFNDEF CBUILDER}
{$IFDEF INDY_THREADVAR}
ThreadVar
  // Warning: This cannot be used outside of the package.
  // It can only be used when compiled into the EXE, or inside the package
  ThreadSelf: TIdThread;
{$ENDIF}
{$ENDIF}

implementation

procedure TIdThread.TerminateAndWaitFor;
begin
  Terminate;
  FStopped := True;
  WaitFor;
end;

procedure TIdThread.AfterRun;
begin
end;

procedure TIdThread.BeforeRun;
begin
end;

procedure TIdThread.Execute;
begin
  try
    {$IFDEF INDY_THREADVAR}
    ThreadSelf := Self;
    {$ENDIF}
    while not Terminated do begin
      if Stopped then begin
        DoStopped;
        // It is possible that either in the DoStopped or from another thread,
        // the thread is restarted, in which case we dont want to restop it.
        if Stopped then begin
          Suspended := True; // Thread manager will revive us
          if Terminated then begin
            Break;
          end;
        end;
      end;
      BeforeRun; try
        while not Stopped do begin
          Run;
        end;
      finally
        AfterRun;
        Cleanup;
      end;
    end;
  except
    on E: Exception do begin
      FTerminatingException := E.Message;
      if Assigned(FOnException) then begin
        FOnException(self, E);
      end;
      Terminate;
    end;
  end;
end;

procedure TIdThread.Synchronize(Method: TThreadMethod);
begin
  inherited Synchronize(Method);
end;

constructor TIdThread.Create(ACreateSuspended: Boolean);
begin
  // Before inherited - inherited creates the actual thread and if not suspeded
  // will start before we initialize
  FStopped := ACreateSuspended;
  FLock := TCriticalSection.Create;
  try
    inherited Create(ACreateSuspended);
  except
    FreeAndNil(FLock);
    raise;
  end;
end;

destructor TIdThread.Destroy;
begin
  FreeAndNil(FLock);
  Cleanup;
  inherited Destroy;
end;

procedure TIdThread.Start;
begin
  FLock.Enter; try
    if Stopped then begin
      // Resume is also called for smTerminate as .Start can be used to initially start a
      // thread that is created suspended
      FStopped := False;
      Suspended := False;
    end;
  finally FLock.Leave; end;
end;

procedure TIdThread.Stop;
begin
  FLock.Enter; try
    if not Stopped then begin
      case FStopMode of
        smTerminate: Terminate;
        // DO NOT suspend here. Suspend is immediate. See Execute for implementation
        smSuspend: ;
      end;
      FStopped := True;
    end;
  finally FLock.Leave; end;
end;

function TIdThread.GetStopped: Boolean;
begin
  FLock.Enter; try
    // Suspended may be true if checking stopped from another thread
    Result := Terminated or FStopped or Suspended;
  finally FLock.Leave; end;
end;

procedure TIdThread.DoStopped;
begin
  if Assigned(OnStopped) then begin
    OnStopped(Self);
  end;
end;

procedure TIdThread.Terminate;
begin
  inherited Terminate;
end;

procedure TIdThread.Cleanup;
begin
  FreeAndNil(FData);
end;

end.
