unit IdTCPConnection;

interface

uses
  Classes,
  IdException, IdComponent, IdGlobal, IdSocketHandle, IdIntercept, IdIOHandler, IdRFCReply;

const
  GRecvBufferSizeDefault = 32768;
  GSendBufferSizeDefault = 32768;

type
  //TODO: Make this a buffered stream for more efficiency.
  TIdBuffer = class(TMemoryStream)
  public
    procedure RemoveXBytes(const AByteCount: integer);
  end;

  TIdTCPConnection = class(TIdComponent)
  protected
    FASCIIFilter: boolean;
    FBuffer: TIdBuffer;
    // TODO - Change the "move" functions to read write functinos. Get as much as possible down
    // to just TStream so we can replace it easily
    FClosedGracefully: Boolean;
    FCmdResultDetails: TStrings;
    FFreeIOHandlerOnDisconnect: Boolean;
    FIntercept: TIdConnectionIntercept;
    FIOHandler: TIdIOHandler;
    FOnDisconnected: TNotifyEvent;
    FReadLnTimedOut: Boolean;
    FReadTimeout: Integer;
    FRecvBufferSize: Integer;
    FRecvBuffer: TIdBuffer; // To be used by ReadFromStack only
    FResultNo: SmallInt;
    FSendBufferSize: Integer;
    FWriteBuffer: TIdBuffer;
    FWriteBufferThreshhold: Integer;
    //
    procedure DoOnDisconnected; virtual;
    function GetCmdResult: string;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure ResetConnection; virtual;
    procedure SetIntercept(AValue: TIdConnectionIntercept);
    procedure SetIOHandler(AValue: TIdIOHandler);
  public
    function AllData: string; virtual;
    procedure CancelWriteBuffer;
    procedure Capture(ADest: TObject; const ADelim: string = '.';
     const AIsRFCMessage: Boolean = True);
    procedure CheckForDisconnect(const ARaiseExceptionIfDisconnected: boolean = true;
     const AIgnoreBuffer: boolean = false); virtual;
    procedure CheckForGracefulDisconnect(const ARaiseExceptionIfDisconnected: Boolean = True);
     virtual;
    function CheckResponse(const AResponse: SmallInt; const AAllowedResponses: array of SmallInt)
     : SmallInt; virtual;
    procedure ClearWriteBuffer;
    procedure CloseWriteBuffer;
    function Connected: Boolean; virtual;
    constructor Create(AOwner: TComponent); override;
    function CurrentReadBuffer: string;
    function CurrentReadBufferSize: Integer;
    destructor Destroy; override;
    procedure Disconnect; virtual;
    procedure DisconnectSocket; virtual;
    function ExtractXBytesFromBuffer(const AByteCount: Integer): string; virtual;
    procedure FlushWriteBuffer(const AByteCount: Integer = -1);
    procedure GetInternalResponse;
    function GetResponse(const AAllowedResponses: array of SmallInt): SmallInt; overload; virtual;
    function GetResponse(const AAllowedResponse: SmallInt): SmallInt; overload;
    function InputLn(const AMask: string = ''): string;
    procedure OpenWriteBuffer(const AThreshhold: Integer = -1);
    // RaiseExceptionForCmdResult - Overload necesary as a exception as a default param doesnt work
    procedure RaiseExceptionForCmdResult; overload; virtual;
    procedure RaiseExceptionForCmdResult(axException: TClassIdException); overload; virtual;
    procedure ReadBuffer(var ABuffer; const AByteCount: Longint);
    function ReadCardinal(const AConvert: boolean = true): Cardinal;
    // ReadFromStack must be only call to Recv
    function ReadFromStack(const ARaiseExceptionIfDisconnected: Boolean = True;
     ATimeout: Integer = IdTimeoutDefault;
     const ARaiseExceptionOnTimeout: Boolean = True): Integer; virtual;
    function ReadInteger(const AConvert: boolean = true): Integer;
    function ReadLn(const ATerminator: string = '';
     const ATimeout: integer = IdTimeoutDefault): string; virtual;
    function ReadLnWait: string;
    function ReadSmallInt(const AConvert: boolean = true): SmallInt;
    procedure ReadStream(AStream: TStream; AByteCount: LongInt = -1;
     const AReadUntilDisconnect: boolean = false);
    function ReadString(const ABytes: Integer): string;
    procedure ReadStrings(var AValue: TStrings; AReadLinesCount: Integer = -1);
    procedure RemoveXBytesFromBuffer(const AByteCount: Integer); virtual;
    function SendCmd(const AOut: string; const AResponse: SmallInt = -1): SmallInt; overload;
    function SendCmd(const AOut: string; const AResponse: Array of SmallInt): SmallInt; overload;
     virtual;
    function WaitFor(const AString: string): string;
    procedure Write(AOut: string); virtual;
    // WriteBuffer must be the ONLY call to SEND - all data goes thru this method
    procedure WriteBuffer(const ABuffer; AByteCount: Longint; const AWriteNow: boolean = False);
    procedure WriteCardinal(AValue: Cardinal; const AConvert: boolean = True);
    procedure WriteHeader(axHeader: TStrings);
    procedure WriteInteger(AValue: Integer; const AConvert: boolean = True);
    procedure WriteLn(const AOut: string = ''); virtual;
    procedure WriteRFCReply(AReply: TIdRFCReply);
    procedure WriteRFCStrings(AStrings: TStrings);
    procedure WriteSmallInt(AValue: SmallInt; const AConvert: boolean = True);
    procedure WriteStream(AStream: TStream; const AAll: boolean = True;
     const AWriteByteCount: Boolean = False); virtual;
    procedure WriteStrings(AValue: TStrings; const AWriteLinesCount: Boolean = False);
    function WriteFile(AFile: String; const AEnableTransferFile: boolean = False): cardinal;
     virtual;
    //
    property ClosedGracefully: Boolean read FClosedGracefully;
    property CmdResult: string read GetCmdResult;
    property CmdResultDetails: TStrings read FCmdResultDetails;
    property ReadLnTimedOut: Boolean read FReadLnTimedOut;
    property ResultNo: SmallInt read FResultNo;
  published
    property ASCIIFilter: boolean read FASCIIFilter write FASCIIFilter default False;
    property Intercept: TIdConnectionIntercept read FIntercept write SetIntercept;
    property IOHandler: TIdIOHandler read FIOHandler write SetIOHandler;
    property OnDisconnected: TNotifyEvent read FOnDisconnected write FOnDisconnected;
    property OnWork;
    property OnWorkBegin;
    property OnWorkEnd;
    property ReadTimeout: Integer read FReadTimeout write FReadTimeout;
    property RecvBufferSize: Integer read FRecvBufferSize write FRecvBufferSize
     default GRecvBufferSizeDefault;
    property SendBufferSize: Integer read FSendBufferSize write FSendBufferSize
     default GSendBufferSizeDefault;
  end;

  EIdTCPConnectionError = class(EIdException);
  EIdObjectTypeNotSupported = class(EIdTCPConnectionError);
  EIdNotEnoughDataInBuffer = class(EIdTCPConnectionError);
  EIdInterceptPropIsNil = class(EIdTCPConnectionError);
  EIdInterceptPropInvalid = class(EIdTCPConnectionError);
  EIdIOHandlerPropInvalid = class(EIdTCPConnectionError);
  EIdNoDataToRead = class(EIdTCPConnectionError);

implementation

uses
  IdAntiFreezeBase, IdStack, IdStackConsts, IdStream, IdResourceStrings, IdIOHandlerSocket,
  SysUtils;

function TIdTCPConnection.AllData: string;
begin
  BeginWork(wmRead); try
    Result := '';
    while Connected do begin
      Result := Result + CurrentReadBuffer;
    end;
  finally EndWork(wmRead); end;
end;

procedure TIdTCPConnection.Capture(ADest: TObject; const ADelim: string = '.';
  const AIsRFCMessage: Boolean = True);
var
  s: string;
begin
  BeginWork(wmRead); try
    repeat
      s := ReadLn(EOL);
      if s = ADelim then begin
        Exit;
      end;

      // For RFC 822 retrieves
      if AIsRFCMessage and (Copy(s, 1, 2) = '..') then begin
        Delete(s, 1, 1);
      end;

      // Write to output
      if ADest is TStrings then begin
        TStrings(ADest).Add(s);
      end else if ADest is TStream then begin
        TIdStream(ADest).WriteLn(s);
      end else if ADest <> nil then begin
        raise EIdObjectTypeNotSupported.Create(RSObjectTypeNotSupported);
      end;
    until False;
  finally EndWork(wmRead); end;
end;

procedure TIdTCPConnection.CheckForDisconnect(const ARaiseExceptionIfDisconnected: Boolean = True;
 const AIgnoreBuffer: Boolean = False);
var
  LDisconnected: Boolean;
begin
  LDisconnected := False;
  // ClosedGracefully // Server disconnected
  // IOHandler = nil // Client disconnected
  if ClosedGracefully or (IOHandler = nil) then begin
    if IOHandler <> nil then begin
      if IOHandler.Connected then begin
      	DisconnectSocket;
        // Call event handlers to inform the user program that we were disconnected
        DoStatus(hsDisconnected);
        DoOnDisconnected;
      end;
      LDisconnected := True;
    end;
  end else if IOHandler <> nil then begin
    LDisconnected := not IOHandler.Connected;
  end;
  if LDisconnected then begin
    // Do not raise unless all data has been read by the user
    if ((CurrentReadBufferSize = 0) or AIgnoreBuffer) and ARaiseExceptionIfDisconnected then begin
      (* ************************************************************* //
      ------ If you receive an exception here, please read. ----------

      If this is a SERVER
      -------------------
      The client has disconnected the socket normally and this exception is used to notify the
      server handling code. This exception is normal and will only happen from within the IDE, not
      while your program is running as an EXE. If you do not want to see this, add this exception
      or EIdSilentException to the IDE options as exceptions not to break on.

      From the IDE just hit F9 again and Indy will catch and handle the exception.

      Please see the FAQ and help file for possible further information.
      The FAQ is at http://www.nevrona.com/Indy/FAQ.html

      If this is a CLIENT
      -------------------
      The server side of this connection has disconnected normaly but your client has attempted
      to read or write to the connection. You should trap this error using a try..except.
      Please see the help file for possible further information.

      // ************************************************************* *)
      raise EIdConnClosedGracefully.Create(RSConnectionClosedGracefully);
    end;
  end;
end;

function TIdTCPConnection.Connected: Boolean;
begin
  CheckForDisconnect(False);
  Result := IOHandler <> nil;
  if Result then begin
    Result := IOHandler.Connected;
  end;
end;

constructor TIdTCPConnection.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCmdResultDetails := TStringList.Create;
  FRecvBuffer := TIdBuffer.Create;

  RecvBufferSize := GRecvBufferSizeDefault;
  FSendBufferSize := GSendBufferSizeDefault;
  FBuffer := TIdBuffer.Create;
end;

function TIdTCPConnection.CurrentReadBuffer: string;
begin
  Result := '';
  if Connected then begin
    ReadFromStack(False);
  end;

  Result := ExtractXBytesFromBuffer(FBuffer.Size);
end;

function TIdTCPConnection.CurrentReadBufferSize: integer;
begin
  Result := FBuffer.Size;
end;

destructor TIdTCPConnection.Destroy;
begin
  // DisconnectSocket closes IOHandler etc. Dont call Disconnect - Disconnect may be override and
  // try to read/write to the socket.
  DisconnectSocket;

  // Becouse DisconnectSocket does not free the IOHandler we have to do it here.
  if FFreeIOHandlerOnDisconnect then begin
    FreeAndNil(FIOHandler);
    FFreeIOHandlerOnDisconnect := False;
  end;

  FreeAndNil(FBuffer);
  FreeAndNil(FRecvBuffer);
  FreeAndNil(FCmdResultDetails);
  inherited Destroy;
end;

procedure TIdTCPConnection.Disconnect;
begin
  DoStatus(hsDisconnecting);
  DisconnectSocket;
  // NOT in DisconnectSocket. DisconnectSocket is used to kick ReadFromStack and others
  // out of their blocking calls and they rely on the binding after that
  if FFreeIOHandlerOnDisconnect then begin
    FreeAndNil(FIOHandler);
    FFreeIOHandlerOnDisconnect := False;
  end;
  DoOnDisconnected;
  DoStatus(hsDisconnected);
end;

procedure TIdTCPConnection.DoOnDisconnected;
begin
  if Assigned(OnDisconnected) then begin
    OnDisconnected(Self);
  end;
end;

function TIdTCPConnection.ExtractXBytesFromBuffer(const AByteCount: Integer): string;
begin
  if AByteCount > FBuffer.Size then begin
    raise EIdNotEnoughDataInBuffer.Create(RSNotEnoughDataInBuffer);
  end;
  SetString(result, PChar(FBuffer.Memory), AByteCount);
  RemoveXBytesFromBuffer(AByteCount);
end;

function TIdTCPConnection.GetCmdResult: string;
begin
  Result := '';
  if CmdResultDetails.Count > 0 then begin
    Result := CmdResultDetails[CmdResultDetails.Count - 1];
  end;
end;

function TIdTCPConnection.GetResponse(const AAllowedResponses: array of SmallInt): SmallInt;
begin
  GetInternalResponse;
  FResultNo := StrToIntDef(Copy(CmdResult, 1, 3), 0);
  Result := CheckResponse(ResultNo, AAllowedResponses);
end;

procedure TIdTCPConnection.RaiseExceptionForCmdResult(axException: TClassIdException);
begin
  raise axException.Create(CmdResult);
end;

procedure TIdTCPConnection.RaiseExceptionForCmdResult;
begin
  raise EIdProtocolReplyError.CreateError(ResultNo, CmdResult);
end;

procedure TIdTCPConnection.ReadBuffer(var ABuffer; const AByteCount: Integer);
begin
  if (AByteCount > 0) and (@ABuffer <> nil) then begin
    // Read from stack until we have enough data
    while (CurrentReadBufferSize < AByteCount) do begin
      ReadFromStack;
      CheckForDisconnect(True, True);
    end;
    // Copy it to the callers buffer
    Move(PChar(FBuffer.Memory)[0], ABuffer, AByteCount);
    // Remove used data from buffer
    RemoveXBytesFromBuffer(AByteCount);
  end;
end;

function TIdTCPConnection.ReadFromStack(const ARaiseExceptionIfDisconnected: Boolean = True;
 ATimeout: Integer = IdTimeoutDefault; const ARaiseExceptionOnTimeout: Boolean = True): Integer;
// Reads any data in tcp/ip buffer and puts it into Indy buffer
// This must be the ONLY raw read from Winsock routine
// This must be the ONLY call to RECV - all data goes thru this method
var
  i: Integer;
  LByteCount: Integer;
begin
  if ATimeout = IdTimeoutDefault then begin
    if ReadTimeOut = 0 then begin
      ATimeout := IdTimeoutInfinite;
    end else begin
      ATimeout := FReadTimeout;
    end;
  end;
  Result := 0;
  // Check here as this side may have closed the socket
  CheckForDisconnect(ARaiseExceptionIfDisconnected);
  if Connected then begin
    if IOHandler.Readable(ATimeout) then begin
      FRecvBuffer.Size := RecvBufferSize;
      // No need to call AntiFreeze, the Readable does that.
      LByteCount := IOHandler.Recv(FRecvBuffer.Memory^, FRecvBuffer.Size);
      FClosedGracefully := LByteCount = 0;
      if not ClosedGracefully then begin
        if GStack.CheckForSocketError(LByteCount, [Id_WSAESHUTDOWN, Id_WSAECONNABORTED]) then begin
          LByteCount := 0;
          if IOHandler <> nil then begin
            DisconnectSocket;
          end;
          // Do not raise unless all data has been read by the user
          if CurrentReadBufferSize = 0 then begin
            GStack.RaiseSocketError(GStack.LastError);
          end;
        end;
        // FBuffer.Size is modified above
        if LByteCount > 0 then begin
          FRecvBuffer.Size := LByteCount;
          if Assigned(Intercept) then begin
            FRecvBuffer.Position := 0;
            Intercept.Receive(FRecvBuffer);
          end;
          if ASCIIFilter then begin
            for i := 1 to FRecvBuffer.Size do begin
              PChar(FRecvBuffer.Memory)[i] := Chr(Ord(PChar(FRecvBuffer.Memory)[i]) and $7F);
            end;
          end;
          FBuffer.Position := FBuffer.Size;
          FBuffer.WriteBuffer(FRecvBuffer.Memory^, FRecvBuffer.Size);
        end;
      end;
      // Check here as other side may have closed connection
      CheckForDisconnect(ARaiseExceptionIfDisconnected);
      Result := LByteCount;
    end else begin
      // Timeout
      if ARaiseExceptionOnTimeout then begin
        raise EIdReadTimeout.Create(RSReadTimeout);
      end;
      Result := -1;
    end;
  end;
end;

function TIdTCPConnection.ReadInteger(const AConvert: boolean = true): Integer;
begin
  ReadBuffer(Result, SizeOf(Result));
  if AConvert then begin
    Result := Integer(GStack.WSNToHL(LongWord(Result)));
  end;
end;

function TIdTCPConnection.ReadLn(const ATerminator: string = '';
 const ATimeout: integer = IdTimeoutDefault): string;
var
  i: Integer;
  LTerminator: string;
begin
  if Length(ATerminator) = 0 then begin
    LTerminator := LF;
  end else begin
    LTerminator := ATerminator;
  end;
  FReadLnTimedOut := False;
  i := 0;
  repeat
    if CurrentReadBufferSize > 0 then begin
      i := SubStrPos(LTerminator, PChar(FBuffer.Memory), FBuffer.Size);
    end;
    // ReadFromStack blocks - do not call unless we need to
    if i = 0 then begin
      // ReadLn needs to call this as data may exist in the buffer, but no EOL yet disconnected
      CheckForDisconnect(True, True);
      // Can only return 0 if error or timeout
      FReadLnTimedOut := ReadFromStack(True, ATimeout, ATimeout = IdTimeoutDefault) = -1;
      if ReadLnTimedout then begin
        Result := '';
        Exit;
      end;
    end;
  until i > 0;
  Result := ExtractXBytesFromBuffer(i + Length(LTerminator) - 1);
  SetLength(Result, i - 1);
  if (Length(ATerminator) = 0) and (Copy(Result, Length(Result), 1) = CR) then begin
    SetLength(Result, Length(Result) - 1);
  end;
end;

function TIdTCPConnection.ReadLnWait: string;
begin
  Result := '';
  while Length(Result) = 0 do begin
    Result := Trim(ReadLn);
  end;
end;

procedure TIdTCPConnection.ReadStream(AStream: TStream; AByteCount: Integer = -1;
 const AReadUntilDisconnect: Boolean = False);
var
  i: Integer;
  s: string;
  LWorkCount: Integer;

  procedure AdjustStreamSize(AStream: TStream; const ASize: integer);
  var
    LStreamPos: LongInt;
  begin
    LStreamPos := AStream.Position;
    AStream.Size := ASize;
    // Must reset to original size as in some cases size changes position
    if AStream.Position <> LStreamPos then begin
      AStream.Position := LStreamPos;
    end;
  end;

begin
  if (AByteCount = -1) and (AReadUntilDisconnect = False) then begin
    // Read size from connection
    AByteCount := ReadInteger;
  end;
  // Presize stream if we know the size - this reduces memory/disk allocations to one time
  if AByteCount > -1 then begin
    AdjustStreamSize(AStream, AStream.Position + AByteCount);
  end;

  if AReadUntilDisconnect then begin
    LWorkCount := High(LWorkCount);
    BeginWork(wmRead);
  end else begin
    LWorkCount := AByteCount;
    BeginWork(wmRead, LWorkCount);
  end;
  try
    // If data already exists in the buffer, write it out first.
    if CurrentReadBufferSize > 0 then begin
      i := Min(CurrentReadBufferSize, LWorkCount);
      FBuffer.Position := 0;
      AStream.CopyFrom(FBuffer, i);
      FBuffer.RemoveXBytes(i);
      Dec(LWorkCount, i);
    end;
    while Connected and (LWorkCount > 0) do begin
      i := Min(LWorkCount, RecvBufferSize);
      //TODO: Improve this - dont like the use of the exception handler
      try
        try
          s := ReadString(i);
        except
          on E: EIdConnClosedGracefully do begin
            if AReadUntilDisconnect then begin
              s := CurrentReadBuffer;
            end else begin
              raise;
            end;
          end;
        end;
      finally
        if Length(S) > 0 then begin
          AStream.WriteBuffer(s[1], Length(s));
          Dec(LWorkCount, Length(S));
        end;
      end;
    end;
  finally EndWork(wmRead); end;
  if AStream.Size > AStream.Position then begin
    AStream.Size := AStream.Position;
  end;
end;

procedure TIdTCPConnection.RemoveXBytesFromBuffer(const AByteCount: Integer);
begin
  FBuffer.RemoveXBytes(AByteCount);
  DoWork(wmRead, AByteCount);
end;

procedure TIdTCPConnection.ResetConnection;
begin
  FBuffer.Clear;
  FClosedGracefully := False;
end;

function TIdTCPConnection.SendCmd(const AOut: string; const AResponse: Array of SmallInt): SmallInt;
begin
  if AOut <> #0 then begin
    WriteLn(AOut);
  end;
  Result := GetResponse(AResponse);
end;

procedure TIdTCPConnection.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, OPeration);

  if (Operation = opRemove) then begin
    if (AComponent = FIntercept) then begin
      FIntercept := nil;
    end;

    if (AComponent = FIOHandler) then begin
      FIOHandler := nil;
    end;
  end;
end;

procedure TIdTCPConnection.SetIntercept(AValue: TIdConnectionIntercept);
begin
  FIntercept := AValue;
  // add self to the Intercept's free notification list
  if Assigned(FIntercept) then begin
    FIntercept.FreeNotification(Self);
  end;
end;

procedure TIdTCPConnection.SetIOHandler(AValue: TIdIOHandler);
begin
  FIOHandler := AValue;
  // add self to the IOHandler's free notification list
  if Assigned(FIOHandler) then begin
    FIOHandler.FreeNotification(Self);
  end;
end;

procedure TIdTCPConnection.Write(AOut: string);
begin
  if Length(AOut) > 0 then begin
    WriteBuffer(AOut[1], length(AOut));
  end;
end;

procedure TIdTCPConnection.WriteBuffer(const ABuffer; AByteCount: Integer;
 const AWriteNow: boolean = false);
var
  LBuffer: TIdBuffer;
  nPos, nByteCount: Integer;
begin
  if (AByteCount > 0) and (@ABuffer <> nil) then begin
    // Check if disconnected
    CheckForDisconnect(True, True);

    if (FWriteBuffer = nil) or AWriteNow then begin
      LBuffer := TIdBuffer.Create; try
        LBuffer.WriteBuffer(ABuffer, AByteCount);
        if Assigned(Intercept) then begin
          LBuffer.Position := 0;
          Intercept.Send(LBuffer);
        end;
        nPos := 1;
        repeat
          nByteCount := IOHandler.Send(PChar(LBuffer.Memory)[nPos - 1], LBuffer.Size - nPos + 1);
          // Write always does someting - never retuns 0
          // TODO - Have a AntiFreeze param which allows the send to be split up so that process
          // can be called more. Maybe a prop of the connection, MexSendSize?
          TIdAntiFreezeBase.DoProcess(False);
          FClosedGracefully := nByteCount = 0;
          //NOTE - this is currently kind of a hack - there is a newer/better plan that I have to find time
          //to implement
          //DoProcess;

          // Check if other side disconnected
          CheckForDisconnect;
          // Check to see if the error signifies disconnection
          if GStack.CheckForSocketError(nByteCount
           , [ID_WSAESHUTDOWN, Id_WSAECONNABORTED, Id_WSAECONNRESET]) then begin
            DisconnectSocket;
            GStack.RaiseSocketError(GStack.WSGetLastError);
          end;
          DoWork(wmWrite, nByteCount);
          nPos := nPos + nByteCount
        until nPos > AByteCount;
      finally FreeAndNil(LBuffer); end;
    // Write Buffering is enabled
    end else begin
      FWriteBuffer.WriteBuffer(ABuffer, AByteCount);
      if (FWriteBuffer.Size >= FWriteBufferThreshhold) and (FWriteBufferThreshhold > 0) then begin
        // TODO: Instead of flushing - Write until buffer is smaller than Threshold. That is do at
        // least one physical send.
        FlushWriteBuffer(FWriteBufferThreshhold);
      end;
    end;
  end;
end;

function TIdTCPConnection.WriteFile(AFile: String; const AEnableTransferFile: boolean = False)
 : Cardinal;
var
  LFileStream: TFileStream;
begin
  if Assigned(GServeFileProc) and (Intercept = nil) and AEnableTransferFile
   and (IOHandler is TIdIOHandlerSocket) then begin
    Result := GServeFileProc(TIdIOHandlerSocket(IOHandler).Binding.Handle, AFile);
  end else begin
    LFileStream := TFileStream.Create(AFile, fmOpenRead	or fmShareDenyNone); try
      WriteStream(LFileStream);
      Result := LFileStream.Size;
    finally LFileStream.free; end;
  end;
end;

procedure TIdTCPConnection.WriteHeader(axHeader: TStrings);
var
  i: Integer;
begin
  for i := 0 to axHeader.Count -1 do begin
    // No ReplaceAll flag - we only want to replace the first one
    WriteLn(StringReplace(axHeader[i], '=', ': ', []));
  end;
  WriteLn('');
end;

procedure TIdTCPConnection.WriteInteger(AValue: Integer; const AConvert: Boolean = True);
begin
  if AConvert then begin
    AValue := Integer(GStack.WSHToNl(LongWord(AValue)));
  end;
  WriteBuffer(AValue, SizeOf(AValue));
end;

procedure TIdTCPConnection.WriteLn(const AOut: string = '');
begin
  Write(AOut + EOL);
end;

procedure TIdTCPConnection.WriteStream(AStream: TStream; const AAll: boolean = true;
 const AWriteByteCount: Boolean = false);
var
  LBuffer: TMemoryStream;
  LSize: Integer;
  LStreamSize: Integer;
begin
  if AAll then begin
    AStream.Position := 0;
  end;
  // This is copied to a local var because accessing .Size is very inefficient
  LStreamSize := AStream.Size;
  LSize := LStreamSize - AStream.Position;
  if AWriteByteCount then begin
  	WriteInteger(LSize);
  end;
  BeginWork(wmWrite, LSize); try
    LBuffer := TMemoryStream.Create; try
      LBuffer.SetSize(FSendBufferSize);
      while True do begin
        LSize := Min(LStreamSize - AStream.Position, FSendBufferSize);
        if LSize = 0 then begin
          Break;
        end;
        // Do not use ReadBuffer. Some source streams are real time and will not
        // return as much data as we request. Kind of like recv()
        // NOTE: We use .Size - size must be supported even if real time
        LSize := AStream.Read(LBuffer.Memory^, LSize);
        if LSize = 0 then begin
          raise EIdNoDataToRead.Create(RSIdNoDataToRead);
        end;
        WriteBuffer(LBuffer.Memory^, LSize);
      end;
    finally FreeAndNil(LBuffer); end;
  finally EndWork(wmWrite); end;
end;

procedure TIdTCPConnection.WriteStrings(AValue: TStrings; const AWriteLinesCount: Boolean = False);
var
  i: Integer;
begin
  if AWriteLinesCount then begin
    WriteInteger(AValue.Count);
  end;
  for i := 0 to AValue.Count - 1 do begin
    WriteLn(AValue.Strings[i]);
  end;
end;

function TIdTCPConnection.SendCmd(const AOut: string; const AResponse: SmallInt): SmallInt;
begin
  if AResponse = -1 then begin
    Result := SendCmd(AOut, []);
  end else begin
    Result := SendCmd(AOut, [AResponse]);
  end;
end;

procedure TIdTCPConnection.DisconnectSocket;
begin
  if IOHandler <> nil then begin
    FClosedGracefully := True;
    // In design time don't use propertyes which point to other compoenents
    if not (csDesigning in ComponentState) then begin
      if Assigned(Intercept) then begin
        Intercept.Disconnect;
      end;
      IOHandler.Close;
    end;
  end;
end;

procedure TIdTCPConnection.OpenWriteBuffer(const AThreshhold: Integer = -1);
begin
  FWriteBuffer := TIdBuffer.Create;
  FWriteBufferThreshhold := AThreshhold;
end;

procedure TIdTCPConnection.CloseWriteBuffer;
begin
  FlushWriteBuffer;
  FreeAndNil(FWriteBuffer);
end;

procedure TIdTCPConnection.FlushWriteBuffer(const AByteCount: Integer = -1);
begin
  if FWriteBuffer.Size > 0 then begin
    if (AByteCount = -1) or (FWriteBuffer.Size < AByteCount) then begin
      WriteBuffer(PChar(FWriteBuffer.Memory)[0], FWriteBuffer.Size, True);
      ClearWriteBuffer;
    end else begin
      WriteBuffer(PChar(FWriteBuffer.Memory)[0], AByteCount, True);
      FWriteBuffer.RemoveXBytes(AByteCount);
    end;
  end;
end;

procedure TIdTCPConnection.ClearWriteBuffer;
begin
  FWriteBuffer.Clear;
end;

function TIdTCPConnection.InputLn(const AMask: string =''): string;
var
  s: string;
begin
  result := '';
  while true do begin
    s := ReadString(1);
    if s = BACKSPACE then begin
      if length(result) > 0 then begin
        SetLength(result, Length(result) - 1);
        Write(BACKSPACE);
      end;
    end else if s = CR then begin
      ReadString(1); // LF
      WriteLn;
      exit;
    end else begin
      result := result + s;
      if Length(AMask) = 0 then begin
        Write(s);
      end else begin
        Write(AMask);
      end;
    end;
  end;
end;

function TIdTCPConnection.ReadString(const ABytes: Integer): string;
begin
  SetLength(Result, ABytes);
  if ABytes > 0 then begin
    ReadBuffer(Result[1], Length(Result));
  end;
end;

procedure TIdTCPConnection.ReadStrings(var AValue: TStrings; AReadLinesCount: Integer = -1);
Var
  i: Integer;
begin
  if AReadLinesCount <= 0 then begin
    AReadLinesCount := ReadInteger;
  end;
  for i := 0 to AReadLinesCount - 1 do begin
    AValue.Add(ReadLn);
  end;
end;

procedure TIdTCPConnection.CancelWriteBuffer;
begin
  ClearWriteBuffer;
  CloseWriteBuffer;
end;

function TIdTCPConnection.ReadSmallInt(const AConvert: boolean = true): SmallInt;
begin
  ReadBuffer(Result, SizeOf(Result));
  if AConvert then begin
    Result := SmallInt(GStack.WSNToHs(Word(Result)));
  end;
end;

procedure TIdTCPConnection.WriteSmallInt(AValue: SmallInt; const AConvert: boolean = true);
begin
  if AConvert then begin
    AValue := SmallInt(GStack.WSHToNs(Word(AValue)));
  end;
  WriteBuffer(AValue, SizeOf(AValue));
end;

procedure TIdTCPConnection.CheckForGracefulDisconnect(const ARaiseExceptionIfDisconnected: boolean);
begin
  ReadFromStack(ARaiseExceptionIfDisconnected, 1, False);
end;

{ TIdBuffer }

procedure TIdBuffer.RemoveXBytes(const AByteCount: integer);
begin
  if AByteCount > Size then begin
    raise EIdNotEnoughDataInBuffer.Create(RSNotEnoughDataInBuffer);
  end;
  if AByteCount = Size then begin
    Clear;
  end else begin
    Move(PChar(Memory)[AByteCount], PChar(Memory)[0], Size - AByteCount);
    SetSize(Size - AByteCount);
  end;
end;

function TIdTCPConnection.WaitFor(const AString: string): string;
//TODO: Add a time out (default to infinite) and event to pass data
begin
  Result := '';
  // NOTE: AnsiPos should be used here, but AnsiPos has problems if result has any #0 in it,
  // which is often the case. So currently this function is not MBCS compliant and should
  // not be used in MBCS environments. However this function should only be used on incoming
  // TCP text data as it is 7 bit.
  while Pos(AString, Result) = 0 do begin
    Result := Result + CurrentReadBuffer;
    CheckForDisconnect;
  end;
end;

function TIdTCPConnection.ReadCardinal(const AConvert: boolean): Cardinal;
begin
  ReadBuffer(Result, SizeOf(Result));
  if AConvert then begin
    Result := GStack.WSNToHL(Result);
  end;
end;

procedure TIdTCPConnection.WriteCardinal(AValue: Cardinal; const AConvert: boolean);
begin
  if AConvert then begin
    AValue := GStack.WSHToNl(AValue);
  end;
	WriteBuffer(AValue, SizeOf(AValue));
end;

function TIdTCPConnection.CheckResponse(const AResponse: SmallInt;
 const AAllowedResponses: array of SmallInt): SmallInt;
var
  i: Integer;
  LResponseFound: Boolean;
begin
  if High(AAllowedResponses) > -1 then begin
    LResponseFound := False;
    for i := Low(AAllowedResponses) to High(AAllowedResponses) do begin
      if AResponse = AAllowedResponses[i] then begin
        LResponseFound := True;
        Break;
      end;
    end;
    if not LResponseFound then begin
      RaiseExceptionForCmdResult;
    end;
  end;
  Result := AResponse;
end;

procedure TIdTCPConnection.GetInternalResponse;
var
  LLine: string;
  LTerm: string;
begin
  CmdResultDetails.Clear;
  LLine := ReadLnWait;
  CmdResultDetails.Add(LLine);
  if Length(LLine) > 3 then begin
    if LLine[4] = '-' then begin // Multi line response coming
      LTerm := Copy(LLine, 1, 3) + ' ';
      {We keep reading lines until we encounter either a line such as "250"
       or "250 Read"}
      repeat
        LLine := ReadLnWait;
        CmdResultDetails.Add(LLine);
      until (Length(LLine) < 4) or (AnsiSameText(Copy(LLine, 1, 4), LTerm));
    end;
  end;
end;

procedure TIdTCPConnection.WriteRFCReply(AReply: TIdRFCReply);
begin
  if AReply.ReplyExists then begin
    Write(AReply.GenerateReply);
  end;
end;

procedure TIdTCPConnection.WriteRFCStrings(AStrings: TStrings);
var
  i: Integer;
begin
  for i := 0 to AStrings.Count - 1 do begin
    if AStrings[i] = '.' then begin
      WriteLn('..');
    end else begin
      WriteLn(AStrings[i]);
    end;
  end;
  WriteLn('.');
end;

function TIdTCPConnection.GetResponse(const AAllowedResponse: SmallInt): SmallInt;
begin
  Result := GetResponse([AAllowedResponse]);
end;

end.
