unit IdSync;

interface

uses
  Classes,
  IdGlobal, IdThread;

type
  TIdSync = class(TObject)
  protected
    FThread: TIdThread;
  public
    constructor Create(AThread: TIdThread); reintroduce; virtual;
    procedure DoSynchronize; virtual; abstract;
    procedure Notify;
    class procedure NotifyMethod(AMethod: TThreadMethod);
    procedure Synchronize;
    //
    property Thread: TIdThread read FThread;
  end;

  TIdSyncMethod = class(TIdSync)
  protected
    FMethod: TThreadMethod;
  public
    constructor Create(AMethod: TThreadMethod); reintroduce;
    procedure DoSynchronize; override;
  end;

implementation

uses
  SysUtils;

type
  // This is done with a NotifyThread instead of PostMessage because starting with D6/Kylix Borland
  // radically modified the mecanisms for .Synchronize. This is a bit more code in the end, but
  // its source compatible and does not rely on Indy directly accessing any OS APIs and performance
  // is still more than acceptable, especiall considering Notifications are low priority.
  TIdNotifyThread = class(TThread)
  protected
    FEvent: TIdLocalEvent;
    FNotifications: TThreadList;
  public
    procedure AddNotification(ASync: TIdSync);
    constructor Create(ASuspended: Boolean);
    destructor Destroy; override;
    procedure Execute; override;
  end;

var
  GNotifyThread: TIdNotifyThread = nil;

{ TIdSync }

constructor TIdSync.Create(AThread: TIdThread);
begin
  inherited Create;
  FThread := AThread;
end;

procedure TIdSync.Notify;
begin
  GNotifyThread.AddNotification(Self);
end;

class procedure TIdSync.NotifyMethod(AMethod: TThreadMethod);
begin
  TIdSyncMethod.Create(AMethod).Notify;
end;

procedure TIdSync.Synchronize;
begin
  FThread.Synchronize(DoSynchronize);
end;

{ TIdNotifyThread }

procedure TIdNotifyThread.AddNotification(ASync: TIdSync);
begin
  FNotifications.Add(ASync);
  FEvent.SetEvent;
end;

constructor TIdNotifyThread.Create(ASuspended: Boolean);
begin
  // FreeOnTerminate := True;
  FEvent := TIdLocalEvent.Create;
  FNotifications := TThreadList.Create;
  // Must be before - Thread starts running when we call inherited
  inherited Create(ASuspended);
end;

destructor TIdNotifyThread.Destroy;
begin
  // Free remaining Notifications if thre is somthing that is still in
  // the queue after thread was terminated
  with FNotifications.LockList do try
    while Count > 0 do begin
      TIdSync(Items[0]).Free;
      Delete(0);
    end;
  finally FNotifications.UnlockList; end;
  FreeAndNil(FNotifications);
  FreeAndNil(FEvent);
  inherited Destroy;
end;

procedure TIdNotifyThread.Execute;
// NOTE: Be VERY careful with making changes to this proc. It is VERY delicate and the order
// of execution is very important. Small changes can have drastic effects
var
  LNotify: TList;
  LSync: TIdSync;
begin
  repeat
    FEvent.WaitFor;
    // If terminated while waiting on the event or during the loop
    while not Terminated do begin
      try
        LNotify := FNotifications.LockList; try
          if LNotify.Count = 0 then begin
            Break;
          end;
          LSync := TIdSync(LNotify.Items[0]);
          LNotify.Delete(0);
        finally FNotifications.UnlockList; end;
        Synchronize(LSync.DoSynchronize);
        FreeAndNil(LSync);
      except // Catch all exceptions especially these which are raised during the application close
      end;
    end;
  until Terminated;
end;

{ TIdSyncMethod }

constructor TIdSyncMethod.Create(AMethod: TThreadMethod);
begin
  FMethod := AMethod;
end;

procedure TIdSyncMethod.DoSynchronize;
begin
  FMethod;
end;

initialization
  GNotifyThread := TIdNotifyThread.Create(False);
finalization
  // Will free itself using FreeOnTerminate
  GNotifyThread.Terminate;
  GNotifyThread.FEvent.SetEvent;
  GNotifyThread.WaitFor;
  GNotifyThread.Free;
end.

