{

  Implementation of the NTLM authentication with SSPI

  Author: Alex Brainman
  Copyright: (c) Chad Z. Hower and The Winshoes Working Group.

}

unit IdSSPIAuthtentication;

interface

uses
  xsspi, IdAuthentication;

type
  TIndySSPINTLMClient = class(TObject)
  protected
    fNTLMPackage: TSSPINTLMPackage;
    fCredentials: TSSPIWinNTCredentials;
    fContext: TSSPIClientConnectionContext;
  public
    procedure SetCredentials(aDomain, aUserName, aPassword: String);
    procedure SetCredentialsAsCurrentUser;
    function InitAndBuildType1Message: String;
    function UpdateAndBuildType3Message(aServerType2Message: String): String;

  public
    constructor Create;
    destructor Destroy; override;

  end;

  TIdSSPINTLMAuthentication = class(TIdAuthentication)
  protected
    FNTLMInfo: String;
    FSSPIClient: TIndySSPINTLMClient;

    function DoNext: TIdAuthWhatsNext; override;
  public
    constructor Create; override;
    function Authentication: String; override;

    procedure Reset; override;
  end;

implementation

Uses
  SysUtils,
  IdGlobal,
  IdException,
  IdCoderMIME;

{----------------------------------------------------------------------------
  TIndySSPINTLMClient
 ----------------------------------------------------------------------------}

procedure TIndySSPINTLMClient.SetCredentials
  (aDomain, aUserName, aPassword: String);
begin
  fCredentials.Acquire(scuOutBound, aDomain, aUserName, aPassword);
end;

procedure TIndySSPINTLMClient.SetCredentialsAsCurrentUser;
begin
  fCredentials.Acquire(scuOutBound);
end;

function TIndySSPINTLMClient.InitAndBuildType1Message: String;
begin
  fContext.GenerateInitialChalenge('', Result);
end;

function TIndySSPINTLMClient.UpdateAndBuildType3Message
  (aServerType2Message: String): String;
begin
  fContext.UpdateAndGenerateReply(aServerType2Message, Result);
end;

{ ------------------------------------------------------------------------}

constructor TIndySSPINTLMClient.Create;
begin
  inherited Create;
  fNTLMPackage := TSSPINTLMPackage.Create;
  fCredentials := TSSPIWinNTCredentials.Create(fNTLMPackage);
  fContext := TSSPIClientConnectionContext.Create(fCredentials);
end;

destructor TIndySSPINTLMClient.Destroy;
begin
  fContext.Free;
  fCredentials.Free;
  fNTLMPackage.Free;
  inherited Destroy;
end;

{ TIdSSPINTLMAuthentication }

constructor TIdSSPINTLMAuthentication.Create;
begin
  inherited Create;

  FSSPIClient := TIndySSPINTLMClient.Create;
end;

function TIdSSPINTLMAuthentication.DoNext: TIdAuthWhatsNext;
begin
  result := wnDoRequest;

  case FStatus of
    0:
      begin
        if (Length(Username) > 0) and (Length(Password) > 0) and (FRetries = 0) then
        begin
          result := wnDoRequest;
          FStatus := 1;
        end
        else begin
          result := wnAskTheProgram;
        end;
      end;
    1:
      begin
        FStatus := 2;
        result := wnDoRequest;
      end;
    3: begin
      FStatus := 1;
      result := wnDoRequest;
    end;
  end;
end;

function TIdSSPINTLMAuthentication.Authentication: String;
Var
  S: String;
begin
  result := '';
  case FStatus of
    1:
      begin
        FSSPIClient.SetCredentials(IndyGetHostName, Username, Password);
        result := 'NTLM ' + TIdEncoderMIME.EncodeString(FSSPIClient.InitAndBuildType1Message);
        FNTLMInfo := '';
      end;
    2:
      begin
        if Length(FNTLMInfo) = 0 then
        begin
          FNTLMInfo := ReadAuthInfo('NTLM');
          Fetch(FNTLMInfo);
        end;
        if Length(FNTLMInfo) = 0 then
        begin
          FStatus := 1;
          Abort;
        end;

        S := TIdDecoderMIME.DecodeString(FNTLMInfo);
        result := 'NTLM ' + TIdEncoderMIME.EncodeString(FSSPIClient.UpdateAndBuildType3Message(S));

        FStatus := 3;
      end;
  end;
end;

procedure TIdSSPINTLMAuthentication.Reset;
begin
  inherited Reset;
  // FStatus := 0;
end;

initialization
  RegisterAuthenticationMethod('NTLM', TIdSSPINTLMAuthentication);
end.
