unit IdSNMP;

{
-2001.02.13 - Kudzu - Misc "Indy" Changes.
-Contributions also by: Hernan Sanchez (hernan.sanchez@iname.com)
-Original Author: Lukas Gebauer

The Synapse SNMP component was converted for use in INDY.

| The Original Code is Synapse Delphi Library.                                 |
|==============================================================================|
| The Initial Developer of the Original Code is Lukas Gebauer (Czech Republic).|
| Portions created by Lukas Gebauer are Copyright (c)2000.                     |
| All Rights Reserved.                                                         |
|==============================================================================|
| Contributor(s):                                                              |
|   Hernan Sanchez (hernan.sanchez@iname.com)                                  |
|==============================================================================|
| History: see HISTORY.HTM from distribution package                           |
|          (Found at URL: http://www.ararat.cz/synapse/)                       |
|==============================================================================}

interface

uses
  Classes,
  SysUtils,
  IdUDPBase,
  IdUDPClient;

const
  //PDU type
  PDUGetRequest=$a0;
  PDUGetNextRequest=$a1;
  PDUGetResponse=$a2;
  PDUSetRequest=$a3;
  PDUTrap=$a4;

  //errors
  ENoError=0;
  ETooBig=1;
  ENoSuchName=2;
  EBadValue=3;
  EReadOnly=4;
  EGenErr=5;

  ASN1_INT       = $02;
  ASN1_OCTSTR    = $04;
  ASN1_NULL      = $05;
  ASN1_OBJID     = $06;
  ASN1_SEQ       = $30;
  ASN1_IPADDR    = $40;
  ASN1_COUNTER   = $41;
  ASN1_GAUGE     = $42;
  ASN1_TIMETICKS = $43;

type
  TSNMPInfo=class(TObject)
  protected
    Buffer: string;
    procedure SyncMIB;
  public
    Host : string;
    Port : integer;
    Enterprise: string;
    GenTrap: integer;
    SpecTrap: integer;
    Version : integer;
    Community : string;
    PDUType : integer;
    TimeTicks : integer;
    ID : integer;
    ErrorStatus : integer;
    ErrorIndex : integer;
    MIBOID : TStringList;
    MIBValue : TStringList;

    constructor Create;
    destructor  Destroy; override;
    function    EncodeTrap: integer;
    function    DecodeTrap: integer;
    procedure   DecodeBuf(Buffer:string);
    function    EncodeBuf:string;
    procedure   Clear;
    procedure   MIBAdd(MIB,Value:string);
    procedure   MIBDelete(Index:integer);
    function    MIBGet(MIB:string):string;
  end;

  TIdSNMP = class(TIdUDPClient)
  public
    Query : TSNMPInfo;
    Reply : TSNMPInfo;
    Trap  : TSNMPInfo;

    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;
    function SendQuery : boolean;
    function QuickSend(Mib, Community, Host:string; var Value:string):Boolean;
    function QuickSendTrap(Dest, Enterprise, Community: string;
                      Port, Generic, Specific: integer; MIBName, MIBValue: TStringList): integer;
    function QuickReceiveTrap(var Source, Enterprise, Community: string;
                      var Port, Generic, Specific, Seconds: integer; var MIBName, MIBValue: TStringList): integer;
    function SendTrap: integer;
    function ReceiveTrap: integer;
  end;

// Procs
  function ASNEncLen(Len: integer): string;
  function ASNDecLen(var Start: integer; Buffer: string): integer;
  function ASNEncInt(Len: integer): string;
  function ASNObject(Data: string; ASNType: integer): string;
  function ASNItem(var Start: integer; Buffer: string): string;
  Function MibToId(mib:string):string;
  Function IdToMib(id:string):string;
  Function IntMibToStr(int:string):string;
  function IPToID(Host: string): string;

implementation

{========================== SNMP INFO OBJECT ==================================}
constructor TSNMPInfo.Create;
begin
  inherited create;
  MIBOID:=TStringList.create;
  MIBValue:=TStringList.create;
  Community := 'public';
end;

destructor TSNMPInfo.Destroy;
begin
  MIBValue.Free;
  MIBOID.Free;
  inherited destroy;
end;

procedure TSNMPInfo.SyncMIB;
var
  n,x:integer;
begin
  x:=MIBValue.Count;
  for n:=x to MIBOID.Count-1 do
    MIBValue.Add('');
end;

procedure TSNMPInfo.DecodeBuf(Buffer:string);
var
  Pos:integer;
  endpos:integer;
  sm,sv:string;
begin
  Pos:=2;
  Endpos:=ASNDecLen(Pos,buffer);
  Self.version:=StrToIntDef(ASNItem(Pos,buffer),0);
  Self.community:=ASNItem(Pos,buffer);
  Self.PDUType:=StrToIntDef(ASNItem(Pos,buffer),0);
  Self.ID:=StrToIntDef(ASNItem(Pos,buffer),0);
  Self.ErrorStatus:=StrToIntDef(ASNItem(Pos,buffer),0);
  Self.ErrorIndex:=StrToIntDef(ASNItem(Pos,buffer),0);
  ASNItem(Pos,buffer);
  while Pos<Endpos do
    begin
      ASNItem(Pos,buffer);
      Sm:=ASNItem(Pos,buffer);
      Sv:=ASNItem(Pos,buffer);
      Self.MIBadd(sm,sv);
    end;
end;

function TSNMPInfo.EncodeBuf:string;
var
  data,s:string;
  n:integer;
begin
  data:='';
  SyncMIB;
  for n:=0 to Self.MIBOID.Count-1 do
    begin
      s:=ASNObject(MibToID(Self.MIBOID[n]),6)+ASNObject(Self.MIBValue[n],4);
      data:=data+ASNObject(s,$30);
    end;
  data:=ASNObject(data,$30);
  data:=ASNObject(char(Self.ID),2)+ASNObject(char(Self.ErrorStatus),2)
         +ASNObject(char(Self.ErrorIndex),2)+data;
  data:=ASNObject(char(Self.Version),2)+ASNObject(Self.community,4)+ASNObject(data,Self.PDUType);
  data:=ASNObject(data,$30);
  Result:=data;
end;

procedure TSNMPInfo.Clear;
begin
  version:=0;
  community:='';
  PDUType:=0;
  ID:=0;
  ErrorStatus:=0;
  ErrorIndex:=0;
  MIBOID.Clear;
  MIBValue.Clear;
end;

procedure TSNMPInfo.MIBAdd(MIB,Value:string);
var
  x:integer;
begin
  SyncMIB;
  MIBOID.Add(MIB);
  x:=MIBOID.Count;
  if MIBValue.Count>x then MIBvalue[x-1]:=value
    else MIBValue.Add(Value);
end;

procedure TSNMPInfo.MIBDelete(Index:integer);
begin
  SyncMIB;
  MIBOID.Delete(Index);
  if (MIBValue.Count-1)>= Index then MIBValue.Delete(Index);
end;

function TSNMPInfo.MIBGet(MIB:string):string;
var
  x:integer;
begin
  SyncMIB;
  x:=MIBOID.IndexOf(MIB);
  if x<0 then Result:=''
    else Result:=MIBValue[x];
end;

{==============================  IdSNMP OBJECT ================================}
constructor TidSNMP.Create(aOwner : TComponent);
begin
  inherited;
  Query:=TSNMPInfo.Create;
  Reply:=TSNMPInfo.Create;
  Trap :=TSNMPInfo.Create;
  Query.Clear;
  Reply.Clear;
  Trap.Clear;
  FReceiveTimeout:=5000;
end;

destructor TidSNMP.Destroy;
begin
  Reply.Free;
  Query.Free;
  Trap.Free;
  inherited destroy;
end;



{======================= SEND QUERY =====================================}
function TidSNMP.SendQuery:boolean;
begin
  Result:=false;
  reply.clear;
  Query.Buffer:=Query.Encodebuf;
  Send(Query.host, Query.port, Query.buffer);
  reply.Buffer := ReceiveString(Query.host, Query.port, FReceiveTimeout);
  if reply.Buffer<>'' then reply.DecodeBuf(reply.Buffer);
end;

function TidSNMP.QuickSend (Mib, Community,Host:string; var Value:string):Boolean;
begin
  Query.community:=Community;
  Query.PDUType:=PDUGetRequest;
  Query.MIBAdd(MIB,'');
  Query.Host := Host;
  Result:=SendQuery;
  if Result then Value:=Reply.MIBGet(MIB);
end;

{======================= TRAPS =====================================}
function TSNMPInfo.EncodeTrap: integer;
var
  s: string;
  n: integer;
begin
  Buffer := '';
  for n:=0 to MIBOID.Count-1 do
    begin
      s := ASNObject(MibToID(MIBOID[n]), ASN1_OBJID)
        + ASNObject(MIBValue[n], ASN1_OCTSTR);
      Buffer := Buffer + ASNObject(s, ASN1_SEQ);
    end;
  Buffer := ASNObject(Buffer, ASN1_SEQ);
  Buffer := ASNObject(ASNEncInt(GenTrap), ASN1_INT)
    + ASNObject(ASNEncInt(SpecTrap), ASN1_INT)
    + ASNObject(ASNEncInt(TimeTicks), ASN1_TIMETICKS) + Buffer;
  Buffer := ASNObject(MibToID(Enterprise), ASN1_OBJID)
    + ASNObject(IPToID(Host), ASN1_IPADDR) + Buffer;
  Buffer := ASNObject(Char(Version), ASN1_INT)
    + ASNObject(Community, ASN1_OCTSTR) + ASNObject(Buffer, Self.PDUType);
  Buffer := ASNObject(Buffer, ASN1_SEQ);
  Result := 1;
end;

function TSNMPInfo.DecodeTrap: integer;
var
  Pos, EndPos: integer;
  Sm, Sv: string;
begin
  Pos := 2;
  EndPos := ASNDecLen(Pos, Buffer);
  Version := StrToIntDef(ASNItem(Pos, Buffer), 0);
  Community := ASNItem(Pos, Buffer);
  PDUType := StrToIntDef(ASNItem(Pos, Buffer), PDUTRAP);
  Enterprise := IdToMIB(ASNItem(Pos, Buffer));
  Host := ASNItem(Pos, Buffer);
  GenTrap := StrToIntDef(ASNItem(Pos, Buffer), 0);
  Spectrap := StrToIntDef(ASNItem(Pos, Buffer), 0);
  TimeTicks := StrToIntDef(ASNItem(Pos, Buffer), 0);
  ASNItem(Pos, Buffer);
  while (Pos < EndPos) do
    begin
      ASNItem(Pos, Buffer);
      Sm := ASNItem(Pos, Buffer);
      Sv := ASNItem(Pos, Buffer);
      MIBAdd(Sm, Sv);
    end;
  Result := 1;
end;

function TidSNMP.SendTrap: integer;
begin
  Trap.PDUType := PDUTrap;
  Trap.EncodeTrap;
  Send(Trap.Host, Trap.Port, Trap.Buffer);
  Result := 1;
end;

function TidSNMP.ReceiveTrap: integer;
begin
  Trap.PDUType := PDUTrap;
  Result := 0;
  Trap.Buffer := ReceiveString(trap.host, trap.port, FReceiveTimeout);
  if Trap.Buffer <> '' then begin
     Trap.DecodeTrap;
     Result := 1;
  end;
end;

function TidSNMP.QuickSendTrap(Dest, Enterprise, Community: string;
  Port, Generic, Specific: integer; MIBName, MIBValue: TStringList): integer;
var
  i: integer;
begin
    Trap.Host := Dest;
    Trap.Port := Port;
    Trap.Enterprise := Enterprise;
    Trap.Community := Community;
    Trap.GenTrap := Generic;
    Trap.SpecTrap := Specific;
    for i:=0 to (MIBName.Count - 1) do
      Trap.MIBAdd(MIBName[i], MIBValue[i]);
    Result := SendTrap;
end;

function TidSNMP.QuickReceiveTrap(var Source, Enterprise, Community: string;
  var Port, Generic, Specific, Seconds: integer; var MIBName, MIBValue: TStringList): integer;
var
  i: integer;
begin
    Result := ReceiveTrap;
    if (Result <> 0) then
    begin
      Source := Trap.Host;
      Port := Trap.Port;
      Enterprise := Trap.Enterprise;
      Community := Trap.Community;
      Generic := Trap.GenTrap;
      Specific := Trap.SpecTrap;
      Seconds := Trap.TimeTicks;
      MIBName.Clear;
      MIBValue.Clear;
      for i:=0 to (Trap.MIBOID.Count - 1) do
        begin
          MIBName.Add(Trap.MIBOID[i]);
          MIBValue.Add(Trap.MIBValue[i]);
        end;
    end;
end;




{======================= CODING FUNCTIONS =================================}
function ASNEncLen(Len: integer): string;
var
  x, y: integer;
begin
  if (Len < $80) then
    Result := Char(Len)
  else
    if (Len < $FF) then
      Result := Char($81) + Char(Len)
    else
      begin
        x := Len div $FF;
        y := Len mod $FF;
        Result := Char($82) + Char(x) + Char(y);
      end;
end;

function ASNDecLen(var Start: integer; Buffer: string): integer;
var
  x: integer;
begin
  x := Ord(Buffer[Start]);
  if (x < $80) then
    begin
      Inc(Start);
      Result := x;
    end
  else
    if (x = $81) then
      begin
        Inc(Start);
        Result := Ord(Buffer[Start]);
        Inc(Start);
      end
    else
      begin
        Inc(Start);
        x := Ord(Buffer[Start]);
        Inc(Start);
        Result := x * $FF + Ord(Buffer[Start]);
        Inc(Start);
      end;
end;

function ASNEncInt(Len: integer): string;
var
  j, y: integer;
begin
  Result := '';
  j := 0;
  y := Len div $FFFFFF;
  Len := Len - (y * $FFFFFF);
  if ((y > 0) or (j = 1)) then
    begin
      j := 1;
      Result := Result + Char(y);
    end;
  y := Len div $FFFF;
  Len := Len - (y * $FFFF);
  if ((y > 0) or (j = 1)) then
    begin
      j := 1;
      Result := Result + Char(y);
    end;
  y := Len div $FF;
  Len := Len - (y * $FF);
  if ((y > 0) or (j = 1)) then
    Result := Result + Char(y);
  Result := Result + Char(Len);
end;

function ASNObject(Data: string; ASNType: integer): string;
begin
  Result := Char(ASNType) + ASNEncLen(Length(Data)) + Data;
end;

function ASNItem(var Start: integer; Buffer: string): string;
var
  ASNType: integer;
  ASNSize: integer;
  y, n: integer;
  s: string;
  c: char;
begin
  ASNType := Ord(Buffer[Start]);
  Inc(start);
  ASNSize := ASNDecLen(Start, Buffer);
  Result := '';
  if ((ASNType and $20) > 0) then
    begin
      Result := '$' + IntToHex(ASNType, 2);
    end
  else
    case ASNType of
      ASN1_INT, ASN1_COUNTER, ASN1_GAUGE, ASN1_TIMETICKS:
        begin
          y := 0;
          for n:=1 to ASNSize do
            begin
              y := y * 256 + Ord(Buffer[Start]);
              Inc(Start);
            end;
          Result := IntToStr(y);
        end;
      ASN1_OCTSTR, $44:
        begin
          for n:=1 to ASNSize do
            begin
              c := Char(Buffer[Start]);
              Inc(Start);
              s := s + c;
            end;
          Result := s;
        end;
      ASN1_OBJID:
        begin
          for n:=1 to ASNSize do
            begin
              c := Char(Buffer[Start]);
              Inc(Start);
              s := s + c;
            end;
          Result := IdToMib(s);
        end;
      ASN1_IPADDR:
        begin
          s:='';
          for n:=1 to ASNSize do
            begin
              if (n<>1) then
                s := s + '.';
              y := Ord(Buffer[Start]);
              Inc(Start);
              s := s + IntToStr(y);
            end;
          Result := s;
        end;
      else  // NULL
        begin
          Result := '';
          Inc(Start);
          Start := Start + ASNSize;
        end;
    end;
end;

function MibToId(mib:string):string;
var
  x:integer;

  Function walkInt(var s:string):integer;
  var
    x:integer;
    t:string;
  begin
    x:=pos('.',s);
    if x<1 then
      begin
        t:=s;
        s:='';
      end
      else
      begin
        t:=copy(s,1,x-1);
        s:=copy(s,x+1,length(s)-x);
      end;
    result:=StrToIntDef(t,0);
  end;
begin
  result:='';
  x:=walkint(mib);
  x:=x*40+walkint(mib);
  result:=char(x);
  while mib<>'' do
    begin
      x:=walkint(mib);
      result:=result+char(x);
    end;
end;

Function IdToMib(id:string):string;
var
  x,y,n:integer;
begin
  result:='';
  For n:=1 to length(id) do
    begin
      x:=ord(id[n]);
      if n=1 then
        begin
          y:=x div 40;
          x:=x mod 40;
          result:=IntTostr(y);
        end;
      result:=result+'.'+IntToStr(x);
    end;
end;

Function IntMibToStr(int:string):string;
Var
  n,y:integer;
begin
  y:=0;
  for n:=1 to length(int)-1 do
    y:=y*256+ord(int[n]);
  result:=IntToStr(y);
end;

//Hernan Sanchez
function IPToID(Host: string): string;
var
  s, t: string;
  i, x: integer;
begin
  Result := '';
  for x:= 1 to 3 do
    begin
      t := '';
      s := StrScan(PChar(Host), '.');
      t := Copy(Host, 1, (Length(Host) - Length(s)));
      Delete(Host, 1, (Length(Host) - Length(s) + 1));
      i := StrTointDef(t, 0);
      Result := Result + Chr(i);
    end;
  i := StrTointDef(Host, 0);
  Result := Result + Chr(i);
end;

end.
