unit IdSMTP;

interface

uses
  Classes,
  IdAssignedNumbers,
  IdEMailAddress,
  IdGlobal,
  IdHeaderList,
  IdMessage, IdMessageClient;

type
  TAuthenticationType = (atNone, atLogin);

  TIdSMTP = class(TIdMessageClient)
  protected
    {This is just an internal flag we use to determine if we already
     authenticated to the server }
    FDidAuthenticate : Boolean;
    FAuthenticationType: TAuthenticationType;
    FAuthSchemesSupported: TStringList;
    FMailAgent: string;
    {HELO Login}
    FHeloName : String;
    //
    procedure GetAuthTypes;
    function IsAuthProtocolAvailable ( Auth : TAuthenticationType )
      : Boolean; virtual;
  public
    procedure Assign(Source: TPersistent); override;
    function Authenticate : Boolean; virtual;
    procedure Connect(const ATimeout: Integer = IdTimeoutDefault); override;
    constructor Create ( AOwner : TComponent ); override;
    destructor Destroy; override;
    procedure Disconnect; override;
    class procedure QuickSend ( const AHost, ASubject, ATo,
      AFrom, AText : String);
    procedure Send (AMsg: TIdMessage); virtual;
    procedure Expand( AUserName : String; AResults : TStrings); virtual;
    function Verify( AUserName : String) : String; virtual;
    //
    property AuthSchemesSupported: TStringList read FAuthSchemesSupported;
  published
    property AuthenticationType : TAuthenticationType read FAuthenticationType
      write FAuthenticationType;
    property MailAgent: string read FMailAgent write FMailAgent;
    property HeloName : string read FHeloName write FHeloName;
    property Password;
    property Username;
  end;

implementation

uses
  IdCoderMIME,
  IdResourceStrings,
  SysUtils;

{ TIdSMTP }

procedure TIdSMTP.Assign(Source: TPersistent);
begin
  if Source is TIdSMTP then begin
    AuthenticationType := TIdSMTP(Source).AuthenticationType;
    Host := TIdSMTP(Source).Host;
    MailAgent := TIdSMTP(Source).MailAgent;
    Password := TIdSMTP(Source).Password;
    Port := TIdSMTP(Source).Port;
    Username := TIdSMTP(Source).Username;
  end else begin
    inherited;
  end;
end;

function TIdSMTP.Authenticate : Boolean;

  function AuthLogin : Boolean;
  begin
    {for some odd reason wcSMTP does not accept lowercase 'LOGIN" (WcSMTP is
     part of the WildCat Interactive Net Server}
    SendCmd('auth LOGIN', 334);
    SendCmd(TIdEncoderMIME.EncodeString(Username), 334);
    SendCmd(TIdEncoderMIME.EncodeString(Password), 235);
    Result := True;
  end;

begin
  Result := False;  //assume failure
  case FAUthenticationType of
   atLogin : Result := AuthLogin;
  end;
  FDidAuthenticate := True;
end;

procedure TIdSMTP.Connect(const ATimeout: Integer = IdTimeoutDefault);
var
  NameToSend : String;
begin
  inherited;
  try
    GetResponse([220]);
    FAuthSchemesSupported.Clear;
    if Length(FHeloName) > 0 then
      NameToSend := FHeloName
    else
      NameToSend := LocalName;
    if SendCmd( 'ehlo ' + NameToSend ) = 250 then
    begin
      GetAuthTypes;
    end
    else
    begin
        SendCmd( 'Helo ' + NameToSend, 250 );
    end;
  except
    Disconnect;
    Raise;
  end;
end;

constructor TIdSMTP.Create(AOwner: TComponent);
begin
  inherited;
  FAuthSchemesSupported := TStringList.Create;
  FAuthSchemesSupported.Duplicates := dupIgnore; //prevent duplicates in the supported AUTH protocol list
  Port := IdPORT_SMTP;
end;

destructor TIdSMTP.Destroy;
begin
  FreeAndNil ( FAuthSchemesSupported );
  inherited;
end;

procedure TIdSMTP.Disconnect;
begin
  try
    if Connected then
    begin
      WriteLn ( 'Quit' );
    end;
  finally
    inherited;
    FDidAuthenticate := False;
  end;
end;



procedure TIdSMTP.Expand(AUserName: String; AResults: TStrings);
var i : Integer;
    sLine : String;
begin
  SendCMD('EXPN '+AUserName,[250,251]);
  for i := 0 to CmdResultDetails.Count - 1 do
  begin
    sLine := CmdResultDetails[i];
    if (length(sLine) > 3 ) and
      (sLine[4] = '-') then begin
        sLine := StringReplace(sLine,'-',' ',[]);
    end;
    Fetch(SLine,'-');
    AResults.Add(sLine);
  end;
end;

procedure TIdSMTP.GetAuthTypes;
var
  Iterator : Integer;
  Buffer : String;
  ListEntry : String;
begin
  Iterator := 1;
  while Iterator < FCmdResultDetails.Count do
  begin
    Buffer := UpperCase ( FCmdResultDetails [ Iterator ] );
    if ( IndyPos( 'AUTH', Buffer ) = 5) and ( ( Copy ( Buffer, 9, 1 ) = ' ' ) or
      ( Copy( Buffer, 9, 1 ) = '=' ) ) then
    begin
      Buffer := Copy ( Buffer, 10, Length ( Buffer ) );
      while Buffer <> '' do
      begin
        StringReplace ( Buffer, '=', ' ', [ rfReplaceAll ] );
        ListEntry := Fetch( Buffer, ' ' );
        if (FAuthSchemesSupported.IndexOf ( ListEntry )=-1) then
          FAuthSchemesSupported.Add ( ListEntry );
      end;
    end;
    Inc ( Iterator );
  end;
end;

function TIdSMTP.IsAuthProtocolAvailable(
  Auth : TAuthenticationType ) : Boolean;
begin
  case Auth of
    atLogin : Result := ( FAuthSchemesSupported.IndexOf ( 'LOGIN' ) <> -1 );
  else
    Result := False;
  end;
end;

class procedure TIdSMTP.QuickSend (const AHost, ASubject, ATo, AFrom, AText : String);
var
  LSMTP: TIdSMTP;
  LMsg: TIdMessage;
begin
  LSMTP := TIdSMTP.Create(nil);
  try
    LMsg := TIdMessage.Create(LSMTP);
    try
      with LMsg do
      begin
        Subject := ASubject;
        Recipients.EMailAddresses := ATo;
        From.Text := AFrom;
        Body.Text := AText;
      end;
      with LSMTP do
      begin
        Host := AHost;
        Connect; try;
          Send(LMsg);
        finally Disconnect; end;
      end;
    finally
      FreeAndNil(LMsg);
    end;
  finally
    FreeAndNil(LSMTP);
  end;
end;

procedure TIdSMTP.Send(AMsg: TIdMessage);

  procedure WriteRecipient(const AEmailAddress: TIdEmailAddressItem);
  begin
    SendCmd('RCPT to:<' + AEMailAddress.Address + '>', [250, 251]);
  end;

  procedure WriteRecipients(AList: TIdEmailAddressList);
  var
    i: integer;
  begin
    for i := 0 to AList.Count - 1 do begin
      WriteRecipient(AList[i]);
    end;
  end;

  function NeedToAuthenticate: Boolean;
  begin
    if FAuthenticationType <> atNone then begin
      Result := IsAuthProtocolAvailable(FAuthenticationType) and (FDidAuthenticate = False);
    end else begin
      Result := False;
    end;
  end;

begin
  SendCmd('Rset');
  if NeedToAuthenticate then begin
    Authenticate;
  end;
  SendCmd('Mail from:<' + AMsg.From.Address + '>', 250);
  WriteRecipients(AMsg.Recipients);
  WriteRecipients(AMsg.CCList);
  WriteRecipients(AMsg.BccList);
  SendCmd('Data', 354);
  AMsg.ExtraHeaders.Values['X-Mailer'] := MailAgent;
  SendMsg(AMsg);
  SendCmd('.', 250);
end;

function TIdSMTP.Verify(AUserName: String): String;
begin
  Result := '';
  if (SendCMD('VRFY '+AUserName,[250,251]) in [250,251]) then begin
    Result := CmdResult;
  end;
  Fetch(Result);
end;

end.