unit IdPOP3;

{*

  POP 3 (Post Office Protocol Version 3)

  2001-AUG-31 DSiders
    Changed TIdPOP3.Connect to use ATimeout when calling
    inherited Connect.

  2000-SEPT-28 SG
    Added GetUIDL as from code by

  2000-MAY-10 HH
    Added RetrieveMailBoxSize and renamed RetrieveSize to RetrieveMsgSize.
    Finished Connect.

  2000-MARCH-03 HH
    Converted to Indy

*}

interface

uses
  Classes,
  IdAssignedNumbers,
  IdGlobal,
  IdMessage,
  IdMessageClient;

type
  TIdPOP3 = class(TIdMessageClient)
  protected
  public
    function CheckMessages: longint;
    procedure Connect(const ATimeout: Integer = IdTimeoutDefault); override;
    constructor Create(AOwner: TComponent); override;
    function Delete(const MsgNum: Integer): Boolean;
    procedure Disconnect; override;
    function GetResponse(const AAllowedResponses: array of SmallInt): SmallInt;
      override;
    procedure KeepAlive;
    function Reset: Boolean;
    function Retrieve(const MsgNum: Integer; AMsg: TIdMessage): Boolean;
    function RetrieveHeader(const MsgNum: Integer; AMsg: TIdMessage): Boolean;
    function RetrieveMsgSize(const MsgNum: Integer): Integer;
    function RetrieveMailBoxSize: integer;
    function RetrieveRaw(const MsgNum: Integer; const Dest: TStrings): boolean;
  published
    property Password;
    property Username;
    property Port default IdPORT_POP3;
  end;

const
  wsOk = 1;
  wsErr = 0;

implementation

uses
  IdException,
  IdTCPConnection,
  IdResourceStrings,
  SysUtils;

{ TIdPOP3 }

function TIdPOP3.CheckMessages: longint;
var
  Value1, Value2: string;
begin
  Result := 0;
  SendCmd('STAT', wsOk);
  // Only gets here if exception is not raised
  Value1 := CmdResult;
  if Value1 <> '' then
  begin
    Value2 := Copy(Value1, 5, Length(Value1) - 5);
    Result := StrToInt(Copy(Value2, 1, IndyPos(' ', Value2) - 1));
  end;
end;

procedure TIdPOP3.Connect(const ATimeout: Integer = IdTimeoutDefault);
begin
  inherited Connect(ATimeout); // ds 2001-AUG-31
  try
    GetResponse([wsOk]);
    SendCmd('USER ' + Username, wsOk);
    SendCmd('PASS ' + Password, wsOk);
  except
    Disconnect;
    raise;
  end;
end;

constructor TIdPOP3.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Port := IdPORT_POP3;
end;

function TIdPOP3.Delete(const MsgNum: Integer): Boolean;
begin
  SendCmd('DELE ' + IntToStr(MsgNum), wsOk);
  Result := ResultNo = wsOk;
end;

procedure TIdPOP3.Disconnect;
begin
  try
    WriteLn('Quit');
  finally
    inherited;
  end;
end;

procedure TIdPOP3.KeepAlive;
begin
  SendCmd('NOOP', wsOk);
end;

function TIdPOP3.Reset: Boolean;
begin
  SendCmd('RSET', wsOK);
  Result := ResultNo = wsOK;
end;


function TIdPOP3.RetrieveRaw(const MsgNum: Integer; const Dest: TStrings):
  boolean;
begin
  result := SendCmd('RETR ' + IntToStr(MsgNum)) = wsOk;
  if result then
  begin
    Capture(Dest);
    result := true;
  end;
end;

function TIdPOP3.Retrieve(const MsgNum: Integer; AMsg: TIdMessage): Boolean;
begin
  if SendCmd('RETR ' + IntToStr(MsgNum)) = wsOk then
  begin
    if ReceiveHeader(AMsg) then
      // Only retreive the body if we do not already have a full RFC
      ReceiveBody(AMsg);
  end;
  // Will only hit here if ok and NO exception, or IF is not executed
  Result := ResultNo = wsOk;
end;

function TIdPOP3.RetrieveHeader(const MsgNum: Integer; AMsg: TIdMessage):
  Boolean;
var
  Dummy: string;
begin
  try
    SendCmd('TOP ' + IntToStr(MsgNum) + ' 0', wsOk);
    // Only gets here if no exception is raised
    Result := not ReceiveHeader(AMsg, '');

    // Only read the trailing lines if there is anything beyond the header
    if not result then
    begin
      Dummy := ReadLn;
      while Length(Dummy) = 0 do
      begin
        Dummy := ReadLn;
      end;
      Result := Dummy = '.';
    end;
  except
    Result := False;
  end;
end;

function TIdPOP3.RetrieveMailBoxSize: integer;
var
  CurrentLine: string;
begin
  // Returns the size of the mailbox. Issues a LIST command and then
  // sums up each message size. The message sizes are returned in the format
  // 1 1400 2 405 3 100 etc....
  // With this routine, we prevent the user having to call REtrieveSize for
  // each message to get the mailbox size
  Result := 0;
  try
    SendCmd('LIST', wsOk);
    CurrentLine := ReadLn;
    while (CurrentLine <> '.') and (CurrentLine <> '') do
    begin
      CurrentLine := Copy(CurrentLine, IndyPos(' ', CurrentLine) + 1,
        Length(CurrentLine) - IndyPos(' ', CurrentLine) + 1);
      Result := Result + StrToIntDef(CurrentLine, 0);
      CurrentLine := ReadLn;
    end;
  except
    Result := -1;
  end;
end;

function TIdPOP3.RetrieveMsgSize(const MsgNum: Integer): Integer;
var
  ReturnResult: string;
begin
  try
    // Returns the size of the message. if an error ocurrs, returns -1.
    SendCmd('LIST ' + IntToStr(MsgNum), wsOk);
    if CmdResult <> '' then
    begin
      ReturnResult := Copy(CmdResult, 5, Length(CmdResult) - 4);
      Result := StrToIntDef(Copy(ReturnResult, IndyPos(' ', ReturnResult) + 1,
        Length(ReturnResult) - IndyPos(' ', ReturnResult) + 1), -1);
    end
    else
      Result := -1;
  except
    Result := -1;
  end;
end;

function TIdPOP3.GetResponse(const AAllowedResponses: array of SmallInt):
  SmallInt;
begin
  GetInternalResponse;
  if AnsiSameText(Copy(CmdResult, 1, 3), '+OK') then
  begin
    FResultNo := wsOK;
  end
  else if AnsiSameText(Copy(CmdResult, 1, 4), '-ERR') then
  begin
    FResultNo := wsErr;
  end
  else
  begin
    raise EIdException.Create(RSUnrecognizedPOP3ResponseHeader);
  end;
  Result := CheckResponse(ResultNo, AAllowedResponses);
end;

end.
