unit IdNNTPServer;

interface

{
2001-Jul-31 Jim Gunkel
  Reorganized for command handlers
2001-Jun-28 Pete Mee
 - Begun transformation to TIdCommandHandler
2000-Apr=22 Mark L. Holmes
  Ported to Indy
2000-Mar-27
  Final Version
2000-Jan-13 MTL
  Moved to new Palette Scheme (Winshoes Servers)
Original Author: Ozz Nixon
}

uses
  Classes,
  IdAssignedNumbers,
  IdTCPServer;

// For more information on NNTP visit http://www.faqs.org/rfcs/
//
// RFC 977 - A Proposed Standard for the Stream-Based Transmission of News
// RFC 2980 - Common NNTP Extensions
// RFC 1036 - Standard for Interchange of USENET Messages
// RFC 822 - Standard for the Format of ARPA Internet Text

const
   NNTPCommands: array[1..26] of string =
    ('ARTICLE',
     'AUTHINFO',
     'BODY',
     'CHECK',
     'DATE',
     'HEAD',
     'HELP',
     'GROUP',
     'IHAVE',
     'LAST',
     'LIST',
     'LISTGROUP',
     'MODE',
     'NEWGROUPS',
     'NEWNEWS',
     'NEXT',
     'POST',
     'QUIT',
     'SLAVE',
     'STAT',
     'TAKETHIS',
     'XGTITLE',
     'XHDR',
     'XOVER',
     'XPAT',
     'XTHREAD'
    );

type
  TBasicEvent = procedure(ACommand: TIdCommand) of object;
  TNumEvent = procedure(ACommand: TIdCommand; AActualNumber: cardinal) of object;

  TIdNNTPServer = class(TIdTCPServer)
  protected
    fOnCommandArticleID: TBasicEvent;    {article <message-id>|<number>}
    fOnCommandArticleNo: TNumEvent;      {article <message-id>|<number>}
    fOnCommandAuthInfo: TBasicEvent;     {authinfo <user|pass|simple|generic> [data]}
    fOnCommandBodyID: TBasicEvent;       {body <message-id>}
    fOnCommandBodyNo: TNumEvent;         {body <number>}
    fOnCommandHeadID: TBasicEvent;       {head <message-id>}
    fOnCommandHeadNo: TNumEvent;         {head <number>}
    fOnCommandHelp: TBasicEvent;         {help}
    fOnCommandGroup: TBasicEvent;        {group net.news}
    fOnCommandList: TBasicEvent;         {list [command] [optional parm]}
    fOnCommandIHave: TBasicEvent;        {ihave <message-id>}
    fOnCommandLast: TBasicEvent;         {last}
    fOnCommandNewGroups: TBasicEvent;    {newsgroups yymmdd hhmmss [GMT] <distributions>}
    fOnCommandNewNews: TBasicEvent;      {newnews newsgroups yymmdd hhmmss [GMT] <distributions>}
    fOnCommandNext: TBasicEvent;         {next}
    fOnCommandPost: TBasicEvent;         {post}
    fOnCommandQuit: TBasicEvent;         {quit}
    fOnCommandSlave: TBasicEvent;        {slave}
    fOnCommandStatID: TBasicEvent;       {stat <message-id>} {useless!}
    fOnCommandStatNo: TNumEvent;         {stat <number>}
    fOnCommandCheck: TBasicEvent;        {nntp transport ext}
    fOnCommandDate: TBasicEvent;         {date}
    fOnCommandMode: TBasicEvent;         {mode <reader|stream>}
    fOnCommandListgroup: TBasicEvent;    {listgroup net.news}
    fOnCommandTakeThis: TBasicEvent;     {nntp transport ext}
    fOnCommandXOver: TBasicEvent;        {xover start#-stop#}
    fOnCommandXHDR: TBasicEvent;         {xhdr header start#-stop#}
    fOnCommandXGTitle: TBasicEvent;      {legacy support - now LIST NEWSGROUP}
    fOnCommandXPat: TBasicEvent;         {XPAT}
    fOnCommandXThread: TBasicEvent;      {XTHREAD support for Tin}
{other support}
    procedure NotHandled(ACommand: TIdCommand);
    procedure CallBasicEvent(ACommand: TIdCommand;
                             AEvent: TBasicEvent);
    procedure CallNumEvent(ACommand: TIdCommand;
                           ABasicEvent: TBasicEvent;
                           ANumEvent: TNumEvent);
    procedure DoNNTPCommand(ACommand: TIdCommand);
  public
    constructor Create(AOwner: TComponent); override;
  published
    property DefaultPort default IdPORT_NNTP;
    property OnCommandArticleID: TBasicEvent read fOnCommandArticleID write fOnCommandArticleID;
    property OnCommandArticleNo: TNumEvent read fOnCommandArticleNo write fOnCommandArticleNo;
    property OnCommandAuthInfo: TBasicEvent read fOnCommandAuthInfo write fOnCommandAuthInfo;
    property OnCommandBodyID: TBasicEvent read fOnCommandBodyID write fOnCommandBodyID;
    property OnCommandBodyNo: TNumEvent read fOnCommandBodyNo write fOnCommandBodyNo;
    property OnCommandCheck: TBasicEvent read fOnCommandCheck write fOnCommandCheck;
    property OnCommandDate: TBasicEvent read fOnCommandDate write fOnCommandDate;
    property OnCommandGroup: TBasicEvent read fOnCommandGroup write fOnCommandGroup;
    property OnCommandHeadID: TBasicEvent read fOnCommandHeadID write fOnCommandHeadID;
    property OnCommandHeadNo: TNumEvent read fOnCommandHeadNo write fOnCommandHeadNo;
    property OnCommandHelp: TBasicEvent read fOnCommandHelp write fOnCommandHelp;
    property OnCommandIHave: TBasicEvent read fOnCommandIHave write fOnCommandIHave;
    property OnCommandLast: TBasicEvent read fOnCommandLast write fOnCommandLast;
    property OnCommandList: TBasicEvent read fOnCommandList write fOnCommandList;
    property OnCommandListgroup: TBasicEvent read fOnCommandListGroup write fOnCommandListGroup;
    property OnCommandMode: TBasicEvent read fOnCommandMode write fOnCommandMode;
    property OnCommandNewGroups: TBasicEvent read fOnCommandNewGroups write fOnCommandNewGroups;
    property OnCommandNewNews: TBasicEvent read fOnCommandNewNews write fOnCommandNewNews;
    property OnCommandNext: TBasicEvent read fOnCommandNext write fOnCommandNext;
    property OnCommandPost: TBasicEvent read fOnCommandPost write fOnCommandPost;
    property OnCommandQuit: TBasicEvent read fOnCommandQuit write fOnCommandQuit;
    property OnCommandSlave: TBasicEvent read fOnCommandSlave write fOnCommandSlave;
    property OnCommandStatID: TBasicEvent read fOnCommandStatID write fOnCommandStatID;
    property OnCommandStatNo: TNumEvent read fOnCommandStatNo write fOnCommandStatNo;
    property OnCommandTakeThis : TBasicEvent read fOnCommandTakeThis write fOnCommandTakeThis;
    property OnCommandXHDR: TBasicEvent read fOnCommandXHDR write fOnCommandXHDR;
    property OnCommandXGTitle: TBasicEvent read fOnCommandXGTitle write fOnCommandXGTitle;
    property OnCommandXOver: TBasicEvent read fOnCommandXOver write fOnCommandXOver;
    property OnCommandXPat: TBasicEvent read fOnCommandXPat write fOnCommandXPat;
    property OnCommandXThread: TBasicEvent read fOnCommandXThread write fOnCommandXThread;
  end;

implementation

uses
  IdGlobal,
  IdResourceStrings,
  SysUtils;

constructor TIdNNTPServer.Create(AOwner: TComponent);
var
  I1: integer;
begin
  inherited Create(AOwner);
  DefaultPort := IdPORT_NNTP;
  ReplyUnknownCommand.NumericCode := 500;
  ReplyUnknownCommand.Text.Text := RSNNTPServerNotRecognized;

  if not (csDesigning in ComponentState) then begin
    for I1 := Low(NNTPCommands) to High(NNTPCommands) do begin
      with CommandHandlers.Add do begin
        Command := NNTPCommands[I1];
        OnCommand := DoNNTPCommand;
        Tag := I1;
      end;
    end;
  end;
end;

procedure TIdNNTPServer.NotHandled(ACommand: TIdCommand);
begin
  DoOnNoCommandHandler(ACommand.RawLine,ACommand.Thread);
end;

procedure TIdNNTPServer.CallBasicEvent(ACommand: TIdCommand;
                                       AEvent: TBasicEvent);
begin
  if Assigned(AEvent) then begin
    AEvent(ACommand);
  end else begin
    NotHandled(ACommand);
  end;
end;

procedure TIdNNTPServer.CallNumEvent(ACommand: TIdCommand;
                                     ABasicEvent: TBasicEvent;
                                     ANumEvent: TNumEvent);
begin
  if (ACommand.Params.Count > 0) and IsNumeric(ACommand.Params[0]) then begin
    if Assigned(ANumEvent) then begin
      ANumEvent(ACommand,StrToInt(ACommand.Params[0]));
    end else begin
      NotHandled(ACommand);
    end;
  end else begin
    CallBasicEvent(ACommand,ABasicEvent);
  end;
end;

procedure TIdNNTPServer.DoNNTPCommand(ACommand: TIdCommand);
begin
  Case ACommand.CommandHandler.Tag of
    1: CallNumEvent(ACommand,OnCommandArticleID,OnCommandArticleNo);
    2: CallBasicEvent(ACommand,OnCommandAuthInfo);
    3: CallNumEvent(ACommand,OnCommandBodyID,OnCommandBodyNo);
    4: CallBasicEvent(ACommand,OnCommandCheck);
    5: CallBasicEvent(ACommand,OnCommandDate);
    6: CallNumEvent(ACommand,OnCommandHeadID,OnCommandHeadNo);
    7: CallBasicEvent(ACommand,OnCommandHelp);
    8: CallBasicEvent(ACommand,OnCommandGroup);
    9: CallBasicEvent(ACommand,OnCommandIHave);
    10: CallBasicEvent(ACommand,OnCommandLast);
    11: CallBasicEvent(ACommand,OnCommandList);
    12: CallBasicEvent(ACommand,OnCommandListGroup);
    13: CallBasicEvent(ACommand,OnCommandMode);
    14: CallBasicEvent(ACommand,OnCommandNewGroups);
    15: CallBasicEvent(ACommand,OnCommandNewNews);
    16: CallBasicEvent(ACommand,OnCommandNext);
    17: CallBasicEvent(ACommand,OnCommandPost);
    18: begin // QUIT
      if Assigned(OnCommandQuit) then begin
        OnCommandQuit(ACommand);
      end else begin
        ACommand.Thread.Connection.WriteLn('205 ' + RSNNTPServerGoodBye);
      end;
      ACommand.Thread.Connection.Disconnect;
    end;
    19: CallBasicEvent(ACommand,OnCommandSlave);
    20: CallNumEvent(ACommand,OnCommandStatID,OnCommandStatNo);
    21: CallBasicEvent(ACommand,OnCommandTakeThis);
    22: CallBasicEvent(ACommand,OnCommandXGTitle);
    23: CallBasicEvent(ACommand,OnCommandXHdr);
    24: CallBasicEvent(ACommand,OnCommandXOver);
    25: CallBasicEvent(ACommand,OnCommandXPat);
    26: CallBasicEvent(ACommand,OnCommandXThread);
    else NotHandled(ACommand);
  end;
end;

end.
