unit IdLogBase;

interface

uses
  Classes,
  IdIntercept,
  IdSocketHandle;

type
  TIdLogBase = class(TIdConnectionIntercept)
  protected
    FActive: Boolean;
    FLogTime: Boolean;
    FReplaceCRLF: Boolean;
    //
    procedure Close; virtual;
    procedure LogStatus(const AText: string); virtual; abstract;
    procedure LogReceivedData(const AText: string; const AData: string); virtual; abstract;
    procedure LogSentData(const AText: string; const AData: string); virtual; abstract;
    procedure Open; virtual;
    procedure SetActive(const AValue: Boolean); virtual;
  public
    procedure Connect(AConnection: TComponent); override;
    constructor Create(AOwner: TComponent); override;
    procedure Receive(ABuffer: TStream); override;
    procedure Send(ABuffer: TStream); override;
    destructor Destroy; override;
    procedure Disconnect; override;
  published
    property Active: Boolean read FActive write SetActive default False;
    property LogTime: Boolean read FLogTime write FLogTime default True;
    property ReplaceCRLF: Boolean read FReplaceCRLF write FReplaceCRLF default true;
  end;

implementation

uses
  IdGlobal,
  SysUtils;

{ TIdLogBase }

procedure TIdLogBase.Close;
begin
end;

procedure TIdLogBase.Connect(AConnection: TComponent);
begin
  if FActive then
  begin
    inherited Connect(AConnection);
    LogStatus('Connected.');
  end;
end;

constructor TIdLogBase.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLogTime := True;
  ReplaceCRLF := True;
end;

destructor TIdLogBase.Destroy;
begin
  Active := False;
  inherited Destroy;
end;

procedure TIdLogBase.Disconnect;
begin
  if FActive then
  begin
    LogStatus('Disconnected.');
    inherited Disconnect;
  end;
end;

procedure TIdLogBase.Open;
begin
end;

procedure TIdLogBase.Receive(ABuffer: TStream);
var
  s: string;
  LMsg: string;
begin
  if FActive then
  begin
    inherited Receive(ABuffer);
    with TStringStream.Create('') do try
      CopyFrom(ABuffer, ABuffer.Size);
      LMsg := '';
      if LogTime then begin
        LMsg := DateTimeToStr(Now);
      end;
      s := DataString;
      if FReplaceCRLF then begin
        s := StringReplace(s, EOL, '<EOL>', [rfReplaceAll]);
        s := StringReplace(s, CR, '<CR>', [rfReplaceAll]);
        s := StringReplace(s, LF, '<LF>', [rfReplaceAll]);
      end;
      LogReceivedData(LMsg, s);
    finally Free; end;
  end;
end;

procedure TIdLogBase.Send(ABuffer: TStream);
var
  s: string;
  LMsg: string;
begin
  if FActive then
  begin
    inherited Send(ABuffer);
    with TStringStream.Create('') do try
      CopyFrom(ABuffer, ABuffer.Size);
      LMsg := '';
      if LogTime then begin
        LMsg := DateTimeToStr(Now);
      end;
      s := DataString;
      if FReplaceCRLF then begin
        s := StringReplace(s, EOL, '<EOL>', [rfReplaceAll]);
        s := StringReplace(s, CR, '<CR>', [rfReplaceAll]);
        s := StringReplace(s, LF, '<LF>', [rfReplaceAll]);
      end;
      LogSentData(LMsg, s);
    finally Free; end;
  end;
end;

procedure TIdLogBase.SetActive(const AValue: Boolean);
begin
  if FActive <> AValue then begin
    FActive := AValue;
    if FActive then begin
      Open;
    end else begin
      Close;
    end;
  end;
end;

end.

