unit IdIMAP4;
{*                                                                        

  IMAP 4 (Internet Message Access Protocol - Version 4 Rev 1)
  By Idan Cohen i_cohen@yahoo.com

  2001-FEB-27 IC: First version most of the IMAP features are implemented and
                  the core IdPOP3 features are implemented to allow a seamless
                  switch.
                  The unit is currently oriented to a session connection and not
                  to constant connection, because of that server events that are
                  raised from another user actions are not supported.
  2001-APR-18 IC: Added support for the session's connection state with a
                  special exception for commands preformed in wrong connection
                  states. Exceptions were also added for response errors.
  2001-MAY-05 IC: 

*}

{ TODO -oIC :
Change the mailbox list commands so that they receive TMailBoxTree
structures and so they can store in them the mailbox name and it's attributes. }

interface

uses Classes,
     IdAssignedNumbers,
     IdGlobal,
     IdMailBox,
     IdMessage,
     IdMessageClient,
     IdMessageCollection;

const
   wsOk  = 1;
   wsNo  = 0;
   wsBad = 0;

type
  TIdIMAP4Commands =
  ( cmdCAPABILITY,
    cmdNOOP,
    cmdLOGOUT,
    cmdAUTHENTICATE,
    cmdLOGIN,
    cmdSELECT,
    cmdEXAMINE,
    cmdCREATE,
    cmdDELETE,
    cmdRENAME,
    cmdSUBSCRIBE,
    cmdUNSUBSCRIBE,
    cmdLIST,
    cmdLSUB,
    cmdSTATUS,
    cmdAPPEND,
    cmdCHECK,
    cmdCLOSE,
    cmdEXPUNGE,
    cmdSEARCH,
    cmdFETCH,
    cmdSTORE,
    cmdCOPY,
    cmdUID,
    cmdXCmd );

  TIdIMAP4ConnectionState = ( csAny, csNonAuthenticated, csAuthenticated, csSelected );
  {****************************************************************************
  Universal commands CAPABILITY, NOOP, and LOGOUT
  Authenticated state commands SELECT, EXAMINE, CREATE, DELETE, RENAME,
  SUBSCRIBE, UNSUBSCRIBE, LIST, LSUB, STATUS, and APPEND
  Selected state commands CHECK, CLOSE, EXPUNGE, SEARCH, FETCH, STORE, COPY, and UID
  *****************************************************************************}

  TIdIMAP4SearchKey =
  ( skAll,       //All messages in the mailbox; the default initial key for ANDing.
    skAnswered,  //Messages with the \Answered flag set.
    skBcc,       //Messages that contain the specified string in the envelope structure's BCC field.
    skBefore,    //Messages whose internal date is earlier than the specified date.
    skBody,      //Messages that contain the specified string in the body of the message.
    skCc,        //Messages that contain the specified string in the envelope structure's CC field.
    skDeleted,   //Messages with the \Deleted flag set.
    skDraft,     //Messages with the \Draft flag set.
    skFlagged,   //Messages with the \Flagged flag set.
    skFrom,      //Messages that contain the specified string in the envelope structure's FROM field.
    skHeader,    //Messages that have a header with the specified field-name (as defined in [RFC-822])
                 //and that contains the specified string in the [RFC-822] field-body.
    skKeyword,   //Messages with the specified keyword set.
    skLarger,    //Messages with an [RFC-822] size larger than the specified number of octets.
    skNew,       //Messages that have the \Recent flag set but not the \Seen flag.
                 //This is functionally equivalent to "(RECENT UNSEEN)".
    skNot,       //Messages that do not match the specified search key.
    skOld,       //Messages that do not have the \Recent flag set. This is functionally
                 //equivalent to "NOT RECENT" (as opposed to "NOT NEW").
    skOn,        //Messages whose internal date is within the specified date.
    skOr,        //Messages that match either search key.
    skRecent,    //Messages that have the \Recent flag set.
    skSeen,      //Messages that have the \Seen flag set.
    skSentBefore,//Messages whose [RFC-822] Date: header is earlier than the specified date.
    skSentOn,    //Messages whose [RFC-822] Date: header is within the specified date.
    skSentSince, //Messages whose [RFC-822] Date: header is within or later than the specified date.
    skSince,     //Messages whose internal date is within or later than the specified date.
    skSmaller,   //Messages with an [RFC-822] size smaller than the specified number of octets.
    skSubject,   //Messages that contain the specified string in the envelope structure's SUBJECT field.
    skText,      //Messages that contain the specified string in the header or body of the message.
    skTo,        //Messages that contain the specified string in the envelope structure's TO field.
    skUID,       //Messages with unique identifiers corresponding to the specified unique identifier set.
    skUnanswered,//Messages that do not have the \Answered flag set.
    skUndeleted, //Messages that do not have the \Deleted flag set.
    skUndraft,   //Messages that do not have the \Draft flag set.
    skUnflagged, //Messages that do not have the \Flagged flag set.
    skUnKeyWord, //Messages that do not have the specified keyword set.
    skUnseen );

  TIdIMAP4SearchKeyArray = array of TIdIMAP4SearchKey;

  TIdIMAP4SearchRec = record
    Date: TDateTime;
    Size: Integer;
    Text: String;
    SearchKey : TIdIMAP4SearchKey;
  end;

  TIdIMAP4SearchRecArray = array of TIdIMAP4SearchRec;

  TIdIMAP4StatusDataItem = ( mdMessages, mdRecent, mdUIDNext, mdUIDValidity, mdUnseen );

  TIdIMAP4StoreDataItem = ( sdReplace, sdReplaceSilent, sdAdd, sdAddSilent, sdRemove, sdRemoveSilent );

  TIdRetrieveOnSelect = ( rsDisabled, rsHeaders, rsMessages );

  TIdIMAP4 = class(TIdMessageClient)
  private
    procedure SetMailBox(const Value: TIdMailBox);
  protected
    FCmdCounter : Integer;
    FConnectionState : TIdIMAP4ConnectionState;
    FMailBox : TIdMailBox;
    FRetrieveOnSelect: TIdRetrieveOnSelect;
    function GetCmdCounter: String;
    function GetConnectionStateName: String;
    function GetNewCmdCounter: String;
    property LastCmdCounter: String read GetCmdCounter;
    property NewCmdCounter: String read GetNewCmdCounter;
  { Parser Functions }
    function ArrayToNumberString ( const AMsgNumList : array of Integer ) : String;
    procedure ParseExpungeResultDetails ( AMB : TIdMailBox; CmdResultDetails : TStrings );
    procedure ParseFlagString ( AFlagsList : String; var AFlags : TIdMessageFlagsSet );
    procedure ParseListResultDetails ( AMBList : TStringList; CmdResultDetails : TStrings );
    procedure ParseSelectResultDetails ( ATag : String; AMB : TIdMailBox; CmdResultDetails : TStrings );
    procedure ParseStatusResultDetails ( AMB : TIdMailBox; CmdResultDetails : TStrings );
    procedure ParseSearchResultDetails ( AMB : TIdMailBox; CmdResultDetails : TStrings );
  { Parser Functions }
  public
  { TIdIMAP4 Commands }
    property ConnectionState: TIdIMAP4ConnectionState read FConnectionState;
    property MailBox: TIdMailBox read FMailBox write SetMailBox;
    //Requests a checkpoint of the currently selected mailbox.
    function CheckMailBox : Boolean;
    //Checks if the message was read or not.
    function CheckMsgSeen ( const AMsgNum : Integer ) : Boolean;
    //Connects and logins to the IMAP4 account.
    procedure Connect(const ATimeout: Integer = IdTimeoutDefault); override;
    //Close's the current selected mailbox in the account.
    function CloseMailBox : Boolean;
    //Copy's a message from the current selected mailbox to the specified mailbox.
    function CopyMsgs ( const AMsgNumList : array of Integer; const AMBName : String ) : Boolean; overload;
    constructor Create(AOwner: TComponent); override;
    //Create's a new mailbox with the specified name in the account.
    function CreateMailBox ( const AMBName : String ) : Boolean;
    //Delete's the specified mailbox from the account.
    function DeleteMailBox ( const AMBName : String ) : Boolean;
    //Marks a message for deletion, it will be deleted when the mailbox will be purged.
    function DeleteMsgs(const AMsgNumList: array of Integer): Boolean; overload;
    destructor Destroy; override;
    //Logouts and disconnects from the IMAP account.
    procedure Disconnect; override;
    //Examine's the specified mailbox and inserts the results to the TIdMailBox provided.
    function ExamineMailBox ( const AMBName : String; AMB : TIdMailBox ) : Boolean; overload;
    //Expunge's (deletes the marked files) the current selected mailbox in the account.
    function ExpungeMailBox : Boolean;
    //Sends a NOOP (No Operation) to keep the account connection with the server alive.
    procedure KeepAlive;
    //Returns a list of all the child mailboxes (one level down) to the mailbox supplied.
    //This should be used when you fear that there are to many mailboxes and the listing of
    //all of them could be time consuming, so this should be used to retrieve specific mailboxes.
    function ListInferiorMailBoxes ( AMailBoxList, AInferiorMailBoxList : TStringList ) : Boolean;
    //Returns a list of all the mailboxes in the user account.
    function ListMailBoxes ( AMailBoxList : TStringList ) : Boolean;
    //Rename's the specified mailbox in the account.
    function RenameMailBox ( const AOldMBName, ANewMBName : String ) : Boolean;
    //Searches the current selected mailbox for messages matching the SearchRec and
    //returnes the results to the mailbox SearchResults array.
    function SearchMailBox ( const ASearchInfo : array of TIdIMAP4SearchRec{Array} ) : Boolean;//array of TIdIMAP4SearchRec ) : Boolean;
    //Select's the current a mailbox in the account.
    function SelectMailBox ( const AMBName : String ) : Boolean; overload;
    //Retrieves the status of the indicated mailbox.
    function StatusMailBox ( const AMBName : String; AMB : TIdMailBox;
             const AStatusDataItems : array of TIdIMAP4StatusDataItem ) : Boolean;
    //Changes (adds or removes) message flags.
    function StoreFlags ( const AMsgNumList : array of Integer;
             const AStoreMethod : TIdIMAP4StoreDataItem;
             const AFlags : TIdMessageFlagsSet ) : Boolean; overload;
    //Retrieves a whole message while marking it read.
    function Retrieve ( const AMsgNum : Integer; AMsg : TIdMessage ) : Boolean;
    //Retrieves all headers of the selected mailbox to the specified TIdMessageCollection.
    function RetrieveAllHeaders ( AMessageList : TIdMessageCollection ) : Boolean;
    //Retrieves all messages of the selected mailbox to the specified TIdMessageCollection.
    function RetrieveAllMsgs ( AMessageList : TIdMessageCollection ) : Boolean;
    //Returnes the message flag values.
    function RetrieveFlags ( const AMsgNum : Integer; AFlags : TIdMessageFlagsSet ) : Boolean;
    //Retrieves only the message header.
    function RetrieveHeader ( const AMsgNum : Integer; AMsg : TIdMessage ) : Boolean;
    //Retrives the current selected mailbox size.
    function RetrieveMailBoxSize: Integer;
    //Returnes the message size.
    function RetrieveMsgSize( const AMsgNum : Integer ) : Integer;
    //Retrieves a whole message while keeping it's Seen flag untucked
    //(preserving the previous value).
    function RetrievePeek ( const AMsgNum : Integer; AMsg : TIdMessage ) : Boolean;
  { TIdIMAP4 Commands }
  { IdTCPConnection Commands }
    function GetResponse(const ATag: String; const
             AAllowedResponses: array of SmallInt): SmallInt; reintroduce; overload;
    function GetResponse(const AAllowedResponses: array of SmallInt):
             SmallInt; reintroduce; overload;
    function GetLineResponse(const ATag: String; const
             AAllowedResponses: array of SmallInt): SmallInt; overload; virtual;
    function SendCmd(const ATag, AOut: string; const AResponse: SmallInt = -1): SmallInt; overload;
    function SendCmd(const ATag, AOut: string; const AResponse: array of SmallInt):
             SmallInt; overload;
  { IdTCPConnection Commands }
  published
    property Password;
    property RetrieveOnSelect: TIdRetrieveOnSelect read FRetrieveOnSelect write FRetrieveOnSelect default rsDisabled;
    property Port default IdPORT_IMAP4;
    property Username;
  end;

implementation

uses
  IdException,
  IdResourceStrings,
  SysUtils;

const
   IMAP4Commands : array [cmdCapability..cmdXCmd] of String =
   ( { Client Commands - Any State}
    'CAPABILITY', {Do not Localize}
    'NOOP', {Do not Localize}
    'LOGOUT', {Do not Localize}
    { Client Commands - Non Authenticated State}
    'AUTHENTICATE', {Do not Localize}
    'LOGIN', {Do not Localize}
    { Client Commands - Authenticated State}
    'SELECT', {Do not Localize}
    'EXAMINE', {Do not Localize}
    'CREATE', {Do not Localize}
    'DELETE', {Do not Localize}
    'RENAME', {Do not Localize}
    'SUBSCRIBE', {Do not Localize}
    'UNSUBSCRIBE', {Do not Localize}
    'LIST', {Do not Localize}
    'LSUB', {Do not Localize}
    'STATUS', {Do not Localize}
    'APPEND', {Do not Localize}
    { Client Commands - Selected State}
    'CHECK', {Do not Localize}
    'CLOSE', {Do not Localize}
    'EXPUNGE', {Do not Localize}
    'SEARCH', {Do not Localize}
    'FETCH', {Do not Localize}
    'STORE', {Do not Localize}
    'COPY', {Do not Localize}
    'UID', {Do not Localize}
    { Client Commands - Experimental/ Expansion}
    'X' ); {Do not Localize}

   IMAP4FetchDataItem : array [0..14] of String =
   ( 'ALL', {Do not Localize}          //Macro equivalent to: (FLAGS INTERNALDATE RFC822.SIZE ENVELOPE)
     'BODY', {Do not Localize}         //Non-extensible form of BODYSTRUCTURE.
     'BODY[]<>', {Do not Localize}
     'BODY.PEEK[]', {Do not Localize}
     'BODYSTRUCTURE', {Do not Localize}//The [MIME-IMB] body structure of the message.  This
                                       //is computed by the server by parsing the [MIME-IMB]
                                       //header fields in the [RFC-822] header and [MIME-IMB] headers.
     'ENVELOPE', {Do not Localize}     //The envelope structure of the message.  This is
                                       //computed by the server by parsing the [RFC-822]
                                       //header into the component parts, defaulting various
                                       //fields as necessary.
     'FAST', {Do not Localize}         //Macro equivalent to: (FLAGS INTERNALDATE RFC822.SIZE)
     'FLAGS', {Do not Localize}        //The flags that are set for this message.
     'FULL', {Do not Localize}         //Macro equivalent to: (FLAGS INTERNALDATE RFC822.SIZE ENVELOPE BODY)
     'INTERNALDATE', {Do not Localize} //The internal date of the message.
     'RFC822', {Do not Localize}       //Functionally equivalent to BODY[], differing in the
                                       //syntax of the resulting untagged FETCH data (RFC822
                                       //is returned).
     'RFC822.HEADER', {Do not Localize}//Functionally equivalent to BODY.PEEK[HEADER],
                                       //differing in the syntax of the resulting untagged
                                       //FETCH data (RFC822.HEADER is returned).
     'RFC822.SIZE', {Do not Localize}  //The [RFC-822] size of the message.
     'RFC822.TEXT', {Do not Localize}  //Functionally equivalent to BODY[TEXT], differing in
                                       //the syntax of the resulting untagged FETCH data
                                       //(RFC822.TEXT is returned).
     'UID' ); {Do not Localize}        //The unique identifier for the message.


   IMAP4SearchKeys : array [skAll..skUnseen] of String =
   ( 'ALL', {Do not Localize}      //All messages in the mailbox; the default initial key for ANDing.
     'ANSWERED', {Do not Localize} //Messages with the \Answered flag set.
     'BCC', {Do not Localize}      //Messages that contain the specified string in the envelope structure's BCC field.
     'BEFORE', {Do not Localize}   //Messages whose internal date is earlier than the specified date.
     'BODY', {Do not Localize}     //Messages that contain the specified string in the body of the message.
     'CC', {Do not Localize}       //Messages that contain the specified string in the envelope structure's CC field.
     'DELETED', {Do not Localize}  //Messages with the \Deleted flag set.
     'DRAFT', {Do not Localize}    //Messages with the \Draft flag set.
     'FLAGGED', {Do not Localize}  //Messages with the \Flagged flag set.
     'FROM', {Do not Localize}     //Messages that contain the specified string in the envelope structure's FROM field.
     'HEADER', {Do not Localize}   //Messages that have a header with the specified field-name (as defined in [RFC-822])
                                   //and that contains the specified string in the [RFC-822] field-body.
     'KEYWORD', {Do not Localize}  //Messages with the specified keyword set.
     'LARGER', {Do not Localize}   //Messages with an [RFC-822] size larger than the specified number of octets.
     'NEW', {Do not Localize}      //Messages that have the \Recent flag set but not the \Seen flag.
                                   //This is functionally equivalent to "(RECENT UNSEEN)".
     'NOT', {Do not Localize}      //Messages that do not match the specified search key.
     'OLD', {Do not Localize}      //Messages that do not have the \Recent flag set. This is functionally
                                   //equivalent to "NOT RECENT" (as opposed to "NOT NEW").
     'ON', {Do not Localize}       //Messages whose internal date is within the specified date.
     'OR', {Do not Localize}       //Messages that match either search key.
     'RECENT', {Do not Localize}   //Messages that have the \Recent flag set.
     'SEEN', {Do not Localize}     //Messages that have the \Seen flag set.
     'SENTBEFORE',{Do not Localize}//Messages whose [RFC-822] Date: header is earlier than the specified date.
     'SENTON', {Do not Localize}   //Messages whose [RFC-822] Date: header is within the specified date.
     'SENTSINCE', {Do not Localize}//Messages whose [RFC-822] Date: header is within or later than the specified date.
     'SINCE', {Do not Localize}    //Messages whose internal date is within or later than the specified date.
     'SMALLER', {Do not Localize}  //Messages with an [RFC-822] size smaller than the specified number of octets.
     'SUBJECT', {Do not Localize}  //Messages that contain the specified string in the envelope structure's SUBJECT field.
     'TEXT', {Do not Localize}     //Messages that contain the specified string in the header or body of the message.
     'TO', {Do not Localize}       //Messages that contain the specified string in the envelope structure's TO field.
     'UID', {Do not Localize}      //Messages with unique identifiers corresponding to the specified unique identifier set.
     'UNANSWERED',{Do not Localize}//Messages that do not have the \Answered flag set.
     'UNDELETED', {Do not Localize}//Messages that do not have the \Deleted flag set.
     'UNDRAFT', {Do not Localize}  //Messages that do not have the \Draft flag set.
     'UNFLAGGED', {Do not Localize}//Messages that do not have the \Flagged flag set.
     'UNKEYWORD', {Do not Localize}//Messages that do not have the specified keyword set.
     'UNSEEN' ); {Do not Localize}

   IMAP4StoreDataItem : array [sdReplace..sdRemoveSilent] of String =
   ( 'FLAGS', {Do not Localize}
     'FLAGS.SILENT', {Do not Localize}
     '+FLAGS', {Do not Localize}
     '+FLAGS.SILENT', {Do not Localize}
     '-FLAGS', {Do not Localize}
     '-FLAGS.SILENT' ); {Do not Localize}

{ TIdIMAP4 }

{ IdTCPConnection Commands... }

//Need to find a solution for the problems with the login OK
function TIdIMAP4.GetResponse(const AAllowedResponses: array of SmallInt): SmallInt;
var sLine : String;
begin
     CmdResultDetails.Clear;
     sLine := ReadLnWait;
     CmdResultDetails.Add ( sLine );
     if ( sLine[1] = '*' ) then {Do not Localize}
     begin
          if AnsiSameText ( Copy ( CmdResult, 3, 2 ), 'OK' ) then {Do not Localize}
             FResultNo := wsOK
          else if AnsiSameText ( Copy ( CmdResult, 3, 2 ), 'NO' ) then {Do not Localize}
          begin
               FResultNo := wsNo;
          end
          else if AnsiSameText ( Copy ( CmdResult, 3, 3 ), 'BAD' ) then {Do not Localize}
               FResultNo := wsBad;
     end
     else
     begin
          raise EIdException.Create(RSUnrecognizedIMAP4ResponseHeader);
     end;

     Result := CheckResponse(ResultNo, AAllowedResponses);
end;

function TIdIMAP4.GetResponse(const ATag: String; const AAllowedResponses: array of SmallInt): SmallInt;
var sLine : String;
begin
  CmdResultDetails.Clear;
  sLine := ReadLnWait;
  CmdResultDetails.Add ( sLine );
  if ( sLine[1] = '*' ) then {Do not Localize}
  begin // Multi line response coming
        //Not sure need check here *********************************************
       repeat
             sLine := ReadLnWait;
             CmdResultDetails.Add ( sLine );
             {We keep reading lines until we encounter either a line such as
             *we still dont know* or the user specified tag}
       until ( AnsiSameText ( Copy ( CmdResult, 1, Length ( ATag ) ), ATag ) );
  end;

  if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 3 ) ), ATag + ' OK' ) then {Do not Localize}
  begin
       FResultNo := wsOK;
  end
  else if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 3 ) ), ATag + ' NO' ) then {Do not Localize}
  begin
       FResultNo := wsNo;
  end
  else if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 4 ) ), ATag + ' BAD' ) then {Do not Localize}
  begin
       FResultNo := wsBad;
  end
  else
  begin
       raise EIdException.Create(RSUnrecognizedIMAP4ResponseHeader);
  end;

  Result := CheckResponse(ResultNo, AAllowedResponses);
end;

function TIdIMAP4.GetLineResponse(const ATag: String;
  const AAllowedResponses: array of SmallInt): SmallInt;
var sLine : String;
begin
  CmdResultDetails.Clear;
  sLine := ReadLnWait;
  CmdResultDetails.Add ( sLine );
  if ( sLine[1] = '*' ) then {Do not Localize}
  begin
       FResultNo := wsOK;
  end
  else
  begin
       if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 3 ) ), ATag + ' OK' ) then {Do not Localize}
       begin
            FResultNo := wsOK;
       end
       else if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 3 ) ), ATag + ' NO' ) then {Do not Localize}
       begin
            FResultNo := wsNo;
       end
       else if AnsiSameText ( Copy ( CmdResult, 1, ( Length ( ATag ) + 4 ) ), ATag + ' BAD' ) then {Do not Localize}
       begin
            FResultNo := wsBad;
       end
       else
       begin
            raise EIdException.Create(RSUnrecognizedIMAP4ResponseHeader);
       end;
  end;
  Result := CheckResponse(ResultNo, AAllowedResponses);
end;

function TIdIMAP4.SendCmd(const ATag, AOut: string; const AResponse: array of SmallInt): SmallInt;
begin
     if ( AOut <> #0 ) then
     begin
          WriteLn ( ATag + ' ' + AOut ); {Do not Localize}
     end;
     Result := GetResponse ( ATag, AResponse );
end;

function TIdIMAP4.SendCmd(const ATag, AOut: string; const AResponse: SmallInt): SmallInt;
begin
     if ( AResponse = -1 ) then
     begin
          result := SendCmd ( ATag, AOut, [] );
     end
     else
     begin
          result := SendCmd ( ATag, AOut, [AResponse] );
     end;
end;

{ ...IdTCPConnection Commands }

procedure TIdIMAP4.Connect(const ATimeout: Integer = IdTimeoutDefault);
begin
     inherited Connect(ATimeout);
     try
        if ( GetResponse ( [wsOk] ) = wsOk ) then
        begin
             FConnectionState := csNonAuthenticated;
             FCmdCounter := 0;
             SendCmd ( NewCmdCounter, IMAP4Commands[cmdLogin] + ' ' + Username + ' ' + Password, wsOk ); {Do not Localize}
             if ( ResultNo = wsOk ) then
                FConnectionState := csAuthenticated;
        end;
     except
           Disconnect;
           raise;
     end;
end;

constructor TIdIMAP4.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     FMailBox := TIdMailBox.Create ( Self );
     Port := IdPORT_IMAP4;
     FCmdCounter := 0;
     FConnectionState := csNonAuthenticated;
     FRetrieveOnSelect := rsDisabled;
end;

procedure TIdIMAP4.Disconnect;
begin
     if Connected then
     begin
          try
             SendCmd ( NewCmdCounter, IMAP4Commands[cmdLogout], wsOk );
          finally
                 inherited;
                 FConnectionState := csNonAuthenticated;
          end;
     end
     else
     begin
         raise EIdClosedSocket.Create ( RSStatusDisconnected );
     end;
end;

procedure TIdIMAP4.KeepAlive;
begin
     SendCmd ( NewCmdCounter, IMAP4Commands[cmdNoop], wsOk );
end;

function TIdIMAP4.GetCmdCounter: String;
begin
     Result := 'C' + IntToStr ( FCmdCounter ); {Do not Localize}
end;

function TIdIMAP4.GetNewCmdCounter: String;
begin
     Inc ( FCmdCounter );
     Result := 'C' + IntToStr ( FCmdCounter ); {Do not Localize}
end;

destructor TIdIMAP4.Destroy;
begin
     FreeAndNil(FMailBox);
     FreeAndNil(FCmdResultDetails);
     inherited;
end;

function TIdIMAP4.SelectMailBox(const AMBName: String): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdSelect] + ' "' + AMBName + '"' ), wsOk); {Do not Localize}
          if ( ResultNo = wsOk ) then
          begin
               //Put the parse in the IMAP Class and send the MB;
               ParseSelectResultDetails ( LastCmdCounter, FMailBox, CmdResultDetails );
               FMailBox.Name := AMBName;
               FConnectionState := csSelected;
               case RetrieveOnSelect of
                 rsHeaders: RetrieveAllHeaders ( FMailBox.MessageList );
                 rsMessages: RetrieveAllMsgs ( FMailBox.MessageList );
               end;
          end;
          Result := ResultNo = wsOk;
     end
     else
     begin
         raise EIdConnectionStateError.CreateFmt (
               RSIMAP4ConnectionStateError, [GetConnectionStateName] );
         Result := False;
     end;
end;

function TIdIMAP4.ExamineMailBox(const AMBName: String;
  AMB: TIdMailBox): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdExamine] + ' "' + AMBName + '"' ), wsOk); {Do not Localize}
          if ( ResultNo = wsOk ) then
          begin
               ParseSelectResultDetails ( LastCmdCounter, AMB, CmdResultDetails );
               AMB.Name := AMBName;
          end;
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.CloseMailBox: Boolean;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, IMAP4Commands[cmdClose], wsOk );
          if ( ResultNo = wsOk ) then
          begin
               //MailBox.Clear;
               FConnectionState := csAuthenticated;
          end;
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.CreateMailBox(const AMBName: String): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdCreate] + ' "' + AMBName + '"' ), wsOk); {Do not Localize}
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.DeleteMailBox(const AMBName: String): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdDelete] + ' "' + AMBName + '"' ), wsOk); {Do not Localize}
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.RenameMailBox(const AOldMBName, ANewMBName: String): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdRename] + ' "' + {Do not Localize}
                                     AOldMBName + '" "' + ANewMBName + '"' ), wsOk); {Do not Localize}
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.StatusMailBox(const AMBName: String;
  AMB: TIdMailBox; const AStatusDataItems: array of TIdIMAP4StatusDataItem): Boolean;
var DataItems : String;
    Fn : Integer;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          for Fn := Low ( AStatusDataItems ) to High ( AStatusDataItems ) do
              case AStatusDataItems[Fn] of
                mdMessages: DataItems := DataItems + 'MESSAGES '; {Do not Localize}
                mdRecent: DataItems := DataItems + 'RECENT '; {Do not Localize}
                mdUIDNext: DataItems := DataItems + 'UIDNEXT '; {Do not Localize}
                mdUIDValidity: DataItems := DataItems + 'UIDVALIDITY '; {Do not Localize}
                mdUnseen: DataItems := DataItems + 'UNSEEN '; {Do not Localize}
              end;
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdStatus] + ' "' + AMBName + '" (' + Trim ( DataItems ) + ')' ), wsOk); {Do not Localize}
          if ( ResultNo = wsOk ) then
             ParseStatusResultDetails ( AMB, CmdResultDetails );
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.CheckMailBox: Boolean;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, IMAP4Commands[cmdCheck], wsOk);
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.ExpungeMailBox: Boolean;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, IMAP4Commands[cmdExpunge], wsOk);
          if ( ResultNo = wsOk ) then
             ParseExpungeResultDetails ( FMailBox, CmdResultDetails );
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.SearchMailBox(
  const ASearchInfo: array of TIdIMAP4SearchRec{Array} ) : Boolean;
var SearchStr : String;
    Fn : Integer;
begin
     ShortDateFormat := 'dd-MMM-yyyy';
     DateSeparator := '-';
     for Fn := Low ( ASearchInfo ) to High ( ASearchInfo ) do
         case ASearchInfo[Fn].SearchKey of
           //skAll: SearchStr := SearchStr + IMAP4SearchKeys[skAll] + ' '; //Need to check
           skAnswered: SearchStr := SearchStr + IMAP4SearchKeys[skAnswered] + ' '; {Do not Localize}
           skBcc: SearchStr := SearchStr + IMAP4SearchKeys[skBcc] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skBefore: SearchStr := SearchStr + IMAP4SearchKeys[skBefore] + ' ' + {Do not Localize}
                                  DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skBody: SearchStr := SearchStr + IMAP4SearchKeys[skBody] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skCc: SearchStr := SearchStr + IMAP4SearchKeys[skCc] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skDeleted: SearchStr := SearchStr + IMAP4SearchKeys[skDeleted] + ' '; {Do not Localize}
           skDraft: SearchStr := SearchStr + IMAP4SearchKeys[skDraft] + ' '; {Do not Localize}
           skFlagged: SearchStr := SearchStr + IMAP4SearchKeys[skFlagged] + ' '; {Do not Localize}
           skFrom: SearchStr := SearchStr + IMAP4SearchKeys[skFrom] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           //skHeader: //Need to check
           //skKeyword: //Need to check
           skLarger: SearchStr := SearchStr + IMAP4SearchKeys[skLarger] + ' ' + {Do not Localize}
                                  IntToStr ( ASearchInfo[Fn].Size ) + ' '; {Do not Localize}
           skNew: SearchStr := SearchStr + IMAP4SearchKeys[skNew] + ' '; {Do not Localize}
           skNot: SearchStr := SearchStr + IMAP4SearchKeys[skNot] + ' '; {Do not Localize}
           skOld: SearchStr := SearchStr + IMAP4SearchKeys[skOld] + ' '; {Do not Localize}
           skOn: SearchStr := SearchStr + IMAP4SearchKeys[skOn] + ' ' + {Do not Localize}
                              DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skOr: SearchStr := SearchStr + IMAP4SearchKeys[skOr] + ' '; {Do not Localize}
           skRecent: SearchStr := SearchStr + IMAP4SearchKeys[skRecent] + ' '; {Do not Localize}
           skSeen: SearchStr := SearchStr + IMAP4SearchKeys[skSeen] + ' '; {Do not Localize}
           skSentBefore: SearchStr := SearchStr + IMAP4SearchKeys[skSentBefore] + ' ' + {Do not Localize}
                                      DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skSentOn: SearchStr := SearchStr + IMAP4SearchKeys[skSentOn] + ' ' + {Do not Localize}
                                  DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skSentSince: SearchStr := SearchStr + IMAP4SearchKeys[skSentSince] + ' ' + {Do not Localize}
                                     DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skSince: SearchStr := SearchStr + IMAP4SearchKeys[skSince] + ' ' + {Do not Localize}
                              DateToStr ( ASearchInfo[Fn].Date ) + ' '; {Do not Localize}
           skSmaller: SearchStr := SearchStr + IMAP4SearchKeys[skSmaller] + ' ' + {Do not Localize}
                                   IntToStr ( ASearchInfo[Fn].Size ) + ' '; {Do not Localize}
           skSubject: SearchStr := SearchStr + IMAP4SearchKeys[skSubject] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skText: SearchStr := SearchStr + IMAP4SearchKeys[skText] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skTo: SearchStr := SearchStr + IMAP4SearchKeys[skTo] + ' "' + ASearchInfo[Fn].Text + '" '; {Do not Localize}
           skUID: SearchStr := SearchStr + IMAP4SearchKeys[skUID] + ' ' + ASearchInfo[Fn].Text + ' '; {Do not Localize}
           skUnanswered: SearchStr := SearchStr + IMAP4SearchKeys[skUnanswered] + ' '; {Do not Localize}
           skUndeleted: SearchStr := SearchStr + IMAP4SearchKeys[skUndeleted] + ' '; {Do not Localize}
           skUndraft: SearchStr := SearchStr + IMAP4SearchKeys[skUndraft] + ' '; {Do not Localize}
           skUnflagged: SearchStr := SearchStr + IMAP4SearchKeys[skUnflagged] + ' '; {Do not Localize}
           skUnKeyWord: SearchStr := SearchStr + IMAP4SearchKeys[skUnKeyWord] + ' '; {Do not Localize}
           skUnseen: SearchStr := SearchStr + IMAP4SearchKeys[skUnseen] + ' '; {Do not Localize}
         end;
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdSearch] + ' ' + Trim ( SearchStr ) ), wsOk ); {Do not Localize}
          if ( ResultNo = wsOk ) then
             ParseSearchResultDetails ( FMailBox, CmdResultDetails );
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.ListMailBoxes(AMailBoxList: TStringList): Boolean;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdList] + ' "" *' ), wsOk ); {Do not Localize}
          if ( ResultNo = wsOk ) then
             ParseListResultDetails ( AMailBoxList, CmdResultDetails );
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.ListInferiorMailBoxes(AMailBoxList, AInferiorMailBoxList: TStringList): Boolean;
var Fn : Integer;
    AuxMailBoxList : TStringList;
begin
     if ( ( FConnectionState = csAuthenticated ) or ( FConnectionState = csSelected ) ) then
     begin
          if ( AMailBoxList = nil ) then
          begin
               SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdList] + ' "" %' ), wsOk ); {Do not Localize}
               if ( ResultNo = wsOk ) then
               begin
                    ParseListResultDetails ( AInferiorMailBoxList, CmdResultDetails );
                    //The INBOX mailbox is added because I think it always has to exist
                    //in an IMAP4 account (default) but it does not list it in this command.
                    AInferiorMailBoxList.Add ( 'INBOX' ); {Do not Localize}
               end;
               Result := ResultNo = wsOk;
          end
          else
          begin
               AuxMailBoxList := TStringList.Create;
               try
                  AInferiorMailBoxList.Clear;
                  for Fn := 0 to ( AMailBoxList.Count - 1 ) do
                  begin
                       SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdList] + ' "" "' + {Do not Localize}
                       AMailBoxList[Fn] + '/%"' ), wsOk ); {Do not Localize}
                       if ( ResultNo = wsOk ) then
                       begin
                            ParseListResultDetails ( AuxMailBoxList, CmdResultDetails );
                            AInferiorMailBoxList.AddStrings ( AuxMailBoxList );
                       end
                       else
                           Break;
                  end;
                  Result := ResultNo = wsOk;
               finally
                      AuxMailBoxList.Free;
               end;
          end;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.StoreFlags(const AMsgNumList: array of Integer;
  const AStoreMethod: TIdIMAP4StoreDataItem;
  const AFlags: TIdMessageFlagsSet): Boolean;
var DataItem,
    MsgSet,
    Flags : String;
begin
     if ( Length ( AMsgNumList ) = 0 ) then
     begin
          Result := False;
          Exit;
     end;
     MsgSet := ArrayToNumberString ( AMsgNumList );
     case AStoreMethod of
       sdReplace: DataItem := IMAP4StoreDataItem[sdReplaceSilent];
       sdAdd: DataItem := IMAP4StoreDataItem[sdAddSilent];
       sdRemove: DataItem := IMAP4StoreDataItem[sdRemoveSilent];
     end;

     if mfAnswered in AFlags then Flags := Flags + MessageFlags[mfAnswered] + ' '; {Do not Localize}
     if mfFlagged in AFlags then Flags := Flags + MessageFlags[mfFlagged] + ' '; {Do not Localize}
     if mfDeleted in AFlags then Flags := Flags + MessageFlags[mfDeleted] + ' '; {Do not Localize}
     if mfDraft in AFlags then Flags := Flags + MessageFlags[mfDraft] + ' '; {Do not Localize}
     if mfSeen in AFlags then Flags := Flags + MessageFlags[mfSeen] + ' '; {Do not Localize}

     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdStore] + ' ' + MsgSet + ' ' + DataItem + ' (' + Trim ( Flags ) + ')' ), wsOk); {Do not Localize}
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.CopyMsgs(const AMsgNumList: array of Integer;
  const AMBName: String): Boolean;
var MsgSet : String;
begin
     if ( Length ( AMsgNumList ) = 0 ) then
     begin
          Result := False;
          Exit;
     end;
     MsgSet := ArrayToNumberString ( AMsgNumList );
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdCopy] + ' ' + MsgSet + ' "' + AMBName + '"' ), wsOk); {Do not Localize}
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );
          Result := False;
     end;
end;

function TIdIMAP4.RetrieveHeader(const AMsgNum: Integer;
  AMsg: TIdMessage): Boolean;
var SlRetrieve : TStringList;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SlRetrieve := TStringList.Create;
          try
             WriteLn ( NewCmdCounter + ' ' + ( {Do not Localize}
             IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + ' (' + IMAP4FetchDataItem[11] + ')' ) ); {Do not Localize}
             if ( GetLineResponse ( GetCmdCounter, [wsOk] ) = wsOk ) then
             begin
                  BreakApart ( CmdResultDetails[0], ' ', SlRetrieve ); {Do not Localize}
                  if ( ( CompareStr ( SlRetrieve[0], '*' ) = 0 ) and {Do not Localize}
                       ( CompareStr ( SlRetrieve[1], IntToStr ( AMsgNum ) ) = 0 ) and
                       ( CompareStr ( SlRetrieve[2], IMAP4Commands[cmdFetch] ) = 0 ) and
                       ( CompareStr ( SlRetrieve[3], '(RFC822.HEADER' ) = 0 ) ) then {Do not Localize}
                  begin
                       ReceiveHeader ( AMsg, ')' ); {Do not Localize}
                       GetResponse ( GetCmdCounter, [wsOk] );
                  end;
             end;
          finally
                 SlRetrieve.Free;
          end;
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] ); 
          Result := False;
     end;
end;

function TIdIMAP4.Retrieve(const AMsgNum: Integer;
  AMsg: TIdMessage): Boolean;
var SlRetrieve : TStringList;
    AuxString: String;
    Fn: Integer;
    LFlags: TIdMessageFlagsSet;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SlRetrieve := TStringList.Create;
          try
             WriteLn (
             NewCmdCounter + ' ' + ( IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + {Do not Localize}
             ' (' + IMAP4FetchDataItem[10] + ')' ) ); {Do not Localize}
             if ( GetLineResponse ( GetCmdCounter, [wsOk] ) = wsOk ) then
             begin
                  BreakApart ( CmdResultDetails[0], ' ', SlRetrieve ); {Do not Localize}
                  if ( ( CompareStr ( SlRetrieve[0], '*' ) = 0 ) and {Do not Localize}
                       ( CompareStr ( SlRetrieve[1], IntToStr ( AMsgNum ) ) = 0 ) and
                       ( CompareStr ( SlRetrieve[2], IMAP4Commands[cmdFetch] ) = 0 ) and
                       ( CompareStr ( SlRetrieve[3], '(RFC822' ) = 0 ) ) then {Do not Localize}
                  begin
                       if ReceiveHeader( AMsg, '' ) then {Do not Localize} // Only retreive the body if we do not already have a full RFC
                       begin
                            ReceiveBody ( AMsg, ')' ); {Do not Localize}
                       end;
                       GetResponse ( GetCmdCounter, [wsOk] );
                       for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
                       begin
                            if ( Pos ( ( '* ' + IntToStr ( AMsgNum ) + ' FETCH (FLAGS ' ), {Do not Localize}
                               CmdResultDetails[Fn] ) > 0 ) then
                            begin
                                 AuxString := Copy ( CmdResultDetails[Fn],
                                 ( Pos ( 'FLAGS (', CmdResultDetails[Fn] ) + Length ( 'FLAGS (' ) ), {Do not Localize}
                                 Length ( CmdResultDetails[Fn] ) );
                                 AuxString := Copy ( AuxString, 1, ( Pos ( '))', AuxString ) - 1 ) ); {Do not Localize}
                                 ParseFlagString ( AuxString, LFlags );
                                 AMsg.Flags := LFlags;
                            end;
                       end;
                  end;
             end;
          finally
                 SlRetrieve.Free;
          end;
          //TODO: Parse the message flags returned after the fetch
          //if ( CmdResultDetails.Count > 1 ) then
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );  
          Result := False;
     end;
end;


function TIdIMAP4.RetrievePeek(const AMsgNum: Integer;
  AMsg: TIdMessage): Boolean;
var SlRetrieve : TStringList;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SlRetrieve := TStringList.Create;
          try
             WriteLn (
             NewCmdCounter + ' ' + ( IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + {Do not Localize}
             ' (' + IMAP4FetchDataItem[3] + ')' ) ); {Do not Localize}
             if ( GetLineResponse ( GetCmdCounter, [wsOk] ) = wsOk ) then
             begin
                  BreakApart ( CmdResultDetails[0], ' ', SlRetrieve ); {Do not Localize}
                  if ( ( CompareStr ( SlRetrieve[0], '*' ) = 0 ) and {Do not Localize}
                       ( CompareStr ( SlRetrieve[1], IntToStr ( AMsgNum ) ) = 0 ) and
                       ( CompareStr ( SlRetrieve[2], IMAP4Commands[cmdFetch] ) = 0 ) and
                       ( CompareStr ( SlRetrieve[3], '(BODY[]' ) = 0 ) ) then {Do not Localize}
                  begin
                       if ReceiveHeader( AMsg, '' ) then {Do not Localize} // Only retreive the body if we do not already have a full RFC
                       begin
                            ReceiveBody ( AMsg, ')' ); {Do not Localize}
                       end;
                       GetResponse ( GetCmdCounter, [wsOk] );
                  end;
             end;
          finally
                 SlRetrieve.Free;
          end;
          //TODO: Parse the message flags returned after the fetch
          //if ( CmdResultDetails.Count > 1 ) then
          Result := ResultNo = wsOk;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );    
          Result := False;
     end;
end;

function TIdIMAP4.RetrieveAllHeaders(
  AMessageList: TIdMessageCollection): Boolean;
var MsgItem : TIdMessageItem;
    Fn : Integer;
begin
     if ( FConnectionState = csSelected ) then
     begin
          if ( AMessageList <> nil ) then
          begin
               Result := True;
               for Fn := 1 to FMailBox.TotalMsgs do
               begin
                    MsgItem := AMessageList.Add;
                    if not RetrieveHeader ( Fn, MsgItem.IdMessage ) then
                    begin
                         Result := False;
                         Break;
                    end;
               end;
          end
          else
          begin
               Result := False;
          end;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );     
          Result := False;
     end;
end;

function TIdIMAP4.RetrieveAllMsgs(
  AMessageList: TIdMessageCollection): Boolean;
var MsgItem : TIdMessageItem;
    Fn : Integer;
begin
     if ( FConnectionState = csSelected ) then
     begin
          if ( AMessageList <> nil ) then
          begin
               Result := True;
               for Fn := 1 to FMailBox.TotalMsgs do
               begin
                    MsgItem := AMessageList.Add;
                    if not Retrieve ( Fn, MsgItem.IdMessage ) then
                    begin
                         Result := False;
                         Break;
                    end;
               end;
          end
          else
          begin
               Result := False;
          end;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );  
          Result := False;
     end;
end;

function TIdIMAP4.DeleteMsgs(const AMsgNumList: array of Integer): Boolean;
begin
     Result := StoreFlags ( AMsgNumList, sdAdd, [mfDeleted] );
end;

function TIdIMAP4.RetrieveMailBoxSize: Integer;
var SlRetrieve : TStringList;
    Fn : Integer;
begin
     if ( FConnectionState = csSelected ) then
     begin
          if ( FMailBox.TotalMsgs > 0 ) then
          begin
               SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdFetch] + ' 1:' + IntToStr ( FMailBox.TotalMsgs ) + {Do not Localize}
               ' (' + IMAP4FetchDataItem[12] + ')' ), wsOk ); {Do not Localize}
               if ( ResultNo = wsOk ) then
               begin
                    Result := 0;
                    SlRetrieve := TStringList.Create;
                    try
                       for Fn := 0 to ( FMailBox.TotalMsgs - 1 )do
                       begin
                            BreakApart ( CmdResultDetails[Fn], ' ', SlRetrieve ); {Do not Localize}
                            if ( ( CompareStr ( SlRetrieve[0], '*' ) = 0 ) and {Do not Localize}
                                 ( CompareStr ( SlRetrieve[1], IntToStr ( Fn + 1 ) ) = 0 ) and
                                 ( CompareStr ( SlRetrieve[2], IMAP4Commands[cmdFetch] ) = 0 ) and 
                                 ( CompareStr ( SlRetrieve[3], '(RFC822.SIZE' ) = 0 ) ) then {Do not Localize}
                               Result := Result + StrToInt ( Copy ( SlRetrieve[4], 1, ( Length ( SlRetrieve[4] ) - 1 ) ) )
                            else
                            begin
                                 Result := 0;
                                 Break;
                            end;
                            SlRetrieve.Clear;
                       end;
                    finally
                           SlRetrieve.Free;
                    end;
               end
               else
                   Result := -1;
          end
          else
              Result := 0;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );     
          Result := -1;
     end;
end;

function TIdIMAP4.RetrieveMsgSize(const AMsgNum: Integer): Integer;
var SlRetrieve : TStringList;
begin
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + {Do not Localize}
          ' (' + IMAP4FetchDataItem[12] + ')' ), wsOk ); {Do not Localize}
          if ( ResultNo = wsOk ) then
          begin
               SlRetrieve := TStringList.Create;
               try
                  BreakApart ( CmdResultDetails[0], ' ', SlRetrieve ); {Do not Localize}
                  if ( ( CompareStr ( SlRetrieve[0], '*' ) = 0 ) and {Do not Localize}
                       ( CompareStr ( SlRetrieve[1], IntToStr ( AMsgNum ) ) = 0 ) and
                       ( CompareStr ( SlRetrieve[2], IMAP4Commands[cmdFetch] ) = 0 ) and 
                       ( CompareStr ( SlRetrieve[3], '(RFC822.SIZE' ) = 0 ) ) then {Do not Localize}
                     Result := StrToInt ( Copy ( SlRetrieve[4], 1, ( Length ( SlRetrieve[4] ) - 1 ) ) )
                  else
                      Result := -1;
               finally
                      SlRetrieve.Free;
               end;
          end
          else
              Result := -1;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );   
          Result := -1;
     end;
end;


function TIdIMAP4.CheckMsgSeen(const AMsgNum: Integer): Boolean;
var Fn : Integer;
begin
     Result := False;
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + {Do not Localize}
          ' (' + IMAP4FetchDataItem[7] + ')' ), wsOk ); {Do not Localize}
          if ( ResultNo = wsOk ) then
          begin
               for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
               begin
                    if ( Pos ( ( '* ' + IntToStr ( AMsgNum ) + ' FETCH (FLAGS ' ), {Do not Localize}
                               CmdResultDetails[Fn] ) > 0 ) then
                    begin
                         if ( Pos ( MessageFlags[mfSeen], CmdResultDetails[Fn] ) > 0 ) then
                            Result := True
                         else
                             Result := False;
                    end;
               end;
          end
          else
              Result := False;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );   
          Result := False;
     end;
end;

function TIdIMAP4.RetrieveFlags(const AMsgNum: Integer; AFlags: {Pointer}TIdMessageFlagsSet): Boolean;
var Fn : Integer;
    AuxString : String;
begin
     Result := False;
     if ( FConnectionState = csSelected ) then
     begin
          SendCmd ( NewCmdCounter, ( IMAP4Commands[cmdFetch] + ' ' + IntToStr ( AMsgNum ) + {Do not Localize}
          ' (' + IMAP4FetchDataItem[7] + ')' ), wsOk ); {Do not Localize}
          if ( ResultNo = wsOk ) then
          begin
               for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
               begin
                    if ( Pos ( ( '* ' + IntToStr ( AMsgNum ) + ' FETCH (FLAGS ' ), {Do not Localize}
                               CmdResultDetails[Fn] ) > 0 ) then
                    begin
                         AuxString := Copy ( CmdResultDetails[Fn],
                         ( Pos ( 'FLAGS (', CmdResultDetails[Fn] ) + Length ( 'FLAGS (' ) ), {Do not Localize}
                         Length ( CmdResultDetails[Fn] ) );
                         AuxString := Copy ( AuxString, 1, ( Pos ( '))', AuxString ) - 1 ) ); {Do not Localize}
                         ParseFlagString ( AuxString, AFlags );
                         Result := True;
                    end;
               end;
          end
          else
              Result := False;
     end
     else
     begin
          raise EIdConnectionStateError.CreateFmt (
                RSIMAP4ConnectionStateError, [GetConnectionStateName] );    
          Result := False;
     end;
end;

function TIdIMAP4.GetConnectionStateName: String;
begin
     case FConnectionState of
       csAny : Result := RSIMAP4ConnectionStateAny;
       csNonAuthenticated : Result := RSIMAP4ConnectionStateNonAuthenticated;
       csAuthenticated : Result := RSIMAP4ConnectionStateAuthenticated;
       csSelected : Result := RSIMAP4ConnectionStateSelected;
     end;
end;

{ ...TIdIMAP4 Commands }

{ Parser Functions... }

procedure TIdIMAP4.ParseExpungeResultDetails(AMB: TIdMailBox;
  CmdResultDetails: TStrings);
var Fn : Integer;
    SlExpunge : TStringList;
begin
     SlExpunge := TStringList.Create;
     SetLength ( AMB.DeletedMsgs, 0 );
     try
        for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
        begin
             BreakApart ( CmdResultDetails[Fn], ' ', SlExpunge ); {Do not Localize}
             if ( ( CompareStr ( SlExpunge[0], '*' ) = 0 ) and {Do not Localize}
                  ( CompareStr ( SlExpunge[2], IMAP4Commands[cmdExpunge] ) = 0 ) ) then 
             begin
                  SetLength ( AMB.DeletedMsgs, ( Length ( AMB.DeletedMsgs ) + 1 ) );
                  AMB.DeletedMsgs[Length ( AMB.DeletedMsgs ) - 1] := StrToInt ( SlExpunge[1] );
             end;
             SlExpunge.Clear;
        end;
     finally
            SlExpunge.Free;
     end;
end;

procedure TIdIMAP4.ParseFlagString(AFlagsList: String;
  var AFlags: TIdMessageFlagsSet);
var SlFlags : TStringList;
    Fn : Integer;
begin
     SlFlags := TStringList.Create;
     AFlags := [];
     BreakApart ( AFlagsList, ' ', SlFlags ); {Do not Localize}
     try
        for Fn := 0 to ( SlFlags.Count - 1 ) do
        begin
             if ( CompareStr ( SlFlags[Fn], MessageFlags[mfAnswered] ) = 0 ) then
             begin
                  AFlags := AFlags + [mfAnswered];
             end
             else if ( CompareStr ( SlFlags[Fn], MessageFlags[mfFlagged] ) = 0 )then
             begin
                  AFlags := AFlags + [mfFlagged];
             end
             else if ( CompareStr ( SlFlags[Fn], MessageFlags[mfDeleted] ) = 0 )then
             begin
                  AFlags := AFlags + [mfDeleted];
             end
             else if ( CompareStr ( SlFlags[Fn], MessageFlags[mfDraft] ) = 0 )then
             begin
                  AFlags := AFlags + [mfDraft];
             end
             else if ( CompareStr ( SlFlags[Fn], MessageFlags[mfSeen] ) = 0 )then
             begin
                  AFlags := AFlags + [mfSeen];
             end;
        end;
     finally
            SlFlags.Free;
     end;
end;

procedure TIdIMAP4.ParseSearchResultDetails(AMB: TIdMailBox;
  CmdResultDetails: TStrings);
var Fn,
    Fi : Integer;
    SlSearch : TStringList;
begin
     SlSearch := TStringList.Create;
     SetLength ( AMB.SearchResult, 0 );
     try
        for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
        begin
             BreakApart ( CmdResultDetails[Fn], ' ', SlSearch ); {Do not Localize}
             if ( ( CompareStr ( SlSearch[0], '*' ) = 0 ) and {Do not Localize}
                  ( CompareStr ( SlSearch[1], IMAP4Commands[cmdSearch] ) = 0 ) ) then 
             begin
                  for Fi := 2 to ( SlSearch.Count - 1 ) do
                  begin
                       SetLength ( AMB.SearchResult, ( Length ( AMB.SearchResult ) + 1 ) );
                       AMB.SearchResult[Length ( AMB.SearchResult ) - 1] := StrToInt ( SlSearch[Fi] );
                  end;
                  Break;
             end;
             SlSearch.Clear;
        end;
     finally
            SlSearch.Free;
     end;
end;

procedure TIdIMAP4.ParseStatusResultDetails(AMB: TIdMailBox;
  CmdResultDetails: TStrings);
var Fn,
    Fi : Integer;
    AuxString : String;
    SlStatus : TStringList;
begin
     SlStatus := TStringList.Create;
     try
        for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
        begin
             if ( Pos ( '* STATUS', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
             begin
                  AuxString := Copy ( CmdResultDetails[Fn],
                  ( Pos ( '* STATUS', CmdResultDetails[Fn] ) + Length ( '* STATUS' ) ), {Do not Localize}
                  Length ( CmdResultDetails[Fn] ) );
                  AMB.Name := Trim ( Copy ( AuxString, 1, ( Pos ( '(', AuxString ) - 1 ) ) ); {Do not Localize}
                  AuxString := Copy ( AuxString, ( Pos ( '(', AuxString ) + 1 ), {Do not Localize}
                  ( Length ( AuxString ) - Pos ( '(', AuxString ) - 1 ) ); {Do not Localize}
                  BreakApart ( AuxString, ' ', SlStatus ); {Do not Localize}
                  Fi := 0;
                  while ( Fi < SlStatus.Count ) do
                  begin
                       if ( CompareStr ( SlStatus[Fi],
                            'MESSAGES' ) = 0 ) then {Do not Localize}
                       begin
                            AMB.TotalMsgs := StrToInt ( SlStatus[Fi + 1] );
                            Fi := Fi + 2;
                       end
                       else if ( CompareStr ( SlStatus[Fi],
                                 'RECENT' ) = 0 ) then {Do not Localize}
                       begin
                            AMB.RecentMsgs := StrToInt ( SlStatus[Fi + 1] );
                            Fi := Fi + 2;
                       end
                       else if ( CompareStr ( SlStatus[Fi],
                                 'UNSEEN' ) = 0 ) then {Do not Localize}
                       begin
                            AMB.UnseenMsgs := StrToInt ( SlStatus[Fi + 1] );
                            Fi := Fi + 2;
                       end
                       else if ( CompareStr ( SlStatus[Fi],
                                 'UIDNEXT' ) = 0 ) then {Do not Localize}
                       begin
                            AMB.UIDNext := SlStatus[Fi + 1];
                            Fi := Fi + 2;
                       end
                       else if ( CompareStr ( SlStatus[Fi],
                                 'UIDVALIDITY' ) = 0 ) then {Do not Localize}
                       begin
                            AMB.UIDValidity := SlStatus[Fi + 1];
                            Fi := Fi + 2;
                       end;
                  end;
                  SlStatus.Clear;
             end;
        end;
     finally
            SlStatus.Free;
     end;
end;

procedure TIdIMAP4.ParseSelectResultDetails(ATag: String;
  AMB : TIdMailBox; CmdResultDetails: TStrings);
var Fn : Integer;
    AuxString : String;
    LFlags: TIdMessageFlagsSet;
begin
     for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
     begin
          if ( Pos ( 'EXISTS', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               AMB.TotalMsgs := StrToInt ( Trim ( Copy ( CmdResultDetails[Fn], 3,
               ( Pos ( 'EXISTS', CmdResultDetails[Fn] ) - 4 ) ) ) ); {Do not Localize}
          end;
          if ( Pos ( 'RECENT', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               AMB.RecentMsgs := StrToInt ( Trim ( Copy ( CmdResultDetails[Fn], 3,
               ( Pos ( 'RECENT', CmdResultDetails[Fn] ) - 4 ) ) ) ); {Do not Localize}
          end;
          if ( Pos ( '* OK [UIDVALIDITY', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               AMB.UIDValidity := Trim ( Copy ( CmdResultDetails[Fn],
               ( Pos ( '* OK [UIDVALIDITY', CmdResultDetails[Fn] ) + {Do not Localize}
               Length ( '* OK [UIDVALIDITY' ) ), {Do not Localize}
               ( Pos ( ']', CmdResultDetails[Fn] ) - Length ( '* OK [UIDVALIDITY' ) - 1 ) ) ); {Do not Localize}
          end;
          if ( Pos ( '* FLAGS', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               ParseFlagString ( Copy ( CmdResultDetails[Fn],
               ( Pos ( '(', CmdResultDetails[Fn] ) + 1 ), {Do not Localize}
               ( Pos ( ')', CmdResultDetails[Fn] ) - Pos ( '(', CmdResultDetails[Fn] ) - 1 ) ), LFlags ); {Do not Localize}
               AMB.Flags := LFlags;
          end;
          if ( Pos ( '* OK [PERMANENTFLAGS', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               ParseFlagString ( Copy ( CmdResultDetails[Fn],
               ( Pos ( '(', CmdResultDetails[Fn] ) + 1 ), {Do not Localize}
               ( Pos ( ')', CmdResultDetails[Fn] ) - Pos ( '(', CmdResultDetails[Fn] ) - 1 ) ), {Do not Localize}
               LFlags );
               AMB.ChangeableFlags := LFlags;
          end;
          if ( Pos ( '* OK [UNSEEN', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               AMB.FirstUnseenMsg := StrToInt ( Trim ( Copy ( CmdResultDetails[Fn],
               ( Pos ( '* OK [UNSEEN ', CmdResultDetails[Fn] ) + {Do not Localize}
               Length ( '* OK [UNSEEN ' ) ), {Do not Localize}
               ( Pos ( ']', CmdResultDetails[Fn] ) - Length ( '* OK [UNSEEN ' ) - 1 ) ) ) ); {Do not Localize}
          end;
          if ( Pos ( ATag + ' OK', CmdResultDetails[Fn] ) > 0 ) then {Do not Localize}
          begin
               AuxString := Trim ( Copy ( CmdResultDetails[Fn],
                    ( Pos ( '[', CmdResultDetails[Fn] ) ), {Do not Localize}
                    ( Pos ( ']', CmdResultDetails[Fn] ) - Pos ( '[', CmdResultDetails[Fn] ) + 1 ) ) ); {Do not Localize}
               if ( CompareStr ( AuxString, '[READ-WRITE]' ) = 0 ) then {Do not Localize}
               begin
                    AMB.State := msReadWrite;
               end
               else if ( CompareStr ( AuxString, '[READ-ONLY]' ) = 0 ) then {Do not Localize}
               begin
                    AMB.State := msReadOnly;
               end;
          end;
     end;
end;

procedure TIdIMAP4.ParseListResultDetails(AMBList: TStringList;
  CmdResultDetails: TStrings);
var Fn : Integer;
    AuxString : String;
    AuxPChar : PChar;
begin
     AMBList.Clear;
     for Fn := 0 to ( CmdResultDetails.Count - 1 ) do
     begin
          AuxString := CmdResultDetails[Fn];
          if ( ( CompareStr ( AuxString[1], '*' ) = 0 ) and {Do not Localize}
               ( CompareStr ( Copy ( AuxString, 3, 4 ), IMAP4Commands[cmdList] ) = 0 ) ) then
          begin
               //TODO: Get mail box attributes here;
               AuxString := Trim ( Copy (
               AuxString, ( Pos ( '"/"', AuxString ) + 4 ), Length ( AuxString ) ) ); {Do not Localize}
               AuxPChar := PChar ( AuxString );
               if ( ( AuxString[1] = '"' ) and {Do not Localize}
                    ( AuxString[Length ( AuxString )] = '"' ) ) then {Do not Localize}
                  AMBList.Add ( AnsiExtractQuotedStr ( AuxPChar, '"' ) ) {Do not Localize}
               else
                   AMBList.Add ( AuxString );
          end;
     end;
end;

{ ...Parser Functions }

function TIdIMAP4.ArrayToNumberString (const AMsgNumList: array of Integer): String;
var Fn : Integer;
begin
     for Fn := 0 to ( Length ( AMsgNumList ) - 1 ) do
         Result := Result + IntToStr ( AMsgNumList[Fn] ) + ','; {Do not Localize}
     SetLength ( Result, ( Length ( Result ) - 1 ) );
end;

procedure TIdIMAP4.SetMailBox(const Value: TIdMailBox);
begin
     FMailBox.Assign ( Value );
end;

end.
