unit IdGlobal;

interface

{$I IdCompilerDefines.inc}

{This is the only unit with references to OS specific units and IFDEFs. NO OTHER units
are permitted to do so except .pas files which are counterparts to dfm/xfm files, and only for
support of that.}

uses
  {$IFNDEF LINUX}
  Windows,
  {$ENDIF}
  Classes,
  IdException,
  SyncObjs, SysUtils;

type
  TIdOSType = (otUnknwon, otLinux, otWindows);

const
  IdTimeoutDefault = -1;
  IdTimeoutInfinite = -2;

  gsIdProductName = 'Indy';  {do not localize}
  gsIdVersion = '9.0.2-B';   {do not localize}
  //
  CHAR0 = #0;
  BACKSPACE = #8;
  LF = #10;
  CR = #13;
  EOL = CR + LF;
  TAB = #9;
  CHAR32 = #32;
  LWS = [TAB, CHAR32];
  {$IFDEF Linux}
  GPathDelim = '/'; {do not localize}
  GOSType = otLinux;
  {$ELSE}
  GPathDelim = '\'; {do not localize}
  GOSType = otWindows;
  {$ENDIF}

  wdays: array[1..7] of string = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri'
   , 'Sat'); {do not localize}
  monthnames: array[1..12] of string = ('Jan', 'Feb', 'Mar', 'Apr', 'May'
   , 'Jun',  'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'); {do not localize}

type
  {$IFDEF LINUX}
  // Dummy Consts, see SetThreadPriority
  TIdThreadPriority = (tpIdle, tpLowest, tpLower, tpNormal, tpHigher, tpHighest
   , pTimeCritical);
  {$ELSE}
  TIdThreadPriority = TThreadPriority;
  {$ENDIF}
  TIdReadLnFunction = function: string of object;
  TStringEvent = procedure(ASender: TComponent; const AString: String);
  TPosProc = function(const Substr, S: string): Integer;
  TIdReuseSocket = (rsOSDependent, rsTrue, rsFalse);

  TIdCardinalBytes = record
    case Integer of
    0: (
      Byte1: Byte;
      Byte2: Byte;
      Byte3: Byte;
      Byte4: Byte;);
    1: (Whole: Cardinal);
    2: (CharArray : array[0..3] of Char);
  end;

  TIdLocalEvent = class(TEvent)
  public
    constructor Create(const AInitialState: Boolean = False;
     const AManualReset: Boolean = False); reintroduce;
    function WaitFor: TWaitResult; overload;
  end;

  TIdMimeTable = class(TObject)
  protected
    FMIMEList: TStringList;
    FFileExt: TStringList;
  public
    procedure BuildCache; virtual;
    function GetFileMIMEType(const AFileName: string): string;
    function GetDefaultFileExt(Const MIMEType: string): string;
    constructor Create(Autofill: boolean=true); virtual;
    destructor Destroy; override;
  end;

  TCharSet = (csGB2312, csBig5, csIso2022jp, csEucKR, csIso88591);

  {$IFNDEF VCL6ORABOVE}
  PByte=^Byte;
  {$ENDIF}

  {$IFDEF LINUX}
  TIdPID = Integer;
  {$ELSE}
  TIdPID = LongWord;
  {$ENDIF}

  {$IFNDEF LINUX}
  TIdWin32Type = (Win32s, WindowsNT40, Windows95, Windows95OSR2, Windows98, Windows98SE,Windows2000, WindowsMe, WindowsXP);
  {$ENDIF}

  //This is called whenever there is a failure to retreive the time zone information
  EIdFailedToRetreiveTimeZoneInfo = class(EIdException);
  //This usually is a property editor exception
  EIdCorruptServicesFile = class(EIdException);

// Procs - KEEP THESE ALPHABETICAL!!!!!
  {$IFNDEF VCL5ORABOVE}
  function AnsiSameText(const S1, S2: string): Boolean;
  procedure FreeAndNil(var Obj);
  {$ENDIF}
  {$IFNDEF LINUX}
  function GetFileCreationTime(const Filename: string): TDateTime;
  function GetInternetFormattedFileTimeStamp(AFile : String):String;
  {$ENDIF}
//  procedure BuildMIMETypeMap(dest: TStringList);
  function BreakApart(BaseString, BreakString: string; StringList: TStrings): TStrings;
  procedure CommaSeperatedToStringList(AList: TStrings; const Value:string);
  function CopyFileTo(const Source, Destination: string): Boolean;
  function CurrentProcessId: TIdPID;
  function DateTimeToGmtOffSetStr(ADateTime: TDateTime; SubGMT: Boolean): string;
  Function DateTimeToInternetStr(const Value: TDateTime; const AIsGMT : Boolean = False) : String;
  procedure DebugOutput(const AText: string);
  function DomainName(AHost: String): String;
  function Fetch(var AInput: string; const ADelim: string = ' '; const ADelete: Boolean = True)
   : string;
  function FileSizeByName(AFilename: string): Int64;
  function GetMIMETypeFromFile(AFile: TFileName): string;
  function GetSystemLocale: TCharSet;
  function GetTickCount: Cardinal;
  function GmtOffsetStrToDateTime(S: string): TDateTime;
  function GMTToLocalDateTime(S: string): TDateTime;
  function IdPorts: TList;
  function iif(const ATest: Boolean; const ATrue: Integer; const AFalse: Integer): Integer;
   overload;
  function iif(const ATest: Boolean; const ATrue: string; const AFalse: string): string; overload;
  function IncludeTrailingSlash(const APath: string): string;
  function IntToBin(Value: cardinal): string;
  function IndyGetHostName: string;
  function IndyInterlockedIncrement(var I: Integer): Integer;
  function IndyInterlockedDecrement(var I: Integer): Integer;
  function IndyInterlockedExchange(var A: Integer; B: Integer): Integer;
  function IndyInterlockedExchangeAdd(var A: Integer; B: Integer): Integer;
  function IsCurrentThread(AThread: TThread): boolean;
  function IsDomain(S: String): Boolean;
  function IsFQDN(S: String): Boolean;
  function IsHostname(S: String): Boolean;
  function IsNumeric(c: char): Boolean; overload;
  function IsNumeric(const AString: string): Boolean; overload;
  function IsTopDomain(s: string): Boolean;
  function IsValidIP(S: String): Boolean;
  function InMainThread: boolean;
  function Max(AValueOne,AValueTwo: Integer): Integer;
  function MakeTempFilename: string;
  function Min(AValueOne, AValueTwo : Integer): Integer;
  function OffsetFromUTC: TDateTime;
  function PosInStrArray(SearchStr: string; Contents: array of string;
   const CaseSensitive: Boolean=True): Integer;
  function ProcessPath(ABasePath: String; const APath: String;
   const APathDelim: string = '/'): string;
  function RightStr(st : String; Len : Integer): String;
  function ROL(val: LongWord; shift: Byte): LongWord;
  function ROR(val: LongWord; shift: Byte): LongWord;
  function RPos(const ASub, AIn: String; AStart: Integer = -1): Integer;
  function SetLocalTime(Value: TDateTime): boolean;
  procedure SetThreadPriority(AThread: TThread;
   const APriority: TIdThreadPriority);
  procedure Sleep(ATime: cardinal);
  function StrToCard(AVal : String) : Cardinal;
  function StrInternetToDateTime(Value: string): TDateTime;
  function StrToDay(const ADay: string): Byte;
  function StrToMonth(const AMonth: string): Byte;
  function SubStrPos(substr: String; MemBuff: PChar; MemorySize: Integer): Integer;
  function TimeZoneBias: TDateTime;
  function UpCaseFirst(S: string): string;
  {$IFNDEF LINUX}
  function Win32Type : TIdWin32Type;
  {$ENDIF}

var
  IndyPos: TPosProc = nil;
  {$IFDEF LINUX}
  // For linux the user needs to set these variables to be accurate where used (mail, etc)
  GOffsetFromUTC: TDateTime = 0;
  GSystemLocale: TCharSet = csIso88591;
  GTimeZoneBias: TDateTime = 0;
  {$ENDIF}

implementation

uses
  {$IFDEF LINUX}
  Libc,
  IdStackLinux,
  {$ELSE}
  IdStackWinsock,
  Registry,
  {$ENDIF}
  IdStack, IdResourceStrings, IdURI;

const
  WhiteSpace = [#0..#12, #14..' ']; {do not localize}

var
  FIdPorts: TList;
  {$IFNDEF LINUX}
  ATempPath: string;
  {$ENDIF}

{This routine is based on JPM Open by J. Peter Mugaas.  Permission is granted
to use this with Indy under Indy's Licenses

Note that JPM Open is under a different Open Source license model.

It is available at http://www.wvnet.edu/~oma00215/jpm.html }

{$IFNDEF LINUX}
function Win32Type: TIdWin32Type;
var
  VerInfo: TOSVersionInfo;
begin
  VerInfo.dwOSVersionInfoSize := SizeOf(TOSVersionInfo);
  GetVersionEx(VerInfo);
  {is this Windows 2000 or XP?}
  if VerInfo.dwMajorVersion >= 5 then
  begin
    if VerInfo.dwMinorVersion >= 1 then
    begin
      Result := WindowsXP;
    end
    else
    begin
      Result := Windows2000;
    end;
  end
  else
  begin
      {is this WIndows 95, 98, Me, or NT 40}
    if VerInfo.dwMajorVersion > 3 then
    begin
      if VerInfo.dwPlatformId = VER_PLATFORM_WIN32_NT then
      begin
        Result := WindowsNT40;
      end
      else
      begin
        {mask off junk}
        VerInfo.dwBuildNumber := VerInfo.dwBuildNumber and $FFFF;
        if VerInfo.dwMinorVersion >= 90 then
        begin
          Result := WindowsMe;
        end
        else
        begin
          if VerInfo.dwMinorVersion >= 10 then
          begin
            {Windows 98}
            if VerInfo.dwBuildNumber >= 2222 then
              Result := Windows98SE
            else
              Result := Windows98;
          end
          else
          begin
            {Windows 95}
            if VerInfo.dwBuildNumber >= 1000 then
              Result := Windows95OSR2
            else
              Result := Windows95;
          end;
        end;
      end;
    end
    else
      Result := Win32s;
  end;
end;
{$ENDIF}

{This is an internal procedure so the StrInternetToDateTime and GMTToLocalDateTime can share common code}
function RawStrInternetToDateTime(var Value: string): TDateTime;
var
  i: Integer;
  Dt, Mo, Yr, Ho, Min, Sec: Word;
  sTime: String;
  ADelim: string;

  Procedure ParseDayOfMonth;
  begin
    Dt :=  StrToIntDef( Fetch(Value, ADelim), 1);
    Value := TrimLeft(Value);
  end;

  Procedure ParseMonth;
  begin
    Mo := StrToMonth( Fetch ( Value, ADelim )  );
    Value := TrimLeft(Value);
  end;
begin
  Result := 0.0;
  Value := Trim(Value);
  if Length(Value) = 0 then begin
    Exit;
  end;

  try
    {Day of Week}
    if StrToDay(Copy(Value, 1, 3)) > 0 then begin
      Fetch(Value);
      Value := TrimLeft(Value);
    end;

    // Workaround for some buggy web servers which use '-' to separate the date parts.
    if (IndyPos('-', Value) > 1) and (IndyPos('-', Value) < IndyPos(' ', Value)) then begin
      ADelim := '-';
    end
    else begin
      ADelim := ' ';
    end;
    //workaround for improper dates such as 'Fri, Sep 7 2001'
    //RFC 2822 states that they should be like 'Fri, 7 Sep 2001'
    if (StrToMonth(Fetch(Value, ADelim,False)) > 0) then
    begin
      {Month}
      ParseMonth;
      {Day of Month}
      ParseDayOfMonth;
    end
    else
    begin
      {Day of Month}
      ParseDayOfMonth;
      {Month}
      ParseMonth;
    end;
    {Year}
    Yr := StrToIntDef ( Fetch ( Value ), 1900 );
    if Yr < 80 then begin
      Inc(Yr, 2000);
    end else if Yr < 100 then begin
      Inc(Yr, 1900);
    end;

    Result := EncodeDate(Yr, Mo, Dt);
    // SG 26/9/00: Changed so that ANY time format is accepted
    i := IndyPos(':', Value); {do not localize}
    if i > 0 then begin
      // Copy time string up until next space (before GMT offset)
      sTime := fetch(Value, ' ');  {do not localize}
      {Hour}
      Ho  := StrToIntDef( Fetch ( sTime,':'), 0);  {do not localize}
      {Minute}
      Min := StrToIntDef( Fetch ( sTime,':'), 0);  {do not localize}
      {Second}
      Sec := StrToIntDef( Fetch ( sTime ), 0);
      {The date and time stamp returned}
      Result := Result + EncodeTime(Ho, Min, Sec, 0);
    end;
    Value := TrimLeft(Value);
  except
    Result := 0.0;
  end;
end;

function IncludeTrailingSlash(const APath: string): string;
begin
  {for some odd reason, the IFDEF's were not working in Delphi 4
  so as a workaround and to ensure some code is actually compiled into
  the procedure, I use a series of $elses}
  {$IFDEF VCL5O}
  Result := IncludeTrailingBackSlash(APath);
  {$ELSE}
    {$IFDEF VCL6ORABOVE}
    Result :=  IncludeTrailingPathDelimiter(APath);
    {$ELSE}
    Result := APath;
    if not IsPathDelimiter(Result, Length(Result)) then begin
      Result := Result + GPathDelim;
    end;
    {$ENDIF}
  {$ENDIF}
end;

{$IFNDEF VCL5ORABOVE}
function AnsiSameText(const S1, S2: string): Boolean;
begin
  Result := CompareString(LOCALE_USER_DEFAULT, NORM_IGNORECASE, PChar(S1)
   , Length(S1), PChar(S2), Length(S2)) = 2;
end;

procedure FreeAndNil(var Obj);
var
  P: TObject;
begin
  if TObject(Obj) <> nil then begin 
    P := TObject(Obj);
    TObject(Obj) := nil;  // clear the reference before destroying the object
    P.Free;
  end;
end;
{$ENDIF}

{$IFNDEF LINUX}
  {$IFNDEF VCL5ORABOVE}
  function CreateTRegistry: TRegistry;
  begin
    Result := TRegistry.Create;
  end;
  {$ELSE}
  function CreateTRegistry: TRegistry;
  begin
    Result := TRegistry.Create(KEY_READ);
  end;
  {$ENDIF}
{$ENDIF}

function Max(AValueOne,AValueTwo: Integer): Integer;
begin
  if AValueOne < AValueTwo then
  begin
    Result := AValueTwo
  end //if AValueOne < AValueTwo then
  else
  begin
    Result := AValueOne;
  end; //else..if AValueOne < AValueTwo then
end;

function Min(AValueOne, AValueTwo : Integer): Integer;
begin
  If AValueOne > AValueTwo then
  begin
    Result := AValueTwo
  end //If AValueOne > AValueTwo then
  else
  begin
    Result := AValueOne;
  end; //..If AValueOne > AValueTwo then
end;

{This should never be localized}
function DateTimeToInternetStr(const Value: TDateTime; const AIsGMT : Boolean = False) : String;
var
  wDay,
  wMonth,
  wYear: Word;
begin
  DecodeDate(Value, wYear, wMonth, wDay);
  Result := Format('%s, %d %s %d %s %s',    {do not localize}
                   [wdays[DayOfWeek(Value)], wDay, monthnames[wMonth],
                    wYear, FormatDateTime('HH":"NN":"SS', Value),  {do not localize}
                    DateTimeToGmtOffSetStr(OffsetFromUTC, AIsGMT)]);
end;

function StrInternetToDateTime(Value: string): TDateTime;
begin
  Result := RawStrInternetToDateTime(Value);
end;

{$IFNDEF LINUX}
function GetInternetFormattedFileTimeStamp(AFile : String):String;
const
  wdays: array[1..7] of string = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'); {do not localize}
  monthnames: array[1..12] of string = ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',   {do not localize}
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'); {do not localize}
var
  DT1, DT2 : TDateTime;
  wDay, wMonth, wYear: Word;
begin
  DT1 := GetFileCreationTime(AFile);
  DecodeDate(DT1, wYear, wMonth, wDay);
  DT2 := TimeZoneBias;
  Result := Format('%s, %d %s %d %s %s', [wdays[DayOfWeek(DT1)], wDay, monthnames[wMonth],   {do not localize}
   wYear, FormatDateTime('HH":"NN":"SS', DT1), DateTimeToGmtOffSetStr(DT2,False)]);   {do not localize}
end;

function GetFileCreationTime(const Filename: string): TDateTime;
var
  Data: TWin32FindData;
  H: THandle;
  FT: TFileTime;
  I: Integer;
begin
  H := FindFirstFile(PCHAR(Filename), Data);
  if H <> INVALID_HANDLE_VALUE then begin
    try
      FileTimeToLocalFileTime(Data.ftLastWriteTime, FT);
      FileTimeToDosDateTime(FT, LongRec(I).Hi, LongRec(I).Lo);
      Result := FileDateToDateTime(I);
    finally
      Windows.FindClose(H);
    end
  end else begin
    Result := 0;
  end;
end;
{$ENDIF}

function BreakApart(BaseString, BreakString: string; StringList: TStrings): TStrings;
var
  EndOfCurrentString: integer;
begin
  repeat
    EndOfCurrentString := Pos(BreakString, BaseString);
    if (EndOfCurrentString = 0) then
    begin
      StringList.add(BaseString);
    end
    else
      StringList.add(Copy(BaseString, 1, EndOfCurrentString - 1));
    delete(BaseString, 1, EndOfCurrentString + Length(BreakString) - 1); //Copy(BaseString, EndOfCurrentString + length(BreakString), length(BaseString) - EndOfCurrentString);
  until EndOfCurrentString = 0;
  result := StringList;
end;

procedure CommaSeperatedToStringList(AList: TStrings; const Value:string);
var
  iStart,
  iEnd,
  iQuote,
  iPos,
  iLength : integer ;
  sTemp : string ;
begin
  iQuote := 0;
  iPos := 1 ;
  iLength := Length(Value) ;
  AList.Clear ;
  while (iPos <= iLength) do
  begin
    iStart := iPos ;
    iEnd := iStart ;
    while ( iPos <= iLength ) do
    begin
      if Value[iPos] = '"' then  {do not localize}
      begin
        inc(iQuote);
      end;
      if Value[iPos] = ',' then  {do not localize}
      begin
        if iQuote <> 1 then
        begin
          break;
        end;
      end;
      inc(iEnd);
      inc(iPos);
    end ;
    sTemp := Trim(Copy(Value, iStart, iEnd - iStart));
    if Length(sTemp) > 0 then
    begin
      AList.Add(sTemp);
    end;
    iPos := iEnd + 1 ;
    iQuote := 0 ;
  end ;
end;

{$IFDEF LINUX}
function CopyFileTo(const Source, Destination: string): Boolean;
var
  SourceStream: TFileStream;
begin
  //TODO: Change to use a Linux copy function
  Result := false;
  if not FileExists(Destination) then begin
    SourceStream := TFileStream.Create(Source, fmOpenRead); try
      with TFileStream.Create(Destination, fmCreate) do try
        CopyFrom(SourceStream, 0);
      finally Free; end;
    finally SourceStream.free; end;
    Result := true;
  end;
end;
{$ELSE}
function CopyFileTo(const Source, Destination: string): Boolean;
begin
  Result := CopyFile(PChar(Source), PChar(Destination), true);
end;
{$ENDIF}

{$IFNDEF LINUX}
function TempPath: string;
var
	i: integer;
begin
  SetLength(Result, MAX_PATH);
	i := GetTempPath(Length(Result), PChar(Result));
	SetLength(Result, i);
  IncludeTrailingSlash(Result);
end;
{$ENDIF}

function MakeTempFilename: string;
begin
  {$IFDEF LINUX}
  Result := tempnam(nil, 'Indy');    {do not localize}
  {$ELSE}
  SetLength(Result, MAX_PATH + 1);
  GetTempFileName(PChar(ATempPath), 'Indy', 0, PChar(result));  {do not localize}
  Result := PChar(Result);
  {$ENDIF}
end;

// Find a token given a direction (>= 0 from start; < 0 from end)
// S.G. 19/4/00:
//  Changed to be more readable
function RPos(const ASub, AIn: String; AStart: Integer = -1): Integer;
var
  i: Integer;
  LStartPos: Integer;
  LTokenLen: Integer;
begin
  result := 0;
  LTokenLen := Length(ASub);
  // Get starting position
  if AStart = -1 then begin
    AStart := Length(AIn);
  end;
  if AStart < (Length(AIn) - LTokenLen + 1) then begin
    LStartPos := AStart;
  end else begin
    LStartPos := (Length(AIn) - LTokenLen + 1);
  end;
  // Search for the string
  for i := LStartPos downto 1 do begin
    if AnsiSameText(Copy(AIn, i, LTokenLen), ASub) then begin
      result := i;
      break;
    end;
  end;
end;

function GetSystemLocale: TCharSet;
begin
{$IFDEF LINUX}
  Result := GSystemLocale;
{$ELSE}
  case SysLocale.PriLangID of
    LANG_CHINESE:
      if SysLocale.SubLangID = SUBLANG_CHINESE_SIMPLIFIED then
        Result := csGB2312
      else
        Result := csBig5;
    LANG_JAPANESE: Result := csIso2022jp;
    LANG_KOREAN: Result := csEucKR;
    else
      Result := csIso88591;
  end;
{$ENDIF}
end;

// OS-independant version
function FileSizeByName(AFilename: string): Int64;
begin
  with TFileStream.Create(AFilename, fmOpenRead or fmShareDenyNone) do try
    Result := Size;
  finally Free; end;
end;


Function RightStr(st : String; Len : Integer) : String;
begin
  if ( Len > Length ( st ) ) or ( Len < 0 ) then
  begin
    Result := st;
  end  //f ( Len > Length ( st ) ) or ( Len < 0 ) then
  else
  begin
    Result := Copy ( St, Length( st ) - Len, Len );
  end; //else ... f ( Len > Length ( st ) ) or ( Len < 0 ) then
end;

{$IFDEF LINUX}
function OffsetFromUTC: TDateTime;
begin
  //TODO: Fix OffsetFromUTC for Linux to be automatic from OS
  Result := GOffsetFromUTC;
end;
{$ELSE}
function OffsetFromUTC: TDateTime;
var
  iBias: Integer;
  tmez: TTimeZoneInformation;
begin
  Case GetTimeZoneInformation(tmez) of
    TIME_ZONE_ID_INVALID:
      raise EIdFailedToRetreiveTimeZoneInfo.Create(RSFailedTimeZoneInfo);
    TIME_ZONE_ID_UNKNOWN  :
       iBias := tmez.Bias;
    TIME_ZONE_ID_DAYLIGHT :
      iBias := tmez.Bias + tmez.DaylightBias;
    TIME_ZONE_ID_STANDARD :
      iBias := tmez.Bias + tmez.StandardBias;
    else
      raise EIdFailedToRetreiveTimeZoneInfo.Create(RSFailedTimeZoneInfo);
  end;
  {We use ABS because EncodeTime will only accept positve values}
  Result := EncodeTime(Abs(iBias) div 60, Abs(iBias) mod 60, 0, 0);
  {The GetTimeZone function returns values oriented towards convertin
   a GMT time into a local time.  We wish to do the do the opposit by returning
   the difference between the local time and GMT.  So I just make a positive
   value negative and leave a negative value as positive}
  if iBias > 0 then begin
    Result := 0 - Result;
  end;
end;
{$ENDIF}

function StrToCard(AVal : String) : Cardinal;
begin
  Result := StrToInt64Def(Trim(AVal),0);
end;

{$IFDEF LINUX}
function TimeZoneBias: TDateTime;
begin
  //TODO: Fix TimeZoneBias for Linux to be automatic
  Result := GTimeZoneBias;
end;
{$ELSE}
function TimeZoneBias: TDateTime;
var
  ATimeZone: TTimeZoneInformation;
begin
  case GetTimeZoneInformation(ATimeZone) of
    TIME_ZONE_ID_DAYLIGHT:
      Result := ATimeZone.Bias + ATimeZone.DaylightBias;
    TIME_ZONE_ID_STANDARD:
      Result := ATimeZone.Bias + ATimeZone.StandardBias;
    TIME_ZONE_ID_UNKNOWN:
      Result := ATimeZone.Bias;
    else
      raise EIdException.Create(SysErrorMessage(GetLastError));
  end;
  Result := Result / 1440;
end;
{$ENDIF}

function GetTickCount: Cardinal;
begin
  {$IFDEF LINUX}
  Result := clock div (CLOCKS_PER_SEC div 1000);
  {$ELSE}
  Result := Windows.GetTickCount;
  {$ENDIF}
end;

{$IFDEF LINUX}
function SetLocalTime(Value: TDateTime): boolean;
begin
  //TODO: Implement SetTime for Linux. This call is not critical.
  result := False;
end;
{$ELSE}
function SetLocalTime(Value: TDateTime): boolean;
{I admit that this routine is a little more complicated than the one
in Indy 8.0.  However, this routine does support Windows NT privillages
meaning it will work if you have administrative rights under that OS

Original author Kerry G. Neighbour with modifications and testing
from J. Peter Mugaas}
var
   dSysTime: TSystemTime;
   buffer: DWord;
   tkp, tpko: TTokenPrivileges;
   hToken: THandle;
begin
  Result := False;
  if SysUtils.Win32Platform = VER_PLATFORM_WIN32_NT then
  begin
    if not Windows.OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES or TOKEN_QUERY,
      hToken) then
    begin
      exit;
    end;
    Windows.LookupPrivilegeValue(nil, 'SE_SYSTEMTIME_NAME', tkp.Privileges[0].Luid);
    tkp.PrivilegeCount := 1;
    tkp.Privileges[0].Attributes := SE_PRIVILEGE_ENABLED;
    if not Windows.AdjustTokenPrivileges(hToken, FALSE, tkp, sizeof(tkp), tpko, buffer) then
    begin
      exit;
    end;
  end;
  DateTimeToSystemTime(Value, dSysTime);
  Result := Windows.SetLocalTime(dSysTime);
  {Undo the Process Privillage change we had done for the set time
  and close the handle that was allocated}
  if SysUtils.Win32Platform = VER_PLATFORM_WIN32_NT then
  begin
    Windows.AdjustTokenPrivileges(hToken, FALSE,tpko, sizeOf(tpko), tkp, Buffer);
    Windows.CloseHandle(hToken);
  end;
end;
{$ENDIF}

// IdPorts returns a list of defined ports in /etc/services
function IdPorts: TList;
var
  sLocation, s: String;
  idx, i, iPrev, iPosSlash: integer;
  sl: TStringList;
begin
  if FIdPorts = nil then
  begin
    FIdPorts := TList.Create;
    {$IFDEF LINUX}
    sLocation := '/etc/';  // assume Berkeley standard placement   {do not localize}
    {$ELSE}
    SetLength(sLocation, MAX_PATH);
    SetLength(sLocation, GetWindowsDirectory(pchar(sLocation), MAX_PATH));
    sLocation := IncludeTrailingSlash(sLocation);
    if Win32Platform = VER_PLATFORM_WIN32_NT then begin
      sLocation := sLocation + 'system32\drivers\etc\'; {do not localize}
    end;
    {$ENDIF}
    sl := TStringList.Create;
    try
      sl.LoadFromFile(sLocation + 'services');  {do not localize}
      iPrev := 0;
      for idx := 0 to sl.Count - 1 do
      begin
        s := sl[idx];
        iPosSlash := IndyPos('/', s);   {do not localize}
        if (iPosSlash > 0) and (not (IndyPos('#', s) in [1..iPosSlash])) then {do not localize}
        begin // presumably found a port number that isn't commented
          i := iPosSlash;
          repeat
            dec(i);
            if i = 0 then begin
              raise EIdCorruptServicesFile.CreateFmt(RSCorruptServicesFile, [sLocation + 'services']); {do not localize}
            end;
          until s[i] in WhiteSpace;
          i := StrToInt(Copy(s, i+1, iPosSlash-i-1));
          if i <> iPrev then begin
            FIdPorts.Add(TObject(i));
          end;
          iPrev := i;
        end;
      end;
    finally
      sl.Free;
    end;
  end;
  Result := FIdPorts;
end;

function Fetch(var AInput: string; const ADelim: string = ' '; const ADelete: Boolean = true)
 : string;
var
  LPos: integer;
  LResult: string;
begin
  if ADelim = #0 then begin
    // AnsiPos does not work with #0
    LPos := Pos(ADelim, AInput);
  end else begin
    LPos := IndyPos(ADelim, AInput);
  end;
  if LPos = 0 then begin
    LResult := AInput;
    if ADelete then begin
      AInput := '';
    end;
  end else begin
    LResult := Copy(AInput, 1, LPos - 1);
    if ADelete then begin
      Delete(AInput, 1, LPos + Length(ADelim) - 1);
    end;
  end;
  // This is necessary to ensure that Result is written last in cases where AInput and Result are
  // assigned to the same variable. Current compilers seem to handle it ok without this, but
  // we cannot safely predict internal workings or future optimizations
  Result := LResult;
end;

{This searches an array of string for an occurance of SearchStr}
function PosInStrArray(SearchStr : string; Contents: array of string; const CaseSensitive: Boolean=True): Integer;
begin
  for Result := Low(Contents) to High(Contents) do begin
    if CaseSensitive then begin
      if SearchStr = Contents[Result] then begin
        Exit;
      end;
    end else begin
      if ANSISameText(SearchStr, Contents[Result]) then begin
        Exit;
      end;
    end;
  end;  //for Result := Low(Contents) to High(Contents) do
  Result := -1;
end;

function IsCurrentThread(AThread: TThread): boolean;
begin
  result := AThread.ThreadID = GetCurrentThreadID;
end;

function IsNumeric(c: char): Boolean;
begin
  // Do not use IsCharAlpha or IsCharAlphaNumeric - they are Win32 routines
  Result := Pos(c, '0123456789') > 0; {do not localize}
end;

{$HINTS OFF}
function IsNumeric(const AString: string): Boolean;
var
  LCode: Integer;
  LVoid: Integer;
begin
  Val(AString, LVoid, LCode);
  Result := LCode = 0;
end;
{$HINTS ON}

function StrToDay(const ADay: string): Byte;
begin
  Result := Succ(PosInStrArray(Uppercase(ADay),
    ['SUN','MON','TUE','WED','THU','FRI','SAT']));   {do not localize}
end;

function StrToMonth(const AMonth: string): Byte;
begin
  Result := Succ(PosInStrArray(Uppercase(AMonth),
    ['JAN','FEB','MAR','APR','MAY','JUN','JUL','AUG','SEP','OCT','NOV','DEC']));   {do not localize}
end;

function UpCaseFirst(S: string): string;
begin
  Result := LowerCase(S);
  if Result <> '' then
  begin
    Result[1] := UpCase(Result[1]);
  end;
end;

function DateTimeToGmtOffSetStr(ADateTime: TDateTime; SubGMT: Boolean): string;
var
  AHour, AMin, ASec, AMSec: Word;
begin
  if (ADateTime = 0.0) and SubGMT then
  begin
    Result := 'GMT'; {do not localize}
    Exit;
  end;
  DecodeTime(ADateTime, AHour, AMin, ASec, AMSec);
  Result := Format(' %0.2d%0.2d', [AHour, AMin]); {do not localize}
  if ADateTime < 0.0 then
  begin
    Result[1] := '-'; {do not localize}
  end
  else
  begin
    Result[1] := '+';  {do not localize}
  end;
end;

// Currently this function is not used
(*
procedure BuildMIMETypeMap(dest: TStringList);
{$IFDEF LINUX}
begin
  // TODO: implement BuildMIMETypeMap in Linux
  raise EIdException.Create('BuildMIMETypeMap not implemented yet.');
end;
{$ELSE}
var
  Reg: TRegistry;
  slSubKeys: TStringList;
  i: integer;
begin
  Reg := CreateTRegistry; try
    Reg.RootKey := HKEY_CLASSES_ROOT;
    Reg.OpenKeyreadOnly('\MIME\Database\Content Type'); {do not localize}
    slSubKeys := TStringList.Create;
    try
      Reg.GetKeyNames(slSubKeys);
      reg.Closekey;
      for i := 0 to slSubKeys.Count - 1 do
      begin
        Reg.OpenKeyreadOnly('\MIME\Database\Content Type\' + slSubKeys[i]);  {do not localize}
        dest.Append(LowerCase(reg.ReadString('Extension')) + '=' + slSubKeys[i]); {do not localize}
        Reg.CloseKey;
      end;
    finally
      slSubKeys.Free;
    end;
  finally
    reg.free;
  end;
end;
{$ENDIF}
*)

function GetMIMETypeFromFile(AFile: TFileName): string;
var
  MIMEMap: TIdMIMETable;
begin
  MIMEMap := TIdMimeTable.Create(true);
  try
    result := MIMEMap.GetFileMIMEType(AFile);
  finally
    MIMEMap.Free;
  end;
end;

function GmtOffsetStrToDateTime(S: string): TDateTime;
begin
  Result := 0.0;
  S := Copy(Trim(s), 1, 5);
  if Length(S) > 0 then
  begin
    if s[1] in ['-', '+'] then   {do not localize}
    begin
      try
        Result := EncodeTime(StrToInt(Copy(s, 2, 2)), StrToInt(Copy(s, 4, 2)), 0, 0);
        if s[1] = '-' then  {do not localize}
        begin
          Result := -Result;
        end;
      except
        Result := 0.0;
      end;
    end;
  end;
end;

function GMTToLocalDateTime(S: string): TDateTime;
var  {-Always returns date/time relative to GMT!!  -Replaces StrInternetToDateTime}
  DateTimeOffset: TDateTime;
begin
  Result := RawStrInternetToDateTime(S);
  if Length(S) < 5 then begin
    DateTimeOffset := 0.0
  end else begin
    DateTimeOffset := GmtOffsetStrToDateTime(S);
  end;
  {-Apply GMT offset here}
  if DateTimeOffset < 0.0 then begin
    Result := Result + Abs(DateTimeOffset);
  end else begin
    Result := Result - DateTimeOffset;
  end;
  // Apply local offset
  Result := Result + OffSetFromUTC;
end;


procedure Sleep(ATime: cardinal);
begin
  {$IFDEF LINUX}
  GStack.WSSelect(nil, nil, nil, ATime)
  {$ELSE}
  Windows.Sleep(ATime);
  {$ENDIF}
end;

{ Takes a cardinal (DWORD)  value and returns the string representation of it's binary value}
function IntToBin(Value: cardinal): string;
var
  i: Integer;
begin
  SetLength(result, 32);
  for i := 1 to 32 do
  begin
    if ((Value shl (i-1)) shr 31) = 0 then
      result[i] := '0'  {do not localize}
    else
      result[i] := '1'; {do not localize}
  end;
end;

function CurrentProcessId: TIdPID;
begin
  {$IFDEF LINUX}
  Result := getpid;
  {$ELSE}
  Result := GetCurrentProcessID;
  {$ENDIF}
end;

function ROL(val : LongWord; shift : Byte) : LongWord; assembler;
asm
  mov  eax, val;
  mov  cl, shift;
  rol  eax, cl;
end;

function ROR(val : LongWord; shift : Byte) : LongWord; assembler;
asm
  mov  eax, val;
  mov  cl, shift;
  ror  eax, cl;
end;

procedure DebugOutput(const AText: string);
begin
  {$IFDEF LINUX}
  __write(stderr, AText, Length(AText));
  __write(stderr, EOL, Length(EOL));
  {$ELSE}
  OutputDebugString(PChar(AText));
  {$ENDIF}
end;

function InMainThread: boolean;
begin
  Result := GetCurrentThreadID = MainThreadID;
end;

{ TIdMimeTable }

{$IFDEF Linux}
procedure TIdMimeTable.BuildCache;
begin
  // TODO: implement for Linux
  // This is not a required function.
end;
{$ELSE}
procedure TIdMimeTable.BuildCache;
var
  reg: TRegistry;
  KeyList: TStringList;
  i: Integer;
begin
  // Build the file type/MIME type map
  Reg := CreateTRegistry; try
    KeyList := TStringList.create;
    try
      Reg.RootKey := HKEY_CLASSES_ROOT;
      Reg.OpenKeyReadOnly('\');  {do not localize}
      Reg.GetKeyNames(KeyList);
      reg.Closekey;
      // get a list of registered extentions
      for i := 0 to KeyList.Count - 1 do
      begin
        if Copy(KeyList[i], 1, 1) = '.' then   {do not localize}
        begin
          reg.OpenKeyReadOnly(KeyList[i]);
          if Reg.ValueExists('Content Type') then  {do not localize}
          begin
            FFileExt.Values[KeyList[i]] := Reg.ReadString('Content Type');  {do not localize}
          end;
          reg.CloseKey;
        end;
      end;
      Reg.OpenKeyreadOnly('\MIME\Database\Content Type'); {do not localize}

      // get a list of registered MIME types
      KeyList.Clear;

      Reg.GetKeyNames(KeyList);
      reg.Closekey;
      for i := 0 to KeyList.Count - 1 do
      begin
        Reg.OpenKeyreadOnly('\MIME\Database\Content Type\' + KeyList[i]); {do not localize}
        FMIMEList.Values[KeyList[i]] := reg.ReadString('Extension');  {do not localize}
        Reg.CloseKey;
      end;
    finally
      KeyList.Free;
    end;
  finally
    reg.free;
  end;
end;
{$ENDIF}

constructor TIdMimeTable.Create(Autofill: boolean);
begin
  FFileExt := TStringList.Create;
  FMIMEList := TStringList.Create;
  if Autofill then begin
    BuildCache;
  end;
end;

destructor TIdMimeTable.Destroy;
begin
  FreeAndNil(FMIMEList);
  FreeAndNil(FFileExt);
  inherited Destroy;
end;

function TIdMimeTable.getDefaultFileExt(const MIMEType: string): String;
begin
  result := FMIMEList.Values[MIMEType];
  if Length(result) = 0 then
  begin
    BuildCache;
    result := FMIMEList.Values[MIMEType];;
  end;
end;

function TIdMimeTable.GetFileMIMEType(const AFileName: string): string;
begin
  Result := FFileExt.Values[ExtractFileExt(AFileName)];
  if Length(Result) = 0 then begin
    BuildCache;
    Result := FFileExt.Values[ExtractFileExt(AFileName)];
    if Length(Result) = 0 then begin
      Result := 'application/octet-stream'; {do not localize}
    end;
  end;
end;

procedure SetThreadPriority(AThread: TThread; const APriority: TThreadPriority);
begin
  {$IFDEF LINUX}
  // Linux only allows root to adjust thread priorities, so we just ingnore this
  // call in Linux
  {$ELSE}
  AThread.Priority := APriority;
  {$ENDIF}
end;

function SBPos(const Substr, S: string): Integer;
// Necessary because of "Compiler magic"
begin
  Result := Pos(Substr, S);
end;

function SubStrPos(substr: String; MemBuff: PChar; MemorySize: Integer): Integer;
var
  ls: Integer;
  i: integer;
  S: String;
begin
  result := 0;
  ls := Length(substr);
  if ls > MemorySize then exit;

  for i := 0 to MemorySize - ls do begin
    if MemBuff[i] = substr[1] then begin
      SetString(S, MemBuff + i, ls);
      if S = substr then begin
        Result := i + 1;
        Exit;
      end;
    end;
  end;
end;

// Assembly is not allowed in Indy, however these routines can only be done in assembly because of
// the LOCK instruction. Both the Windows API and Kylix support these routines, but Windows 95
// fubars them up (Win98 works ok) so its necessary to have our own implementations.
function IndyInterlockedIncrement(var I: Integer): Integer;
asm
  MOV     EDX,1
  XCHG    EAX,EDX
  LOCK  XADD    [EDX],EAX
  INC     EAX
end;

function IndyInterlockedDecrement(var I: Integer): Integer;
asm
  MOV     EDX,-1
  XCHG    EAX,EDX
  LOCK  XADD    [EDX],EAX
  DEC     EAX
end;

function IndyInterlockedExchange(var A: Integer; B: Integer): Integer;
asm
  XCHG    [EAX],EDX
  MOV     EAX,EDX
end;

function IndyInterlockedExchangeAdd(var A: Integer; B: Integer): Integer;
asm
  XCHG    EAX,EDX
  LOCK  XADD    [EDX],EAX
end;

{$IFDEF LINUX}
function IndyGetHostName: string;
var
  LHost: array[1..255] of Char;
  i: LongWord;
begin
  //TODO: No need for LHost at all? Prob can use just Result
  if GetHostname(@LHost[1], 255) <> -1 then begin
    i := IndyPos(#0, LHost);
    SetLength(Result, i - 1);
    Move(LHost, Result[1], i - 1);
  end;
end;
{$ELSE}
function IndyGetHostName: string;
var
  i: LongWord;
begin
  SetLength(Result, MAX_COMPUTERNAME_LENGTH + 1);
  i := Length(Result);
  if GetComputerName(@Result[1], i) then begin
    SetLength(Result, i);
  end;
end;
{$ENDIF}


function IsValidIP(S: String): Boolean;
Var
  j, i: Integer;
begin
  result := true;
  s := Trim(S);
  for i := 1 to 4 do begin
    j := StrToIntDef(Fetch(S, '.'), -1);
    result := result and (j > -1) and (j < 256);
    if not result then break;
  end;
end;

// everething that does not start with '.' is treathed as hostname

function IsHostname(S: String): Boolean;
begin
  result := ((IndyPos('.', S) = 0) or (S[1] <> '.')) and (not IsValidIP(S));
end;

function IsTopDomain(s: string): Boolean;
Var
  i: Integer;
  S1: String;
begin
  i := 0;

  S := AnsiUpperCase(S);
  while IndyPos('.', S) > 0 do begin
    S1 := S;
    Fetch(S, '.');
    i := i + 1;
  end;

  result := ((Length(S) > 2) and (i = 1));
  if Length(S) = 2 then begin  // Country domain names
    S1 := Fetch(S1, '.');
    // here will be the exceptions check: com.uk, co.uk, com.tw and etc.
    if S = 'UK' then begin
      if S1 = 'CO' then result := i = 2;
      if S1 = 'COM' then result := i = 2;
    end;

    if S = 'TW' then begin
      if S1 = 'CO' then result := i = 2;
      if S1 = 'COM' then result := i = 2;
    end;
  end;
end;

function IsDomain(S: String): Boolean;
begin
  result := (not IsHostname(S)) and (IndyPos('.', S) > 0) and (not IsTopDomain(S));
end;

function DomainName(AHost: String): String;
begin
  result := Copy(AHost, IndyPos('.', AHost), Length(AHost));
end;

function IsFQDN(S: String): Boolean;
begin
  result := IsHostName(S) and IsDomain(DomainName(S));
end;

function ProcessPath(ABasePath: string; const APath: string;
 const APathDelim: string = '/'): string;
// Dont add / - sometimes a file is passed in as well and the only way to determine is
// to test against the actual targets
var
  i: Integer;
  LPreserveTrail: Boolean;
  LWork: string;
begin
  if IndyPos(APathDelim, APath) = 1 then begin
    Result := APath;
  end else begin
    Result := '';
    LPreserveTrail := (Copy(APath, Length(APath), 1) = APathDelim) or (Length(APath) = 0);
    LWork := ABasePath;
    // If LWork = '' then we just want it to be APath, no prefixed /
    if (Length(LWork) > 0) and (Copy(LWork, Length(LWork), 1) <> APathDelim) then begin
      LWork := LWork + APathDelim;
    end;
    LWork := LWork + APath;
    if Length(LWork) > 0 then begin
      i := 1;
      while i <= Length(LWork) do begin
        if LWork[i] = APathDelim then begin
          if i = 1 then begin
            Result := APathDelim;
          end else if Copy(Result, Length(Result), 1) <> APathDelim then begin
            Result := Result + LWork[i];
          end;
        end else if LWork[i] = '.' then begin
          // If the last character was a PathDelim then the . is a relative path modifier.
          // If it doesnt follow a PathDelim, its part of a filename
          if (Copy(Result, Length(Result), 1) = APathDelim) and (Copy(LWork, i, 2) = '..') then begin
            // Delete the last PathDelim
            Delete(Result, Length(Result), 1);
            // Delete up to the next PathDelim
            while (Length(Result) > 0) and (Copy(Result, Length(Result), 1) <> APathDelim) do begin
              Delete(Result, Length(Result), 1);
            end;
            // Skip over second .
            Inc(i);
          end else begin
            Result := Result + LWork[i];
          end;
        end else begin
          Result := Result + LWork[i];
        end;
        Inc(i);
      end;
    end;
    // Sometimes .. semantics can put a PathDelim on the end
    // But dont modify if it is only a PathDelim and nothing else, or it was there to begin with
    if (Result <> APathDelim) and (Copy(Result, Length(Result), 1) = APathDelim)
     and (LPreserveTrail = False) then begin
      Delete(Result, Length(Result), 1);
    end;
  end;
end;

{ TIdLocalEvent }

{$IFDEF LINUX}
const
  INFINITE = LongWord($FFFFFFFF);     { Infinite timeout }
{$ENDIF}

constructor TIdLocalEvent.Create(const AInitialState: Boolean = False;
 const AManualReset: Boolean = False);
begin
  inherited Create(nil, AManualReset, AInitialState, '');
end;

function TIdLocalEvent.WaitFor: TWaitResult;
begin
  Result := WaitFor(Infinite);
end;

function iif(const ATest: Boolean; const ATrue: Integer; const AFalse: Integer): Integer;
begin
  if ATest then begin
    Result := ATrue;
  end else begin
    Result := AFalse;
  end;
end;

function iif(const ATest: Boolean; const ATrue: string; const AFalse: string): string;
begin
  if ATest then begin
    Result := ATrue;
  end else begin
    Result := AFalse;
  end;
end;

initialization
  {$IFDEF LINUX}
  GStackClass := TIdStackLinux;
  {$ELSE}
  ATempPath := TempPath;
  GStackClass := TIdStackWinsock;
  {$ENDIF}
  // AnsiPos does not handle strings with #0 and is also very slow compared to Pos
  if LeadBytes = [] then begin
    IndyPos := SBPos;
  end else begin
    IndyPos := AnsiPos;
  end;
finalization
  FreeAndNil(FIdPorts);
end.
