unit IdFTPServer;
{
 Original Author: Sergio Perry
 Date: 04/21/2001

 Fixes and modifications: Doychin Bondzhev
 Date: 08/10/2001

 Further changes by Chad Z. Hower (Kudzu)

TODO:
 - Change events to use DoXXXX
}

interface

uses
  Classes,
  SysUtils,
  IdException, IdFTPList, IdTCPServer, IdTCPConnection, IdUserAccounts,
  IdFTPCommon, IdThread, IdRFCReply;

type
  TIdFTPServerThread = class;

  TOnUserLoginEvent = procedure(ASender: TIdFTPServerThread; AUsername, APassword: string;
   var AAuthenticated: Boolean) of object;
  TOnAfterUserLoginEvent = procedure(ASender: TIdFTPServerThread) of object;
  TOnDirectoryEvent = procedure(ASender: TIdFTPServerThread; var VDirectory: string) of object;
  TOnGetFileSizeEvent = procedure(ASender: TIdFTPServerThread; AFilename: string;
   var VFileSize: Int64) of object;
  TOnListDirectoryEvent = procedure(ASender: TIdFTPServerThread; APath: string;
   ADirectoryListing: TIdFTPListItems) of object;
  TOnFileEvent = procedure(ASender: TIdFTPServerThread; const APathname: string) of object;
  TOnRenameFileEvent = procedure(ASender: TIdFTPServerThread; ARenameFromFile,
   ARenameToFile: string) of object;
  TOnRetrieveFileEvent = procedure(ASender: TIdFTPServerThread; AFilename: string;
   var VStream: TStream) of object;
  TOnStoreFileEvent = procedure(ASender: TIdFTPServerThread; AFilename: string;
   AAppend: Boolean; var VStream: TStream) of object;

  TIdFTPUserType = (utNone, utAnonymousUser, utNormalUser);
  TIdFTPSystems = (ftpsOther, ftpsDOS, ftpsUNIX, ftpsVAX);
  TIdFTPOperation = (ftpRetr, ftpStor);

  TIdDataChannelThread = class(TIdThread)
  protected
    FControlChannel: TIdTCPServerConnection;
    FDataChannel: TIdTCPConnection;
    FErrorReply: TIdRFCReply;
    FFtpOperation: TIdFTPOperation;
    FOKReply: TIdRFCReply;
    //
    procedure Run; override;
    procedure SetErrorReply(const AValue: TIdRFCReply);
    procedure SetOKReply(const AValue: TIdRFCReply);
  public
    constructor Create(APASV: Boolean; AControlConnection: TIdTCPServerConnection); reintroduce;
    destructor Destroy; override;
    procedure StartThread(AOperation: TIdFTPOperation);
    procedure SetupDataChannel(const AIP: string; APort: Integer);
    //
    property OKReply: TIdRFCReply read FOKReply write SetOKReply;
    property ErrorReply: TIdRFCReply read FErrorReply write SetErrorReply;
  end;

  TIdFTPServerThread = class(TIdPeerThread)
  protected
    FUserType: TIdFTPUserType;
    FAuthenticated: Boolean;
    FALLOSize: Integer;
    FCurrentDir: string;
    FDataType: TIdFTPTransferType;
    FDataMode: TIdFTPTransferMode;
    FDataPort: Integer;
    FDataStruct: TIdFTPDataStructure;
    FDataChannelThread: TIdDataChannelThread;
    FHomeDir: string;
    FUsername: string;
    FPassword: string;
    FPASV: Boolean;
    FRESTPos: Integer;
    FRNFR: string;
    //
    procedure CreateDataChannel(APASV: Boolean = False);
    function IsAuthenticated(ASender: TIdCommand): Boolean;
    procedure KillDataChannel;
    procedure ReInitialize;
  public
    constructor Create(ACreateSuspended: Boolean = True); override;
    destructor Destroy; override;
    //
    property Authenticated: Boolean read FAuthenticated write FAuthenticated;
    property ALLOSize: Integer read FALLOSize write FALLOSize;
    property CurrentDir: string read FCurrentDir write FCurrentDir;
    property DataChannelThread: TIdDataChannelThread read FDataChannelThread
     write FDataChannelThread;
    property DataType: TIdFTPTransferType read FDataType write FDataType;
    property DataMode: TIdFTPTransferMode read FDataMode write FDataMode;
    property DataPort: Integer read FDataPort write FDataPort;
    property DataStruct: TIdFTPDataStructure read FDataStruct write FDataStruct;
    property HomeDir: string read FHomeDir write FHomeDir;
    property Password: string read FPassword write FPassword;
    property PASV: Boolean read FPASV write FPASV;
    property RESTPos: Integer read FRESTPos write FRESTPos;
    property Username: string read FUsername write FUsername;
    property UserType: TIdFTPUserType read FUserType write FUserType;
  end;

  TIdFTPServer = class;

  TIdOnGetCustomListFormat = procedure(ASender: TIdFTPServer; AItem: TIdFTPListItem;
   var VText: string) of object;

  { FTP Server }
  TIdFTPServer = class(TIdTCPServer)
  protected
    FAnonymousAccounts: TstringList;
    FAllowAnonymousLogin: Boolean;
    FAnonymousPassStrictCheck: Boolean;
    FCmdHandlerList: TIdCommandHandler;
    FCmdHandlerNlst: TIdCommandHandler;
    FEmulateSystem: TIdFTPSystems;
    FHelpReply: Tstrings;
    FSystemType: string;
    FUserAccounts: TIdUserManager;
    FOnAfterUserLogin: TOnAfterUserLoginEvent;
    FOnGetCustomListFormat: TIdOnGetCustomListFormat;
    FOnUserLogin: TOnUserLoginEvent;
    FOnChangeDirectory: TOnDirectoryEvent;
    FOnGetFileSize: TOnGetFileSizeEvent;
    FOnListDirectory: TOnListDirectoryEvent;
    FOnRenameFile: TOnRenameFileEvent;
    FOnDeleteFile: TOnFileEvent;
    FOnRetrieveFile: TOnRetrieveFileEvent;
    FOnStoreFile: TOnStoreFileEvent;
    FOnMakeDirectory: TOnDirectoryEvent;
    FOnRemoveDirectory: TOnDirectoryEvent;
    //Command replies
    procedure CommandUSER(ASender: TIdCommand);
    procedure CommandPASS(ASender: TIdCommand);
    procedure CommandCWD(ASender: TIdCommand);
    procedure CommandCDUP(ASender: TIdCommand);
    procedure CommandREIN(ASender: TIdCommand);
    procedure CommandPORT(ASender: TIdCommand);
    procedure CommandPASV(ASender: TIdCommand);
    procedure CommandTYPE(ASender: TIdCommand);
    procedure CommandSTRU(ASender: TIdCommand);
    procedure CommandMODE(ASender: TIdCommand);
    procedure CommandRETR(ASender: TIdCommand);
    procedure CommandSSAP(ASender: TIdCommand);
    procedure CommandALLO(ASender: TIdCommand);
    procedure CommandREST(ASender: TIdCommand);
    procedure CommandRNFR(ASender: TIdCommand);
    procedure CommandRNTO(ASender: TIdCommand);
    procedure CommandABOR(ASender: TIdCommand);
    procedure CommandDELE(ASender: TIdCommand);
    procedure CommandRMD(ASender: TIdCommand);
    procedure CommandMKD(ASender: TIdCommand);
    procedure CommandPWD(ASender: TIdCommand);
    procedure CommandLIST(ASender: TIdCommand);
    procedure CommandSITE(ASender: TIdCommand);
    procedure CommandSYST(ASender: TIdCommand);
    procedure CommandSTAT(ASender: TIdCommand);
    procedure CommandSIZE(ASender: TIdCommand);
    procedure CommandFEAT(ASender: TIdCommand);
    procedure CommandOPTS(ASender: TIdCommand);
    //
    procedure DoChangeDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
    procedure DoMakeDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
    procedure DoRemoveDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
    procedure DoGetCustomListFormat(AItem: TIdFTPListItem; var VText: string);
    procedure InitializeCommandHandlers; override;
    procedure ListDirectory(ASender: TIdFTPServerThread; ADirectory: string;
     var ADirContents: TstringList; ADetails: Boolean);
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SetAnonymousAccounts(const AValue: TstringList);
    procedure SetHelpReply(const AValue: Tstrings);
    procedure SetUserAccounts(const AValue: TIdUserManager);
    procedure SetEmulateSystem(const AValue: TIdFTPSystems);
    procedure ThreadException(ASender: TObject; AException: Exception);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property AllowAnonymousLogin: Boolean read FAllowAnonymousLogin write FAllowAnonymousLogin;
    property AnonymousAccounts: TStringList read FAnonymousAccounts write SetAnonymousAccounts;
    property AnonymousPassStrictCheck: Boolean read FAnonymousPassStrictCheck
     write FAnonymousPassStrictCheck;
    property EmulateSystem: TIdFTPSystems read FEmulateSystem write SetEmulateSystem;
    property HelpReply: Tstrings read FHelpReply write SetHelpReply;
    property UserAccounts: TIdUserManager read FUserAccounts write SetUserAccounts;
    property SystemType: string read FSystemType write FSystemType;
    property OnAfterUserLogin: TOnAfterUserLoginEvent read FOnAfterUserLogin
     write FOnAfterUserLogin;
    property OnChangeDirectory: TOnDirectoryEvent read FOnChangeDirectory write FOnChangeDirectory;
    property OnGetCustomListFormat: TIdOnGetCustomListFormat read FOnGetCustomListFormat
     write FOnGetCustomListFormat;
    property OnGetFileSize: TOnGetFileSizeEvent read FOnGetFileSize write FOnGetFileSize;
    property OnUserLogin: TOnUserLoginEvent read FOnUserLogin write FOnUserLogin;
    property OnListDirectory: TOnListDirectoryEvent read FOnListDirectory write FOnListDirectory;
    property OnRenameFile: TOnRenameFileEvent read FOnRenameFile write FOnRenameFile;
    property OnDeleteFile: TOnFileEvent read FOnDeleteFile write FOnDeleteFile;
    property OnRetrieveFile: TOnRetrieveFileEvent read FOnRetrieveFile write FOnRetrieveFile;
    property OnStoreFile: TOnStoreFileEvent read FOnStoreFile write FOnStoreFile;
    property OnMakeDirectory: TOnDirectoryEvent read FOnMakeDirectory write FOnMakeDirectory;
    property OnRemoveDirectory: TOnDirectoryEvent read FOnRemoveDirectory write FOnRemoveDirectory;
  end;

implementation

uses
  IdAssignedNumbers,
  IdGlobal,
  IdIOHandlerSocket,
  IdResourcestrings,
  IdSimpleServer,
  IdSocketHandle,
  Idstrings,
  IdTCPClient,
  IdEmailAddress;

{ TIdDataChannelThread }

constructor TIdDataChannelThread.Create(APASV: Boolean; AControlConnection: TIdTCPServerConnection);
begin
  inherited Create;
  StopMode := smSuspend;
  FOKReply := TIdRFCReply.Create;
  FErrorReply := TIdRFCReply.Create;
  FControlChannel := AControlConnection;
  if APASV then begin
    FDataChannel := TIdSimpleServer.Create(nil);
    TIdSimpleServer(FDataChannel).BoundIP := TIdIOHandlerSocket(FControlChannel.IOHandler).Binding.IP;
  end else begin
    FDataChannel := TIdTCPClient.Create(nil);
    TIdTCPClient(FDataChannel).BoundPort := 20;  //Default dataport
  end;
end;

destructor TIdDataChannelThread.Destroy;
begin
  FreeAndNil(FOKReply);
  FreeAndNil(FErrorReply);
  FreeAndNil(FDataChannel);
  inherited Destroy;
end;

procedure TIdDataChannelThread.StartThread(AOperation: TIdFTPOperation);
begin
  FFtpOperation := AOperation; try
    if FDataChannel is TIdSimpleServer then begin
      TIdSimpleServer(FDataChannel).Listen;
    end else if FDataChannel is TIdTCPClient then begin
      TIdTCPClient(FDataChannel).Connect;
    end;
  except
    FControlChannel.WriteRFCReply(FErrorReply); //426
    raise;
  end;
  inherited Start;
end;

procedure TIdDataChannelThread.Run;
var
  StrStream: TStringStream;
begin
  try
    try
      try
        try
          if Data is TStream then begin
            case FFtpOperation of
              ftpRetr: FDataChannel.WriteStream(TStream(Data));
              ftpStor: FDataChannel.ReadStream(TStream(Data), -1, True);
            end;
          end else begin
            case FFtpOperation of
              ftpRetr: FDataChannel.Writestrings(Data as Tstrings);
              ftpStor: begin
                StrStream := TStringStream.Create(''); try
                  FDataChannel.ReadStream(StrStream, -1, True);
                  TStrings(Data).Text := StrStream.Datastring;
                finally FreeAndNil(StrStream); end;
              end;
            end;
          end;
        finally
          Data.Free;
          Data := nil;
        end;
      finally FDataChannel.Disconnect; end;
      FControlChannel.WriteRFCReply(FOKReply); //226
    except
      FControlChannel.WriteRFCReply(FErrorReply); //426
    end;
  finally Stop; end;
end;

procedure TIdDataChannelThread.SetupDataChannel(const AIP: string; APort: Integer);
begin
  if FDataChannel is TIdSimpleServer then begin
    with TIdSimpleServer(FDataChannel) do begin
      BoundIP := AIP;
      BoundPort := APort;
    end;
  end else begin
    with TIdTCPClient(FDataChannel) do begin
      Host := AIP;
      Port := APort;
    end;
  end;
end;

procedure TIdDataChannelThread.SetErrorReply(const AValue: TIdRFCReply);
begin
  FErrorReply.Assign(AValue);
end;

procedure TIdDataChannelThread.SetOKReply(const AValue: TIdRFCReply);
begin
  FOKReply.Assign(AValue);
end;

{ TIdFTPClient }

constructor TIdFTPServerThread.Create(ACreateSuspended: Boolean = True);
begin
  inherited Create(ACreateSuspended);
  ReInitialize;
end;

destructor TIdFTPServerThread.Destroy;
begin
  if Assigned(FDataChannelThread) then begin
    FDataChannelThread.Terminate;
  end;
  inherited Destroy;
end;

procedure TIdFTPServerThread.CreateDataChannel(APASV: Boolean = False);
begin
  FDataChannelThread := TIdDataChannelThread.Create(APASV, Connection);
  FDataChannelThread.OnException := TIdFTPServer(FConnection.Server).ThreadException;
  FDataChannelThread.FreeOnTerminate := True;
end;

procedure TIdFTPServerThread.KillDataChannel;
begin
  with FDataChannelThread do try
    if not Stopped then begin
      FDataChannel.DisconnectSocket;
      WaitFor;
    end;
  except
    { absorb }
  end;
end;

procedure TIdFTPServerThread.ReInitialize;
begin
  UserType := utNone;
  FAuthenticated := False;
  FALLOSize := 0;
  FCurrentDir := '/';
  FDataType := ftASCII;
  FDataMode := dmStream;
  FDataPort := 20;
  FDataStruct := dsFile;
  FHomeDir := '';
  FUsername := '';
  FPassword := '';
  FPASV := False;
  FRESTPos := 0;
  FRNFR := '';
end;

function TIdFTPServerThread.IsAuthenticated(ASender: TIdCommand): Boolean;
begin
  if not FAuthenticated then begin
    ASender.Reply.SetReply(530, RSFTPUserNotLoggedIn);
  end else begin
    if Assigned(FDataChannelThread) then begin
      if not FDataChannelThread.Stopped and
       not AnsiSameText(ASender.CommandHandler.Command, 'ABOR') then begin
        Result := False;
        Exit;
      end;
    end;
  end;
  Result := FAuthenticated;
end;

{ TIdFTPServer }

constructor TIdFTPServer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FAnonymousAccounts :=  TstringList.Create;
  // By default these user names will be treated as anonymous.
  with FAnonymousAccounts do begin
    Add('anonymous'); { do not localize }
    Add('ftp'); { do not localize }
    Add('guest'); { do not localize }
  end;
  FAllowAnonymousLogin := False;
  FAnonymousPassStrictCheck := True;
  DefaultPort := IDPORT_FTP;
  FEmulateSystem := ftpsDOS;
  Greeting.NumericCode := 220;
  Greeting.Text.Text := RSFTPDefaultGreeting;
  FHelpReply := TstringList.Create;
  ThreadClass := TIdFTPServerThread;
  ReplyUnknownCommand.NumericCode := 500;
  ReplyUnknownCommand.Text.Text := RSFTPCmdSyntaxError;
  FUserAccounts := nil;
  FSystemType := 'Windows 9x/NT.';
end;

procedure TIdFTPServer.InitializeCommandHandlers;
begin
  { FTP Server Events }
  //ACCESS CONTROL COMMANDS
  //USER <SP> <username> <CRLF>
  with CommandHandlers.Add do begin
    Command := 'USER';
    OnCommand := CommandUSER;
  end;
  //PASS <SP> <password> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'PASS';
    OnCommand := CommandPASS;
  end;
  //ACCT <SP> <account-information> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'ACCT';
    ReplyNormal.NumericCode := 202;
    ReplyNormal.Text.Text := Format(RSFTPCmdNotImplemented, ['ACCT']);
  end;
  //CWD  <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'CWD';
    OnCommand := CommandCWD;
    ReplyExceptionCode := 550;
  end;
  //CDUP <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'CDUP';
    OnCommand := CommandCDUP;
    ReplyExceptionCode := 550;
  end;
  //SMNT <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'SMNT';
    ReplyNormal.NumericCode := 250;
    ReplyNormal.Text.Text := RSFTPFileActionCompleted;
  end;
  //QUIT <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'QUIT';
    Disconnect := True;
    ReplyNormal.NumericCode := 221;
    ReplyNormal.Text.Text := 'Goodbye.';
  end;
  //REIN <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'REIN';
    OnCommand := CommandREIN;
  end;
  //PORT <SP> <host-port> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'PORT';
    OnCommand := CommandPORT;
  end;
  //PASV <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'PASV';
    OnCommand := CommandPASV;
  end;
  //TYPE <SP> <type-code> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'TYPE';
    OnCommand := CommandTYPE;
  end;
  //STRU <SP> <structure-code> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'STRU';
    OnCommand := CommandSTRU;
  end;
  //MODE <SP> <mode-code> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'MODE';
    OnCommand := CommandMODE;
  end;
  //FTP SERVICE COMMANDS
  //RETR <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'RETR';
    OnCommand := CommandRETR;
    ReplyExceptionCode := 550;
  end;
  //STOR <SP> <pathname> <CRLF>
  with CommandHandlers.Add do begin
    Command := 'STOR';
    OnCommand := CommandSSAP;
    ReplyExceptionCode := 550;
  end;
  //STOU <CRLF>
  with CommandHandlers.Add do begin
    Command := 'STOU';
    OnCommand := CommandSSAP;
    ReplyExceptionCode := 550;
  end;
  //APPE <SP> <pathname> <CRLF>
  with CommandHandlers.Add do begin
    Command := 'APPE';
    OnCommand := CommandSSAP;
    ReplyExceptionCode := 550;
  end;
  //ALLO <SP> <decimal-integer>
  //    [<SP> R <SP> <decimal-integer>] <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'ALLO';
    OnCommand := CommandALLO;
  end;
  //REST <SP> <marker> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'REST';
    OnCommand := CommandREST;
  end;
  //RNFR <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'RNFR';
    OnCommand := CommandRNFR;
  end;
  //RNTO <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'RNTO';
    OnCommand := CommandRNTO;
  end;
  //ABOR <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'ABOR';
    OnCommand := CommandABOR;
  end;
  //DELE <SP> <pathname> <CRLF>
  with CommandHandlers.Add do begin
    Command := 'DELE';
    OnCommand := CommandDELE;
  end;
  //RMD  <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'RMD';
    OnCommand := CommandRMD;
  end;
  //MKD  <SP> <pathname> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'MKD';
    OnCommand := CommandMKD;
  end;
  //PWD  <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'PWD';
    OnCommand := CommandPWD;
  end;
  //LIST [<SP> <pathname>] <CRLF>
  FCmdHandlerList := CommandHandlers.Add;
  with FCmdHandlerList do begin
    Command := 'LIST';
    OnCommand := CommandLIST;
  end;
  //NLST [<SP> <pathname>] <CRLF>
  FCmdHandlerNlst := CommandHandlers.Add;
  with FCmdHandlerNlst do begin
    Command := 'NLST';
    OnCommand := CommandLIST;
  end;
  //SITE <SP> <string> <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'SITE';
    OnCommand := CommandSITE;
  end;
  //SYST <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'SYST';
    OnCommand := CommandSYST;
  end;
  //STAT [<SP> <pathname>] <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'STAT';
    OnCommand := CommandSTAT;
  end;
  //HELP [<SP> <string>] <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'HELP';
    ReplyNormal.NumericCode := 214;
    //
    if Length(FHelpReply.Text) <> 0 then
     ReplyNormal.Text := FHelpReply
    else
     ReplyNormal.Text.Text := 'HELP Command';
  end;
  //NOOP <CRLF>
  with CommandHandlers.Add do
  begin
    Command := 'NOOP';
    ReplyNormal.NumericCode := 200;
    ReplyNormal.Text.Text := Format(RSFTPCmdSuccessful, ['NOOP']);
  end;
  with CommandHandlers.Add do
  begin
    Command := 'XMKD';
    OnCommand := CommandMKD;
  end;
  with CommandHandlers.Add do
  begin
    Command := 'XRMD';
    OnCommand := CommandRMD;
  end;
  with CommandHandlers.Add do
  begin
    Command := 'XPWD';
    OnCommand := CommandPWD;
  end;
  with CommandHandlers.Add do
  begin
    Command := 'XCUP';
    OnCommand := CommandCDUP;
  end;
  with CommandHandlers.Add do
  begin
    Command := 'FEAT';
    OnCommand := CommandFEAT;
  end;
  //TODO: OPTS - what is this for? Cannot find in RFC 959
  with CommandHandlers.Add do begin
    Command := 'OPTS';
    OnCommand := CommandOPTS;
  end;
  //SIZE [<FILE>] CRLF
  with CommandHandlers.Add do
  begin
    Command := 'SIZE';
    OnCommand := CommandSIZE;
  end;
end;

destructor TIdFTPServer.Destroy;
begin
  FreeAndNil(FAnonymousAccounts);
  FreeAndNil(FHelpReply);
  inherited Destroy;
end;

procedure TIdFTPServer.ListDirectory(ASender: TIdFTPServerThread; ADirectory: string;
 var ADirContents: TstringList; ADetails: Boolean);
var
  i: Integer;
  LDirectoryList: TIdFTPListItems;
  LPathSep: string;
begin
  if Assigned(FOnListDirectory) then begin
    LDirectoryList := TIdFTPListItems.Create; try
      LPathSep := '/';
      // Emulated System
      case FEmulateSystem of
        ftpsOther: begin
          if Assigned(OnGetCustomListFormat) then begin
            LDirectoryList.ListFormat := flfCustom;
            LDirectoryList.OnGetCustomListFormat := DoGetCustomListFormat;
          end else begin
            LDirectoryList.ListFormat := flfNone;
          end;
        end;
        ftpsDOS: begin
          LDirectoryList.ListFormat := flfDos;
          LPathSep := '\';
        end;
        ftpsUNIX: begin
          LDirectoryList.ListFormat := flfUnix;
        end;
        ftpsVAX: begin
          LDirectoryList.ListFormat := flfVax;
        end;
      end;
      if Copy(ADirectory, Length(LPathSep), 1) <> LPathSep then begin
        ADirectory := ADirectory + LPathSep;
      end;
      // Event
      FOnListDirectory(ASender, ADirectory, LDirectoryList);

      for i := 0 to LDirectoryList.Count - 1 do begin
        if ADetails then begin
          ADirContents.Add(LDirectoryList.Items[i].Text);
        end else begin
          ADirContents.Add(LDirectoryList.Items[i].Filename);
        end;
      end;
    finally FreeAndNil(LDirectoryList); end;
  end else begin
    raise EIdTFTPException.Create('No OnListDirectory event found!');
  end;
end;

procedure TIdFTPServer.SetHelpReply(const AValue: Tstrings);
begin
  FHelpReply.Assign(AValue);
end;

procedure TIdFTPServer.SetUserAccounts(const AValue: TIdUserManager);
begin
  FUserAccounts := AValue;
  if Assigned(FUserAccounts) then
  begin
    FUserAccounts.FreeNotification(Self);
  end;
end;

procedure TIdFTPServer.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FUserAccounts) then
    FUserAccounts := nil;
end;

procedure TIdFTPServer.SetAnonymousAccounts(const AValue: TstringList);
begin
  if Assigned(AValue) then
  begin
    FAnonymousAccounts.Assign(AValue);
  end;
end;

procedure TIdFTPServer.SetEmulateSystem(const AValue: TIdFTPSystems);
begin
  if AnsiSameText(FSystemType, 'Windows 9x/NT.') or AnsiSameText(FSystemType, 'UNIX type: L8.') then
  begin
    case AValue of
      ftpsDOS: FSystemType := 'Windows 9x/NT.';
      ftpsUNIX,
      ftpsVAX: FSystemType := 'UNIX type: L8.';
    end;
  end;
  FEmulateSystem := AValue;
end;

procedure TIdFTPServer.ThreadException(ASender: TObject;
  AException: Exception);
begin
  ShowException(AException, nil);
end;

//Command Replies/Handling
procedure TIdFTPServer.CommandUSER(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if (FAnonymousAccounts.IndexOf(Lowercase(ASender.UnparsedParams)) >= 0)
     and (AllowAnonymousLogin) then begin
      UserType := utAnonymousUser;
      FUsername := ASender.UnparsedParams;
      ASender.Reply.SetReply(331, RSFTPAnonymousUserOkay);
    end else begin
      UserType := utNormalUser;
      if Length(ASender.UnparsedParams) > 0 then begin
        FUsername := ASender.UnparsedParams;
        ASender.Reply.SetReply(331, RSFTPUserOkay);
      end else begin
        ASender.Reply.SetReply(332, RSFTPNeedAccountForLogin);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandPASS(ASender: TIdCommand);
var
  LValidated: Boolean;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    case FUserType of
      utAnonymousUser: begin
        LValidated := Length(ASender.UnparsedParams) > 0;
        if (FAnonymousPassStrictCheck) and (LValidated) then begin
          LValidated := False;
          if FindFirstOf('@.', ASender.UnparsedParams) > 0 then begin
            LValidated := True;
          end;
        end;
        if LValidated then begin
          FAuthenticated := True;
          FPassword := ASender.UnparsedParams;
          ASender.Reply.SetReply(230, RSFTPAnonymousUserLogged);
        end else begin
          FUserType := utNone;
          FAuthenticated := False;
          FPassword := '';
          ASender.Reply.SetReply(530, RSFTPUserNotLoggedIn);
        end;
      end;
      utNormalUser: begin
        if Assigned(FUserAccounts) then begin
          FAuthenticated := FUserAccounts.AuthenticateUser(FUsername, ASender.UnparsedParams);
          if FAuthenticated then begin
            FPassword := ASender.UnparsedParams;
            ASender.Reply.SetReply(230, RSFTPUserLogged);
          end else begin
            FPassword := '';
            ASender.Reply.SetReply(530, RSFTPUserNotLoggedIn);
          end;
        end else begin
          if Assigned(FOnUserLogin) then begin
            LValidated := False;
            FOnUserLogin(TIdFTPServerThread(ASender.Thread), FUsername, ASender.UnparsedParams
             , LValidated);
            FAuthenticated := LValidated;
            if LValidated then begin
              FPassword := ASender.UnparsedParams;
              ASender.Reply.SetReply(230, RSFTPUserLogged);
            end else begin
              FPassword := '';
              ASender.Reply.SetReply(530, RSFTPUserNotLoggedIn);
            end;
          end;
        end;
      end else begin
        ASender.Reply.SetReply(503, RSFTPNeedLoginWithUser);
      end;
    end;
  end;
  //After login
  if TIdFTPServerThread(ASender.Thread).FAuthenticated and Assigned(FOnAfterUserLogin) then begin
    FOnAfterUserLogin(TIdFTPServerThread(ASender.Thread));
  end;
end;

procedure TIdFTPServer.CommandCWD(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      if Assigned(OnChangeDirectory) then begin
        case FEmulateSystem of
          ftpsDOS: s := ProcessPath(FCurrentDir, ASender.UnparsedParams, '\');
          ftpsOther, ftpsUNIX, ftpsVAX: s := ProcessPath(FCurrentDir, ASender.UnparsedParams);
        end;
        DoChangeDirectory(TIdFTPServerThread(ASender.Thread), s);
        ASender.Reply.SetReply(250, Format(RSFTPCmdSuccessful, ['CWD']));
        FCurrentDir := s;
      end else begin
        ASender.Reply.SetReply(550, Format(RSFTPCmdNotImplemented, ['CWD']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandCDUP(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      case FEmulateSystem of
        ftpsDOS: s := '..\';
        ftpsOther, ftpsUNIX, ftpsVAX: s := '../';
      end;
      if Assigned(FOnChangeDirectory) then begin
        DoChangeDirectory(TIdFTPServerThread(ASender.Thread), s);
        FCurrentDir := s;
        ASender.Reply.SetReply(212, Format(RSFTPCurrentDirectoryIs, [FCurrentDir]));
      end else begin
        ASender.Reply.SetReply(550, Format(RSFTPCmdNotImplemented, ['CWD']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandREIN(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      ReInitialize;
      ASender.Reply.SetReply(220, RSFTPServiceOpen);
    end;
  end;
end;

procedure TIdFTPServer.CommandPORT(ASender: TIdCommand);
var
  LLo, LHi: Integer;
  LParm, IP: string;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      FPASV := False;
      LParm := ASender.UnparsedParams;
      IP := '';
      { h1 }
      IP := IP + Fetch(LParm, ',') + '.';
      { h2 }
      IP := IP + Fetch(LParm, ',') + '.';
      { h3 }
      IP := IP + Fetch(LParm, ',') + '.';
      { h4 }
      IP := IP + Fetch(LParm, ',');
      { p1 }
      LLo := StrToInt(Fetch(LParm, ','));
      { p2 }
      LHi := StrToInt(LParm);
      FDataPort := (LLo * 256) + LHi;
      CreateDataChannel(False);
      FDataChannelThread.SetupDataChannel(IP, FDataPort);
      ASender.Reply.SetReply(200, Format(RSFTPCmdSuccessful, ['PORT']));
    end;
  end;
end;

procedure TIdFTPServer.CommandPASV(ASender: TIdCommand);
var
  LParam: string;
  LBPort: Word;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      CreateDataChannel(True);
      FDataChannelThread.SetupDataChannel(TIdIOHandlerSocket(Connection.IOHandler).Binding.IP
       , FDataPort);
      with TIdSimpleServer(FDataChannelThread.FDataChannel) do begin
        BeginListen;
        LBPort := Binding.Port;
        LParam := stringReplace(BoundIP, '.', ',', [rfReplaceAll]);
        LParam := LParam + ',' + IntToStr(LBPort div 256) + ',' + IntToStr(LBPort mod 256);

        ASender.Reply.SetReply(227, Format(RSFTPPassiveMode, [LParam]));
        FPASV := True;
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandTYPE(ASender: TIdCommand);
var
  LType: Char;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      if Length(ASender.UnparsedParams) = 1 then
      begin
        //Default data type is ASCII
        LType := Uppercase(ASender.UnparsedParams)[1];
        case LType of
          'A': FDataType := ftASCII;
          'I': FDataType := ftBinary;
        end;
        if FDataType in  [ftASCII, ftBinary] then
        begin
          ASender.Reply.SetReply(200, Format(RSFTPTYPEChanged, [LType]));
        end;
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandSTRU(ASender: TIdCommand);
var
  LDataStruct: Char;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      if Length(ASender.UnparsedParams) = 1 then
      begin
        //Default structure is file
        LDataStruct := Uppercase(ASender.UnparsedParams)[1];
        case LDataStruct of
          'F': FDataStruct := dsFile;
          'R': FDataStruct := dsRecord;
          'P': FDataStruct := dsPage;
        end;
        if FDataStruct in [dsFile, dsRecord, dsPage] then
        begin
          ASender.Reply.SetReply(200, Format(RSFTPSTRUChanged, [LDataStruct]));
        end;
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandMODE(ASender: TIdCommand);
var
  LMode: Char;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      if Length(ASender.UnparsedParams) = 1 then
      begin
        //Default data mode is stream
        LMode := Uppercase(ASender.UnparsedParams)[1];
        case LMode of
          'B': FDataMode := dmBlock;
          'C': FDataMode := dmCompressed;
          'S': FDataMode := dmStream;
        end;
        if FDataMode in [dmBlock, dmCompressed, dmStream] then
        begin
          ASender.Reply.SetReply(200, Format(RSFTPMODEChanged, [LMode]));
        end;
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandRETR(ASender: TIdCommand);
var
  s: string;
  LStream: TStream;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      //TODO: Fix reference to /
      s := ProcessPath(CurrentDir, ASender.UnparsedParams, '/');
      if Assigned(FOnRetrieveFile) then begin
        LStream := nil;
        FOnRetrieveFile(TIdFTPServerThread(ASender.Thread), s, LStream);
        if Assigned(LStream) then begin
          LStream.Position := FRESTPos;
          FRESTPos := 0;
          FDataChannelThread.Data := LStream;
          FDataChannelThread.OKReply.SetReply(226, RSFTPDataConnClosed);
          FDataChannelThread.ErrorReply.SetReply(426, RSFTPDataConnClosedAbnormally);
          FDataChannelThread.StartThread(ftpRetr);
          ASender.Reply.SetReply(150, RSFTPDataConnToOpen);
        end else begin
          ASender.Reply.SetReply(550, RSFTPFileActionAborted);
        end;
      end else begin
        // TODO: RETR not implemented
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandSSAP(ASender: TIdCommand);
var
  LStream: TStream;
  LTmp1: string;
  LAppend: Boolean;
  Reply: TIdRFCReply;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      if AnsiSameText(ASender.CommandHandler.Command, 'STOU') then begin
        //TODO: Find a better method of finding unique names
        RandSeed := 9944;
        Randomize;
        LTmp1 := 'Tmp' + IntToStr(Random(192));
      end else begin
        LTmp1 := ASender.UnparsedParams;
      end;
      //
      LTmp1 := ProcessPath(FCurrentDir, LTmp1);
      LAppend := AnsiSameText(ASender.CommandHandler.Command, 'APPE');
      //
      if Assigned(FOnStoreFile) then begin
        LStream := nil;
        FOnStoreFile(TIdFTPServerThread(ASender.Thread), LTmp1, LAppend, LStream);
        if Assigned(LStream) then begin
          //Issued previously by ALLO cmd
          if FALLOSize > 0 then begin
            LStream.Size := FALLOSize;
          end;
          if LAppend then begin
            LStream.Position := LStream.Size;
          end else begin
            LStream.Position := 0;
          end;
          { Data transfer }
          try
            Reply := TIdRFCReply.Create;
            FDataChannelThread.Data := LStream;
            Reply.SetReply(226, RSFTPDataConnClosed);
            FDataChannelThread.OKReply := Reply;
            Reply.SetReply(426, RSFTPDataConnClosedAbnormally);
            FDataChannelThread.ErrorReply := Reply;
            FDataChannelThread.StartThread(ftpStor);
            ASender.Reply.SetReply(150, RSFTPDataConnToOpen);
            ASender.SendReply;
          finally FreeAndNil(Reply); end;
        end else begin
          ASender.Reply.SetReply(550, RSFTPFileActionAborted);
        end;
      end else begin
        ASender.Reply.SetReply(550, RSFTPCmdNotImplemented);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandALLO(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      s := Uppercase(ASender.UnparsedParams);
      case s[1] of
        'R':
           begin
             if s[2] = #32 then begin
               FALLOSize := StrToIntDef(Copy(s, 2, Length(s) - 2), 0);
             end;
           end;
      else
        FALLOSize := StrToIntDef(ASender.UnparsedParams, 0);
      end;
      ASender.Reply.SetReply(200, Format(RSFTPCmdSuccessful, ['ALLO']));
    end;
  end;
end;

procedure TIdFTPServer.CommandREST(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      FRESTPos := StrToIntDef(ASender.UnparsedParams, 0);
      ASender.Reply.SetReply(350, RSFTPFileActionPending);
    end;
  end;
end;

procedure TIdFTPServer.CommandRNFR(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      s := ASender.UnparsedParams;
      if Assigned(FOnRenameFile) then
      begin
        ASender.Reply.SetReply(350, RSFTPFileActionPending);
        FRNFR := s;
      end
      else
      begin
        ASender.Reply.SetReply(350, RSFTPFileActionPending);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandRNTO(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      s := ASender.UnparsedParams;
      if Assigned(FOnRenameFile) then
      begin
        try
          FOnRenameFile(TIdFTPServerThread(ASender.Thread), FRNFR, s);
        except
          ASender.Reply.NumericCode := 550;
          raise;
        end;
      end
      else
      begin
        ASender.Reply.SetReply(550, RSFTPFileActionNotTaken);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandABOR(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      if not FDataChannelThread.Stopped then begin
        KillDataChannel;
        ASender.Reply.SetReply(226, RSFTPDataConnClosed);
      end else begin
        ASender.Reply.SetReply(200, Format(RSFTPCmdSuccessful, ['ABOR']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandDELE(ASender: TIdCommand);
(*
DELE <SP> <pathname> <CRLF>
  250 Requested file action okay, completed.
  450 Requested file action not taken. - File is busy
  550 Requested action not taken. - File unavailable, no access permitted, etc
  500 Syntax error, command unrecognized.
  501 Syntax error in parameters or arguments.
  502 Command not implemented.
  421 Service not available, closing control connection. - During server shutdown, etc
  530 Not logged in.
*)
//TODO: Need to set replies when not authenticated and set replynormal to 250
// do for all procs, list valid replies in comments. Or maybe default is 550
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      if Assigned(FOnDeleteFile) then begin
        FOnDeleteFile(TIdFTPServerThread(ASender.Thread), ASender.UnparsedParams);
        ASender.Reply.SetReply(250, RSFTPFileActionCompleted);
      end else begin
        ASender.Reply.SetReply(550, RSFTPFileActionNotTaken);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandRMD(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      s := ProcessPath(FCurrentDir, ASender.UnparsedParams);
      if Assigned(FOnRemoveDirectory) then begin
        DoRemoveDirectory(TIdFTPServerThread(ASender.Thread), s);
        ASender.Reply.SetReply(250, RSFTPFileActionCompleted);
      end else begin
        ASender.Reply.SetReply(550, Format(RSFTPCmdNotImplemented, ['RMD']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandMKD(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      ProcessPath(FCurrentDir, ASender.UnparsedParams);
      if Assigned(FOnMakeDirectory) then
      begin
        FOnMakeDirectory(TIdFTPServerThread(ASender.Thread), s);
        ASender.Reply.SetReply(250, RSFTPFileActionCompleted);
      end
      else
      begin
        ASender.Reply.SetReply(550, Format(RSFTPCmdNotImplemented, ['MKD']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandPWD(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      ASender.Reply.SetReply(257, Format(RSFTPCurrentDirectoryIs, [FCurrentDir]));
    end;
  end;
end;

procedure TIdFTPServer.CommandLIST(ASender: TIdCommand);
var
  LStream: TstringList;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      LStream := TstringList.Create;
      try
        ListDirectory(TIdFTPServerThread(ASender.Thread), ProcessPath(FCurrentDir
         , ASender.UnparsedParams), LStream, ASender.CommandHandler = FCmdHandlerList);
      finally
        FDataChannelThread.Data := LStream;
        FDataChannelThread.OKReply.SetReply(226, RSFTPDataConnClosed);
        FDataChannelThread.ErrorReply.SetReply(426, RSFTPDataConnClosedAbnormally);
        ASender.Reply.SetReply(125, RSFTPDataConnToOpen);
        FDataChannelThread.StartThread(ftpRetr);
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandSITE(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do
  begin
    if IsAuthenticated(ASender) then
    begin
      s := Uppercase(ASender.UnparsedParams);
      if AnsiSameText(s, 'HELP') then
      begin
        ASender.Reply.SetReply(214, RSFTPSITECmdsSupported);
      end
      else
      begin
        case FEmulateSystem of
          ftpsDOS: ASender.Reply.SetReply(214, Format(RSFTPDirectorySTRU, ['MS-DOS']));
          ftpsUNIX: ASender.Reply.SetReply(214, Format(RSFTPDirectorySTRU, ['UNIX']));
          ftpsVAX: ASender.Reply.SetReply(214, Format(RSFTPDirectorySTRU, ['VAX/VMS']));
        end;
      end;
    end;
  end;
end;

procedure TIdFTPServer.CommandSYST(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      ASender.Reply.SetReply(215, FSystemType);
    end;
  end;
end;

procedure TIdFTPServer.CommandSTAT(ASender: TIdCommand);
var
  LStream: TstringList;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      if not FDataChannelThread.Suspended then begin
        ASender.Reply.SetReply(211, RSFTPOpenDataConn);
      end;
      //else act as LIST command without a data channel
      ASender.Reply.SetReply(211, RSFTPDataConnToOpen);
      ASender.SendReply;
      LStream := TStringList.Create; try
        ListDirectory(TIdFTPServerThread(ASender.Thread), ProcessPath(FCurrentDir
         , ASender.UnparsedParams), LStream, True);
      finally
        Connection.Writestrings(LStream);
        FreeAndNil(LStream);
      end;
      ASender.Reply.SetReply(211, RSFTPCmdEndOfStat);
    end;
  end;
end;

procedure TIdFTPServer.CommandFEAT(ASender: TIdCommand);
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      ASender.Reply.SetReply(211, RSFTPCmdExtsSupported);
    end;
  end;
end;

procedure TIdFTPServer.CommandOPTS(ASender: TIdCommand);
var
  s: string;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then begin
      //TODO: Actually call event
      s := ASender.UnparsedParams;
      ASender.Reply.SetReply(202, Format(RSFTPCmdNotImplemented, ['OPTS']));
    end;
  end;
end;

procedure TIdFTPServer.CommandSIZE(ASender: TIdCommand);
var
  s: string;
  LSize: Int64;
begin
  with TIdFTPServerThread(ASender.Thread) do begin
    if IsAuthenticated(ASender) then
    begin
      ProcessPath(FCurrentDir, ASender.UnparsedParams);
      if Assigned(FOnGetFileSize) then
      begin
        try
          LSize := -1;
          FOnGetFileSize(TIdFTPServerThread(ASender.Thread), s, LSize);
          if LSize > -1 then begin
            ASender.Reply.SetReply(213, IntToStr(LSize));
          end else begin
            ASender.Reply.SetReply(550, RSFTPFileActionAborted);
          end;
        except
          ASender.Reply.NumericCode := 550;
          raise;
        end;
      end else begin
        ASender.Reply.SetReply(550, Format(RSFTPCmdNotImplemented, ['SIZE']));
      end;
    end;
  end;
end;

procedure TIdFTPServer.DoGetCustomListFormat(AItem: TIdFTPListItem; var VText: string);
begin
  if Assigned(OnGetCustomListFormat) then begin
    OnGetCustomListFormat(Self, AItem, VText);
  end;
end;

procedure TIdFTPServer.DoChangeDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
begin
  if Assigned(FOnChangeDirectory) then begin
    FOnChangeDirectory(AThread, VDirectory);
  end;
end;

procedure TIdFTPServer.DoRemoveDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
begin
  if Assigned(FOnRemoveDirectory) then begin
    FOnRemoveDirectory(AThread, VDirectory);
  end;
end;

procedure TIdFTPServer.DoMakeDirectory(AThread: TIdFTPServerThread; var VDirectory: string);
begin
  if Assigned(FOnMakeDirectory) then begin
    FOnMakeDirectory(AThread, VDirectory);
  end;
end;

end.
