unit IdFTPList;

{
 - Fixes as per user request for parsing non-detailed lists (SP).
   [Added flfNoDetails list format].

Initial version by
  D. Siders
  Integral Systems
  October 2000

Additions and extensions
  Doychin Bondzhev (doychin@dsoft-bg.com)
  dSoft-Bulgaria

  February 2001
  - TFTPListItem now descends from TCollectionItem
  - TFTPList now descends from TCollection
  Jun 2001
  - Fixes in UNIX format parser
  Aug 2001
  - It is now used in the FTP server component
}

interface

uses
  Classes, SysUtils, IdException, IdResourceStrings, IdGlobal;

{ Indy TIdFtp extensions to support automatic parsing of FTP directory listings }

type
  EIdInvalidFTPListingFormat = class(EIdException);

  // TFTPListFormat directory listing format.  flfNone, flfUnknown, flfCustom are not parsed
  TIdFTPListFormat = (flfNone, flfDos, flfUnix, flfVax, flfNoDetails, flfUnknown, flfCustom);
  TIdDirItemType = (ditDirectory, ditFile, ditSymbolicLink);

  TIdFTPListItems = class;

  // TIdFTPListItem stores an item in the FTP directory listing
  TIdFTPListItem = class(TCollectionItem)
  protected
    FSize: Int64;
    FItemCount: Integer;
    FData: string;
    FFileName: string;
    FGroupPermissions: string;
    FGroupName: string;
    FOwnerPermissions: string;
    FOwnerName: string;
    FUserPermissions: string;
    FModifiedDate: TDateTime;
    FLinkedItemName : string;
    FItemType: TIdDirItemType;
    //
    function DoGetCustomListFormat: string;
  public
    procedure Assign(Source: TPersistent); override;
    constructor Create(AOwner: TCollection); override;
    function Text: string;
    //
    property Data: string read FData write FData;
    property OwnerPermissions: string read FOwnerPermissions write FOwnerPermissions;
    property GroupPermissions: string read FGroupPermissions write FGroupPermissions;
    property UserPermissions: string read FUserPermissions write FUserPermissions;
    property ItemCount: Integer read FItemCount write FItemCount;
    property OwnerName: string read FOwnerName write FOwnerName;
    property GroupName: string read FGroupName write FGroupName;
    property Size: Int64 read FSize write FSize;
    property ModifiedDate: TDateTime read FModifiedDate write FModifiedDate;
    property FileName: string read FFileName write FFileName;
    property ItemType: TIdDirItemType read FItemType write FItemType;
    property LinkedItemName: string read FLinkedItemName write FLinkedItemName;
  end;

  TIdOnGetCustomListFormat = procedure(AItem: TIdFTPListItem; var VText: string) of object;
  TIdOnParseCustomListFormat = procedure(AItem: TIdFTPListItem) of object;

  // TFTPList is the container and parser for items in the directory listing
  TIdFTPListItems = class(TCollection)
  protected
    FDirectoryName: string;
    //
    procedure SetDirectoryName(const AValue: string);
  protected
    FOnGetCustomListFormat: TIdOnGetCustomListFormat;
    FOnParseCustomListFormat: TIdOnParseCustomListFormat;
    FListFormat: TIdFTPListFormat;
    //
    function GetItems(AIndex: Integer): TIdFTPListItem;
    procedure ParseDOS(AItem: TIdFTPListItem);
    procedure ParseUnix(AItem: TIdFTPListItem);
    procedure ParseVax(AItem: TIdFTPListItem);
    procedure SetItems(AIndex: Integer; const Value: TIdFTPListItem);
  public
    function Add: TIdFTPListItem;
    function CheckListFormat(Data: string; const ADetails: Boolean = False): TIdFTPListFormat; virtual;
    constructor Create; overload;
    function IndexOf(AItem: TIdFTPListItem): Integer;
    procedure LoadList(AData: TStrings);
    procedure Parse(ListFormat: TIdFTPListFormat; AItem: TIdFTPListItem);
    procedure ParseUnknown(AItem: TIdFTPListItem);
    procedure ParseCustom(AItem: TIdFTPListItem); virtual;
    //
    property DirectoryName: string read FDirectoryName write SetDirectoryName;
    property Items[AIndex: Integer]: TIdFTPListItem read GetItems write SetItems; default;
    property ListFormat: TIdFTPListFormat read FListFormat write FListFormat;
    property OnGetCustomListFormat: TIdOnGetCustomListFormat read FOnGetCustomListFormat
     write FOnGetCustomListFormat;
    property OnParseCustomListFormat: TIdOnParseCustomListFormat read FOnParseCustomListFormat
     write FOnParseCustomListFormat;
  end;

implementation

{ TFTPListItem }

constructor TIdFTPListItem.Create(AOwner: TCollection);
begin
  inherited Create(AOwner);
  Data := '';
  FItemType := ditFile;
  OwnerPermissions := '???';
  GroupPermissions := '???';
  UserPermissions := '???';
  ItemCount := 0;
  OwnerName := '????????';
  GroupName := '????????';
  Size := 0;
  ModifiedDate := 0.0;
  FileName := '';
  LinkedItemName := '';
end;

procedure TIdFTPListItem.Assign(Source: TPersistent);
Var
  Item: TIdFTPListItem;
begin
  Item := TIdFTPListItem(Source);
  Data := Item.Data;
  ItemType := Item.ItemType;
  OwnerPermissions := Item.OwnerPermissions;
  GroupPermissions := Item.GroupPermissions;
  UserPermissions := Item.UserPermissions;
  ItemCount := Item.ItemCount;
  OwnerName := Item.OwnerName;
  GroupName := Item.GroupName;
  Size := Item.Size;
  ModifiedDate := Item.ModifiedDate;
  FileName := Item.FileName;
  LinkedItemName := Item.LinkedItemName;
end;

{ TFTPList }

constructor TIdFTPListItems.Create;
begin
  inherited Create(TIdFTPListItem);
  ListFormat := flfUnix;
end;

function TIdFTPListItems.Add: TIdFTPListItem;
begin
  Result := TIdFTPListItem(inherited Add);
end;

procedure TIdFTPListItems.LoadList(AData: TStrings);
var
  iCtr: Integer;
  LStartLine: Integer;
  AItem: TIdFTPListItem;
begin
  Clear;
  // Some Unix ftp servers retunr 'total' in the first line of the directory listing
  if (FListFormat = flfUnix) and (AData.Count > 0) and
    (IndyPos('TOTAL', UpperCase(AData.Strings[0])) = 1) then begin
    LStartLine := 1;
  end
  else LStartLine := 0;
  for iCtr := LStartLine to AData.Count - 1 do
    if Length(Trim(AData.Strings[iCtr])) > 0 then begin
      AItem := Add;
      AItem.Data := AData.Strings[iCtr];
      try
        if (ListFormat <> flfNone) then begin
          Parse(ListFormat, AItem);
        end;
      except
        {on E: Exception do
          raise EIdException.Create('Please report this exception into Indy Bug list.' + #13 +
            E.Message + #13 + AItem.Data);}
         // When We don't know the exact listing type we will just ignore it and nothing will happen   
         Clear;
      end;
    end;
end;

function TIdFTPListItems.CheckListFormat(Data: string; const ADetails: Boolean = false): TIdFTPListFormat;
  function IsUnixItem(SData: string): Boolean;
  begin
    result := (SData[1] in ['L', 'D', '-']) and
    (SData[2] in ['R','W','X','-']) and
    (SData[3] in ['R','W','X','-']) and
    (SData[4] in ['R','W','X','-']) and
    (SData[5] in ['R','W','X','-']) and
    (SData[6] in ['R','W','X','-']) and
    (SData[7] in ['R','W','X','-']) and
    (SData[8] in ['R','W','X','-']) and
    (SData[9] in ['R','W','X','-']) and
    (SData[10] in ['R','W','X','-']);
  end;

var
  sData: string;
  sDir: string;
  sSize: string;
begin
  Result := flfUnknown;
  if ADetails then
  begin
    SData := UpperCase(Data);

    if IsUnixItem(SData) or (Pos('TOTAL', SData) = 1) then
    begin
      Result := flfUnix;
    end
    else
    begin
      if (IndyPos('DSK:', SData) <> 0) then
      begin
        Result := flfVax;
      end
      else
      begin
        sDir := Trim(Copy(SData, 25, 6));
        sSize := StringReplace(Trim(Copy(SData, 31, 8)), ',', '', [rfReplaceAll]);

        if ((SData[3] in ['/', '-']) and (SData[6] in ['/', '-'])) and ((sDir = '<DIR>') or ((sDir = '') and
            (StrToInt64Def(sSize, -1) <> -1))) then
        begin
          Result := flfDos;
        end;
      end;
    end;
  end
  else
  begin
    Result := flfNoDetails;
  end;
end;

function TIdFTPListItems.GetItems(AIndex: Integer): TIdFTPListItem;
begin
  Result := TIdFTPListItem(inherited Items[AIndex]);
end;

function TIdFTPListItems.IndexOf(AItem: TIdFTPListItem): Integer;
Var
  i: Integer;
begin
  result := -1;
  for i := 0 to Count - 1 do 
    if AItem = Items[i] then begin
      result := i;
      break;
    end;
end;

procedure TIdFTPListItems.Parse(ListFormat: TIdFTPListFormat; AItem: TIdFTPListItem);
begin
  case ListFormat of
    //flfNone - Data unchanged
    flfDos: ParseDos(AItem);
    flfUnix: ParseUnix(AItem);
    flfVax: ParseVax(AItem);
    flfNoDetails: AItem.FileName := Trim(AItem.Data);
    flfCustom: ParseCustom(AItem);
    flfUnknown: ParseUnknown(AItem);
  end;
end;

procedure TIdFTPListItems.ParseDOS(AItem: TIdFTPListItem);
var
  LModified: string;
  LTime: string;
  LDir: string;
  LSize: string;
  LName: string;
  LValue: string;
  LBuffer: string;
  LDateSeparator: Char;
  LTimeSeparator: Char;
  LShortTimeFormat: string;
  LShortDateFormat: string;
begin
  LModified := Copy(AItem.Data, 1, 2) + '/' + Copy(AItem.Data, 4, 2) + '/' +
    Copy(AItem.Data, 7, 2) + ' ';

  LBuffer := Trim(Copy(AItem.Data, 9, Length(AItem.Data)));

  // Scan time info
  LTime := Fetch(LBuffer);

  // Scan optional letter in a[m]/p[m]
  LModified := LModified + LTime;

  LBuffer := Trim(LBuffer);

  LDir := '';
  LSize := '';

  // Scan file size or dir marker
  LValue := Fetch(LBuffer);

  // Strip commas or StrToInt64Def will barf
  if (IndyPos(',', LValue) <> 0) then
    LValue := StringReplace(LValue, ',', '', [rfReplaceAll]);

  // What did we get?
  if (UpperCase(LValue) = '<DIR>') then
    LDir := '<DIR>'
  else
    LSize := LValue;

  // Rest of the buffer is item name
  LName := Trim(LBuffer);

  if LDir = '<DIR>' then begin
    AItem.ItemType := ditDirectory;
  end else begin
    AItem.ItemType := ditFile;
  end;

  AItem.Size := StrToInt64Def(LSize, 0);
  AItem.FileName := LName;

  // Convert modified to date time
  try
    // preserve the current locale settings
    LShortDateFormat := ShortDateFormat;
    LDateSeparator := DateSeparator;
    LTimeSeparator := TimeSeparator;
    LShortTimeFormat := ShortTimeFormat;
    DateSeparator := '/';
    ShortDateFormat := 'mm/dd/yyyy';
    TimeSeparator := ':';
    ShortTimeFormat := 'hh:mm';

    AItem.ModifiedDate := StrToDateTime(LModified);
    ShortDateFormat := LShortDateFormat;
    DateSeparator := LDateSeparator;
    TimeSeparator := LTimeSeparator;
    ShortTimeFormat := LShortTimeFormat;
  except
    AItem.ModifiedDate := 0.0;
  end;
end;

procedure TIdFTPListItems.ParseUnix(AItem: TIdFTPListItem);
var
  LDir, LGPerm, LOPerm, LUPerm, LCount, LOwner, LGroup: string;
  LSize, LName, LLinkTo: string;
  LBuffer: string;
  wYear, wMonth, wDay: Word;
  wHour, wMin, wSec, wMSec: Word;
  ADate: TDateTime;
  i: Integer;
begin
  // Get defaults for modified date/time
  ADate := Now;
  DecodeDate(ADate, wYear, wMonth, wDay);
  DecodeTime(ADate, wHour, wMin, wSec, wMSec);

  // Copy the predictable pieces
  LDir := UpperCase(Copy(AItem.Data, 1, 1));
  LOPerm := Copy(AItem.Data, 2, 3);
  LGPerm := Copy(AItem.Data, 5, 3);
  LUPerm := Copy(AItem.Data, 8, 3);
  LCount := Trim(Copy(AItem.Data, 11, 5));

  // Scan for the rest
  LBuffer := Trim(Copy(AItem.Data, 16, Length(AItem.Data)));

  LOwner := Fetch(LBuffer);
  LBuffer := Trim(LBuffer);

  LGroup := Fetch(LBuffer);
  LBuffer := Trim(LBuffer);

  // Scan size
  LSize := Fetch(LBuffer);

  // Scan modified MMM
  LBuffer := Trim(LBuffer);
  wMonth := StrToMonth(Fetch(LBuffer));

  // Scan DD
  LBuffer := Trim(LBuffer);
  wDay := StrToIntDef(Fetch(LBuffer), wDay);

  LBuffer := Trim(LBuffer);

  // Not time info, scan year
  if (IndyPos(':', LBuffer) = 0) then begin
    wYear := StrToIntDef(Fetch(LBuffer), wYear);

    // Set time info to 00:00:00.999
    wHour := 0;
    wMin := 0;
    wSec := 0;
    wMSec := 999;
  end {if (IndyPos(':', SBuffer) = 0) }
  // Time info, scan hour, min
  else begin
    // Scan hour
    wHour := StrToIntDef(Fetch(LBuffer, ':'), 0);

    // Scan minutes
    wMin := StrToIntDef(Fetch(LBuffer), 0);

    // Set sec and ms to 0.999
    wSec := 0;
    wMSec := 999;
  end;

  // Scan remaining data into name
  LName := Trim(LBuffer);

  if LDir = 'D' then begin
    AItem.ItemType := ditDirectory;
  end else if LDir = 'L' then begin
    AItem.ItemType := ditSymbolicLink;
  end else begin
    AItem.ItemType := ditFile;
  end;

  AItem.OwnerPermissions := LOPerm;
  AItem.GroupPermissions := LGPerm;
  AItem.UserPermissions := LUPerm;
  AItem.ItemCount := StrToIntDef(LCount, 0);
  AItem.OwnerName := LOwner;
  AItem.GroupName := LGroup;
  AItem.Size := StrToInt64Def(LSize, 0);
  AItem.ModifiedDate := EncodeDate(wYear, wMonth, wDay) + EncodeTime(wHour, wMin, wSec, wMSec);

  if AItem.ItemType = ditSymbolicLink then begin
    i := IndyPos(' -> ', LName);
    LLinkTo := Copy(LName, i + 4, Length(LName) - i - 3);
    LName := Copy(LName, 1, i - 1);
    AItem.LinkedItemName := LLinkTo;
  end;
  AItem.FileName := LName
end;

procedure TIdFTPListItems.ParseVax(AItem: TIdFTPListItem);
begin
  // TODO: determine special characteristics for VAX other than disk prefix
  ParseUnix(AItem);
end;

procedure TIdFTPListItems.ParseUnknown(AItem: TIdFTPListItem);
begin
  raise EIdInvalidFTPListingFormat.Create(RSInvalidFTPListingFormat);
end;

procedure TIdFTPListItems.ParseCustom(AItem: TIdFTPListItem);
begin
  if Assigned(FOnParseCustomListFormat) then begin
    FOnParseCustomListFormat(AItem);
  end else begin
    raise EIdInvalidFTPListingFormat.Create(RSInvalidFTPListingFormat);
  end;
end;

procedure TIdFTPListItems.SetItems(AIndex: Integer; const Value: TIdFTPListItem);
begin
  inherited Items[AIndex] := Value;
end;

procedure TIdFTPListItems.SetDirectoryName(const AValue: string);
begin
  if not AnsiSameText(FDirectoryName, AValue) then begin
    FDirectoryName := AValue;
    Clear;
  end;
end;

function TIdFTPListItem.Text: string;
var
  LSize, LTime: string;
  l, month: Word;
begin
  case TIdFTPListItems(Collection).FListFormat of
    flfNone: Result := Data;
    flfNoDetails: Result := FileName;
    //flfUnknown: - No handler
    flfCustom: Result := DoGetCustomListFormat;
    flfDos: begin
      if ItemType = ditDirectory then begin
        LSize := '      ' + '<DIR>' + StringOfChar(' ', 9);
      end else begin
        LSize := StringOfChar(' ', 20 - Length(IntToStr(Size))) + IntToStr(Size);
      end;
      Result := FormatDateTime('mm-dd-yy  hh:mma/p', ModifiedDate) + ' ' + LSize
       + '  ' + FileName;
    end;
    flfUnix, flfVax: begin
      LSize := '-';
      case ItemType of
        ditDirectory: begin
          Size := 512;
          LSize := 'd';
        end;
        ditSymbolicLink: LSize := 'l';
      end;
      LSize := LSize + Format('%3:3s%4:3s%5:3s   1 %1:8s %2:8s %0:8d'
       , [Size, OwnerName, GroupName, OwnerPermissions, GroupPermissions, UserPermissions]);
      DecodeDate(ModifiedDate, l, month, l);
      LTime := MonthNames[month] + FormatDateTime(' dd', ModifiedDate);
      if FormatDateTime('yy', ModifiedDate) = FormatDateTime('yy', Now) then begin
        LTime := LTime + FormatDateTime(' hh:mm', ModifiedDate);
      end else begin
        LTime := LTime + FormatDateTime(' yyyy ', ModifiedDate);
      end;
      Result := LSize + ' ' + LTime + '  ' + FileName;
    end;
  end;
end;

function TIdFTPListItem.DoGetCustomListFormat: string;
begin
  Result := '';
  if Assigned(TIdFTPListItems(Collection).OnGetCustomListFormat) then begin
    TIdFTPListItems(Collection).OnGetCustomListFormat(Self, Result);
  end;
end;

end.
