unit IdFTP;

{
-Added a few more functions - by SP - on 06/28/2001

ChageLog:
  Doychin - 02/18/2001
    OnAfterLogin event handler and Login method

    OnAfterLogin is executed after successfull login  but before setting up the
      connection properties. This event can be used to provide FTP proxy support
      from the user application. Look at the FTP demo program for more information
      on how to provide such support.

  Doychin - 02/17/2001
    New onFTPStatus event
    New Quote method for executing commands not implemented by the compoent

-CleanDir contributed by Amedeo Lanza

TODO: Chage the FTP demo to demonstrate the use of the new events and add proxy support
}

interface

uses
  Classes,
  IdAssignedNumbers,
  IdException,
  IdSocketHandle, IdTCPConnection, IdTCPClient, IdThread, IdFTPList, IdFTPCommon;

type
  // TIdFTPStatus = (ftpTransfer, ftpReady, ftpAborted);
  //Added by SP
  // TIdFTPStatusEvent = procedure(ASender: TObject; const AFTPStatus: TIdFTPStatus) of object;
  TIdCreateFTPList = procedure(ASender: TObject; Var VFTPList: TIdFTPListItems) of object;
  TIdCheckListFormat = procedure(ASender: TObject; const ALine: String; Var VListFormat: TIdFTPListFormat) of object;
  TOnAfterClientLogin = TNotifyEvent;

const
  Id_TIdFTP_TransferType = ftBinary;
  Id_TIdFTP_Passive = False;

type
  TIdFTP = class(TIdTCPClient)
  protected
    FPassive: boolean;
    FCanResume: Boolean;
    FResumeTested: Boolean;
    FSystemDesc: string;
    FTransferType: TIdFTPTransferType;
    FDataChannel: TIdTCPConnection;
    FLoginMsg : TStrings;
    FWelcomeMsg : TStrings;
    FDirectoryListing: TIdFTPListItems;
    FListResult: TStringList;
    //
    // FOnFTPStatus: TIdFTPStatusEvent;
    FOnAfterClientLogin: TNotifyEvent;
    FOnCreateFTPList: TIdCreateFTPList;
    FOnCheckListFormat: TIdCheckListFormat;
    //
    procedure ConstructDirListing;
    procedure DoAfterLogin;
    // procedure DoFTPStatus(AStatus: TIdFTPStatus);
    procedure DoFTPList;
    procedure DoCheckListFormat(const ALine: String);
    function GetDirectoryListing: TIdFTPListItems;
    function GetOnParseCustomListFormat: TIdOnParseCustomListFormat;
    procedure InitDataChannel;
    procedure InternalGet(const ACommand: string; ADest: TStream; AResume: Boolean);
    procedure InternalPut(const ACommand: string; ASource: TStream);
    procedure SetOnParseCustomListFormat(const AValue: TIdOnParseCustomListFormat);
    procedure SendPassive(var VIP: string; var VPort: integer);
    procedure SendPort(AHandle: TIdSocketHandle);
    procedure SendTransferType;
    procedure SetTransferType(AValue: TIdFTPTransferType);
  public
    procedure Abort; virtual;
    procedure ChangeDir(const ADirName: string);
    procedure ChangeDirUp;
    procedure Connect(AAutoLogin: boolean = True); reintroduce;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Delete(const AFilename: string);
    procedure Get(const ASourceFile: string; ADest: TStream; AResume: Boolean = false); overload;
    procedure Get(const ASourceFile, ADestFile: string; const ACanOverwrite: boolean = false; AResume: Boolean = false);
     overload;
    procedure KillDataChannel; virtual;
    procedure List(ADest: TStrings; const ASpecifier: string = ''; const ADetails: boolean = true);
    procedure Login(AUserName, APassword: String);
    procedure MakeDir(const ADirName: string);
    procedure Noop;
    procedure Put(const ASource: TStream; const ADestFile: string = '';
     const AAppend: boolean = false); overload;
    procedure Put(const ASourceFile: string; const ADestFile: string = '';
     const AAppend: boolean = false); overload;
    procedure Quit;
    procedure RemoveDir(const ADirName: string);
    procedure Rename(const ASourceFile, ADestFile: string);
    function ResumeSupported: Boolean;
    function RetrieveCurrentDir: string;
    procedure Site(const ACommand: string);
    function Quote(const ACommand: String): SmallInt;
    function Size(const AFileName: String): Integer;
    procedure ReInitialize(ADelay: Cardinal = 10);
    procedure Allocate(AAllocateBytes: Integer);
    procedure Status(var AStatusList: TStringList);
    procedure Help(var AHelpContents: TStringList; ACommand: String = '');
    procedure Account(AInfo: String);
    procedure StructureMount(APath: String);
    procedure FileStructure(AStructure: TIdFTPDataStructure);
    procedure TransferMode(ATransferMode: TIdFTPTransferMode);
    //
    property CanResume: Boolean read ResumeSupported;
    property DirectoryListing: TIdFTPListItems read GetDirectoryListing;// FDirectoryListing;
    property LoginMsg : TStrings read FLoginMsg;
    property SystemDesc: string read FSystemDesc;
    property WelcomeMsg : TStrings read FWelcomeMsg;
  published
    property Passive: boolean read FPassive write FPassive default Id_TIdFTP_Passive;
    property Password;
    property TransferType: TIdFTPTransferType read FTransferType write SetTransferType default Id_TIdFTP_TransferType;
    property Username;
    property Port default IDPORT_FTP;

    property OnAfterClientLogin: TOnAfterClientLogin read FOnAfterClientLogin write FOnAfterClientLogin;
    property OnCheckListFormat: TIdCheckListFormat read FOnCheckListFormat write FOnCheckListFormat;
    property OnCreateFTPList: TIdCreateFTPList read FOnCreateFTPList write FOnCreateFTPList;
    // property OnFTPStatus: TIdFTPStatusEvent read FOnFTPStatus write FOnFTPStatus;
    property OnParseCustomListFormat: TIdOnParseCustomListFormat read GetOnParseCustomListFormat
     write SetOnParseCustomListFormat;
  end;
  EIdFTPFileAlreadyExists = class(EIdException);

implementation

uses
  IdComponent, IdGlobal, IdResourceStrings, IdStack, IdSimpleServer, IdIOHandlerSocket,
  SysUtils;

function CleanDirName(const APWDReply: string): string;
begin
  Result := APWDReply;
  Delete(result, 1, IndyPos('"', result)); // Remove first doublequote
  Result := Copy(result, 1, IndyPos('"', result) - 1); // Remove anything from second doublequote                                 // to end of line
end;

constructor TIdFTP.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Port := IDPORT_FTP;
  Passive := Id_TIdFTP_Passive;
  FTransferType := Id_TIdFTP_TransferType;
  FLoginMsg := TStringList.Create;
  FListResult := TStringList.Create;
  FWelcomeMsg := TStringList.Create;
  FCanResume := false;
  FResumeTested := false;
end;

procedure TIdFTP.Connect(AAutoLogin: boolean = True);
begin
  try
    inherited Connect;
    GetResponse([220]);
    WelcomeMsg.Assign(CmdResultDetails);
    if AAutoLogin then begin
      Login(UserName, Password);
      DoAfterLogin;

      SendTransferType;

      // OpenVMS 7.1 replies with 200 instead of 215 - What does the RFC say about this?
      if SendCmd('syst', [200, 215, 500]) = 500 then begin
        FSystemDesc := RSFTPUnknownHost;
      end else begin
        FSystemDesc := Copy(CmdResult, 4, MaxInt);
      end;

      DoStatus(ftpReady);
    end;
  except
    Disconnect;
    raise;
  end;
end;

procedure TIdFTP.SetTransferType(AValue: TIdFTPTransferType);
begin
  if AValue <> FTransferType then begin
    if not Assigned(FDataChannel) then begin
      FTransferType := AValue;
      if Connected then begin
        SendTransferType;
      end;
    end
  end;
end;

procedure TIdFTP.SendTransferType;
var
  s: string;
begin
  case TransferType of
    ftAscii: s := 'A';
    ftBinary: s := 'I';
  end;
  SendCmd('type ' + s, 200);
end;

function TIdFTP.ResumeSupported: Boolean;
begin
  if FResumeTested then result := FCanResume
  else begin
    FResumeTested := true;
    FCanResume := Quote('REST 1') = 350;
    result := FCanResume;
    Quote('REST 0');
  end;
end;

procedure TIdFTP.Get(const ASourceFile: string; ADest: TStream; AResume: Boolean = false);
begin
  AResume := AResume and CanResume;
  InternalGet('retr ' + ASourceFile, ADest, AResume);
end;

procedure TIdFTP.Get(const ASourceFile, ADestFile: string; const ACanOverwrite: boolean = false;
  AResume: Boolean = false);
var
  LDestStream: TFileStream;
begin
  if FileExists(ADestFile) then begin
    AResume := AResume and CanResume;
    if ACanOverwrite and (not AResume) then begin
      LDestStream := TFileStream.Create(ADestFile, fmCreate);
    end
    else begin
      if (not ACanOverwrite) and AResume then begin
        LDestStream := TFileStream.Create(ADestFile, fmOpenWrite);
        LDestStream.Seek(0, soFromEnd);
      end
      else begin
        raise EIdFTPFileAlreadyExists.Create(RSDestinationFileAlreadyExists);
      end;
    end;
  end
  else begin
    LDestStream := TFileStream.Create(ADestFile, fmCreate);
  end;

  try
    Get(ASourceFile, LDestStream, AResume);
  finally
    FreeAndNil(LDestStream);
  end;
end;

procedure TIdFTP.ConstructDirListing;
begin
  if not Assigned(FDirectoryListing) then begin
    if not (csDesigning in ComponentState) then begin
      DoFTPList;
    end;
    if not Assigned(FDirectoryListing) then begin
      FDirectoryListing := TIdFTPListItems.Create;
    end;
  end else begin
    FDirectoryListing.Clear;
  end;
end;

procedure TIdFTP.List(ADest: TStrings; const ASpecifier: string = '';
 const ADetails: boolean = true);
var
  LDest: TStringStream;
begin
  LDest := TStringStream.Create(''); try
    if ADetails then begin
      InternalGet(trim('list ' + ASpecifier), LDest, false);
    end else begin
      InternalGet(trim('nlst ' + ASpecifier), LDest, false);
    end;
    FreeAndNil(FDirectoryListing);
    ADest.Text := LDest.DataString;
    FListResult.Text := LDest.DataString;
  finally LDest.Free; end;
end;

procedure TIdFTP.InternalGet(const ACommand: string; ADest: TStream; AResume: Boolean);
var
  LIP: string;
  LPort: Integer;
begin
  DoStatus(ftpTransfer); try
    if FPassive then begin
      SendPassive(LIP, LPort);
      if AResume then begin
        SendCmd('REST ' + IntToStr(ADest.Position), [350]);
      end;
      WriteLn(ACommand);
      FDataChannel := TIdTCPClient.Create(nil); try
        with (FDataChannel as TIdTCPClient) do begin
          if (IOHandler is TIdIOHandlerSocket) and
            (Self.IOHandler is TIdIOHandlerSocket) then
          begin
            (IOHandler as TIdIOHandlerSocket).SocksInfo.Assign((Self.IOHandler as TIdIOHandlerSocket).SocksInfo);
          end;
          InitDataChannel;
          Host := LIP;
          Port := LPort;
          Connect; try
            Self.GetResponse([125, 150]);
            ReadStream(ADest, -1, True);
          finally Disconnect; end;
        end;
      finally FreeAndNil(FDataChannel); end;
    end else begin
      FDataChannel := TIdSimpleServer.Create(nil); try
        with TIdSimpleServer(FDataChannel) do begin
          InitDataChannel;
          BoundIP := (Self.IOHandler as TIdIOHandlerSocket).Binding.IP;
          BeginListen;
          SendPort(Binding);
          if AResume then begin
            Self.SendCmd('REST ' + IntToStr(ADest.Position), [350]);
          end;
          Self.SendCmd(ACommand, [125, 150]);
          Listen;
          ReadStream(ADest, -1, True);
        end;
      finally
        FreeAndNil(FDataChannel);
      end;
    end;
  finally
    DoStatus(ftpReady);
  end;
  // ToDo: Change that to properly handle response code (not just success or except)
  // 226 = download successful, 225 = Abort successful}
  if GetResponse([225, 226, 250, 426]) = 426 then begin
    GetResponse([226, 225]);
    DoStatus(ftpAborted);
  end;
end;

procedure TIdFTP.Quit;
begin
  WriteLn('Quit');
  Disconnect;
end;

procedure TIdFTP.KillDataChannel;
begin
  // Had kill the data channel ()
  if Assigned(FDataChannel) then begin
    FDataChannel.DisconnectSocket;
  end;
end;

procedure TIdFTP.Abort;
begin
  // only send the abort command. The Data channel is supposed to disconnect
  if Connected then begin
    WriteLn('ABOR');
  end;
  // Kill the data channel: usually, the server doesn't close it by itself
  KillDataChannel;
end;

procedure TIdFTP.SendPort(AHandle: TIdSocketHandle);
begin
  SendCmd('PORT ' + StringReplace(AHandle.IP, '.', ',', [rfReplaceAll])
   + ',' + IntToStr(AHandle.Port div 256) + ',' + IntToStr(AHandle.Port mod 256), [200]);
end;

procedure TIdFTP.InternalPut(const ACommand: string; ASource: TStream);
var
  LIP: string;
  LPort: Integer;
begin
  DoStatus(ftpTransfer); try
    if FPassive then begin
      SendPassive(LIP, LPort);

      WriteLn(ACommand);
      FDataChannel := TIdTCPClient.Create(nil);
      with TIdTCPClient(FDataChannel) do try
        InitDataChannel;
        Host := LIP;
        Port := LPort;
        if (IOHandler is TIdIOHandlerSocket) and
          (Self.IOHandler is TIdIOHandlerSocket) then
        begin
          (IOHandler as TIdIOHandlerSocket).SocksInfo.Assign((Self.IOHandler as TIdIOHandlerSocket).SocksInfo);
        end;

        Connect; try
          Self.GetResponse([110, 125, 150]);
          try
            WriteStream(ASource, false);
          except
            on E: EIdSocketError do begin
              // If 10038 - abort was called. Server will return 225
              if E.LastError <> 10038 then begin
                raise;
              end;
            end;
          end;
        finally Disconnect; end;
      finally FreeAndNil(FDataChannel); end;
    end else begin
      FDataChannel := TIdSimpleServer.Create(nil); try
        with TIdSimpleServer(FDataChannel) do begin
          InitDataChannel;
          BoundIP := (Self.IOHandler as TIdIOHandlerSocket).Binding.IP;
          BeginListen;
          SendPort(Binding);
          Self.SendCmd(ACommand, [125, 150]);
          Listen;
          WriteStream(ASource);
        end;
      finally FreeAndNil(FDataChannel); end;
    end;
  finally
    DoStatus(ftpReady);
  end;
  // 226 = download successful, 225 = Abort successful}
  if GetResponse([225, 226, 250, 426]) = 426 then begin
    // some servers respond with 226 on ABOR
    GetResponse([226, 225]);
    DoStatus(ftpAborted);
  end;
end;

procedure TIdFTP.InitDataChannel;
begin
  FDataChannel.SendBufferSize := SendBufferSize;
  FDataChannel.RecvBufferSize := RecvBufferSize;
  FDataChannel.OnWork := OnWork;
  FDataChannel.OnWorkBegin := OnWorkBegin;
  FDataChannel.OnWorkEnd := OnWorkEnd;
end;

procedure TIdFTP.Put(const ASource: TStream; const ADestFile: string = '';
 const AAppend: boolean = false);
begin
  if length(ADestFile) = 0 then begin
    InternalPut('STOU ' + ADestFile, ASource);
  end else if AAppend then begin
    InternalPut('APPE ' + ADestFile, ASource);
  end else begin
    InternalPut('STOR ' + ADestFile, ASource);
  end;
end;

procedure TIdFTP.Put(const ASourceFile: string; const ADestFile: string = '';
 const AAppend: boolean = false);
var
  LSourceStream: TFileStream;
begin
  LSourceStream := TFileStream.Create(ASourceFile, fmOpenRead or fmShareDenyNone); try
    Put(LSourceStream, ADestFile, AAppend);
  finally FreeAndNil(LSourceStream); end;
end;

procedure TIdFTP.SendPassive(var VIP: string; var VPort: integer);
var
  i,bLeft,bRight: integer;
  s: string;
begin
  SendCmd('PASV', 227);
  s := Trim(CmdResult);
  // Case 1 (Normal)
  // 227 Entering passive mode(100,1,1,1,23,45)
  bLeft:=IndyPos('(',s);
  bRight:=IndyPos(')',s);
  if (bLeft=0) or (bRight=0) then begin
    // Case 2
    // 227 Entering passive mode on 100,1,1,1,23,45
    bLeft:=RPos(#32,s);
    s:=Copy(s,bLeft+1,Length(s)-bLeft);
  end else begin
    s:=Copy(s,bLeft+1,bRight-bLeft-1);
  end;
  VIP := '';
  for i := 1 to 4 do begin
    VIP := VIP + '.' + Fetch(s, ',');
  end;
  System.Delete(VIP, 1, 1);
  // Determine port
  VPort := StrToInt(Fetch(s, ',')) * 256;
  VPort := VPort + StrToInt(Fetch(s, ','));
end;

procedure TIdFTP.Noop;
begin
  SendCmd('NOOP', 200);
end;

procedure TIdFTP.MakeDir(const ADirName: string);
begin
  SendCmd('MKD ' + ADirName, 257);
end;

function TIdFTP.RetrieveCurrentDir: string;
begin
  SendCmd('PWD', 257);
  Result := CleanDirName(CmdResult);
end;

procedure TIdFTP.RemoveDir(const ADirName: string);
begin
  SendCmd('RMD ' + ADirName, 250);
end;

procedure TIdFTP.Delete(const AFilename: string);
begin
  SendCmd('DELE ' + AFilename, 250);
end;

procedure TIdFTP.ChangeDir(const ADirName: string);
begin
  SendCmd('CWD ' + ADirName, 250);
end;

procedure TIdFTP.ChangeDirUp;
begin
  SendCmd('CDUP', 200);
end;

procedure TIdFTP.Site(const ACommand: string);
begin
  SendCmd('SITE ' + ACommand, 200);
end;

procedure TIdFTP.Rename(const ASourceFile, ADestFile: string);
begin
  SendCmd('RNFR ' + ASourceFile, 350);
  SendCmd('RNTO ' + ADestFile, 250);
end;

function TIdFTP.Size(const AFileName: String): Integer;
var
  SizeStr: String;
begin
  result := -1;
  if SendCmd('SIZE ' + AFileName) = 213 then begin
    SizeStr := Trim(CmdResultDetails.text);
    system.delete(SizeStr, 1, IndyPos(' ', SizeStr)); // delete the response
    result := StrToIntDef(SizeStr, -1);
  end;
end;

//Added by SP
procedure TIdFTP.ReInitialize(ADelay: Cardinal = 10);
begin
  Sleep(ADelay); //Added
  if SendCmd('REIN', [120, 220, 500]) <> 500 then
  begin
    FLoginMsg.Clear;
    FWelcomeMsg.Clear;
    FCanResume := False;
    FDirectoryListing.Clear;
    FUsername := '';
    FPassword := '';
    FPassive := Id_TIdFTP_Passive;
    FCanResume := False;
    FResumeTested := False;
    FSystemDesc := '';
    FTransferType := Id_TIdFTP_TransferType;
  end;
end;

procedure TIdFTP.Allocate(AAllocateBytes: Integer);
begin
  SendCmd('ALLO ' + IntToStr(AAllocateBytes), [200]);
end;

procedure TIdFTP.Status(var AStatusList: TStringList);
var
  LStrm: TStringStream;
  LList: TStringList;
begin
  if SendCmd('STAT', [211, 212, 213, 500]) <> 500 then
  begin
    if not Assigned(FDirectoryListing) then
    begin
      DoFTPList;
    end;
    LStrm := TStringStream.Create('');
    LList := TStringList.Create;
    //Read stream through control connection - not data channel
    ReadStream(LStrm, -1, True);
    LList.Text := LStrm.DataString;
    try
      try
        ConstructDirListing;
        FDirectoryListing.Clear;
      except
        on EAccessViolation do ConstructDirListing;
      end;
      // Parse directory listing
      if LList.Count > 0 then
      begin
        FDirectoryListing.ListFormat := FDirectoryListing.CheckListFormat(LList[0], True);
        DoCheckListFormat(LList[0]);
        FDirectoryListing.LoadList(LList);
      end;
    except
      if Assigned(AStatusList) = True then
      begin
        AStatusList.Text := LStrm.DataString;
      end;
    end;
    FreeAndNil(LStrm);
    FreeAndNil(LList);
  end;
end;

procedure TIdFTP.Help(var AHelpContents: TStringList; ACommand: String = '');
var
  LStrm: TStringStream;
begin
  LStrm := TStringStream.Create('');
  if SendCmd('HELP ' + ACommand, [211, 214, 500]) <> 500 then
  begin
    ReadStream(LStrm, -1, True);
    AHelpContents.Text := LStrm.DataString;
  end;
  FreeAndNil(LStrm);
end;

procedure TIdFTP.Account(AInfo: String);
begin
  SendCmd('ACCT ' + AInfo, [202, 230, 500]);
end;

procedure TIdFTP.StructureMount(APath: String);
begin
  SendCmd('SMNT ' + APath, [202, 250, 500]);
end;

procedure TIdFTP.FileStructure(AStructure: TIdFTPDataStructure);
var
  s: String;
begin
  case AStructure of
    dsFile: s := 'F';
    dsRecord: s := 'R';
    dsPage: s := 'P';
  end;
  SendCmd('STRU ' + s, [200, 500]);
  { TODO: Needs to be finished }
end;

procedure TIdFTP.TransferMode(ATransferMode: TIdFTPTransferMode);
var
  s: String;
begin
  case ATransferMode of
    dmBlock: begin
      s := 'B';
    end;
    dmCompressed: begin
      s := 'C';
    end;
    dmStream: begin
      s := 'S';
    end;
  end;
  SendCmd('MODE ' + s, [200, 500]);
  { TODO: Needs to be finished }
end;

destructor TIdFTP.Destroy;
begin
  FreeAndNil(FListResult);
  FreeAndNil(FLoginMsg);
  FreeAndNil(FWelcomeMsg);
  FreeAndNil(FDirectoryListing);
  inherited Destroy;
end;

{procedure TIdFTP.DoFTPStatus(AStatus: TIdFTPStatus);
begin
  if Assigned(FOnFTPStatus) then begin
    OnFTPStatus(self, AStatus);
  end;
end;}

function TIdFTP.Quote(const ACommand: String): SmallInt;
begin
  result := SendCmd(ACommand);
end;

procedure TIdFTP.Login(AUserName, APassword: String);
begin
  if SendCmd('user ' + AUserName, [230, 331]) = 331 then begin
    SendCmd('pass ' + APassword, 230);
  end;
  FLoginMsg.Assign(CmdResultDetails);
end;

procedure TIdFTP.DoAfterLogin;
begin
  if Assigned(FOnAfterClientLogin) then begin
    OnAfterClientLogin(self);
  end;
end;

procedure TIdFTP.DoFTPList;
begin
  if Assigned(FOnCreateFTPList) then begin
    FOnCreateFTPList(self, FDirectoryListing);
  end;
end;

procedure TIdFTP.DoCheckListFormat(const ALine: String);
Var
  LListFormat: TIdFTPListFormat;
begin
  if (FDirectoryListing.ListFormat = flfUnknown) and Assigned(FOnCheckListFormat) then begin
    LListFormat := flfUnknown;
    OnCheckListFormat(Self, ALine, LListFormat);
    FDirectoryListing.ListFormat := LListFormat;
  end;
end;

function TIdFTP.GetDirectoryListing: TIdFTPListItems;
begin
  if not Assigned(FDirectoryListing) then begin
    try
      ConstructDirListing;
    except
      on EAccessViolation do ConstructDirListing;
    end;
    // Parse directory listing
    if FListResult.Count > 0 then begin
      FDirectoryListing.ListFormat := FDirectoryListing.CheckListFormat(FListResult[0]);
      DoCheckListFormat(FListResult[0]);
      FDirectoryListing.LoadList(FListResult);
    end;
  end;
  Result := FDirectoryListing;
end;

function TIdFTP.GetOnParseCustomListFormat: TIdOnParseCustomListFormat;
begin
  Result := DirectoryListing.OnParseCustomListFormat
end;

procedure TIdFTP.SetOnParseCustomListFormat(const AValue: TIdOnParseCustomListFormat);
begin
  DirectoryListing.OnParseCustomListFormat := AValue;
end;

end.

