unit IdCookieManager;

{
  Implementation of the HTTP State Management Mechanism as specified in RFC 2109, 2965.

  Author: Doychin Bondzhev (doychin@dsoft-bg.com)
  Copyright: (c) Chad Z. Hower and The Indy Team.

Details of implementation
-------------------------

2001-Mar-31 Doychin Bondzhev
 - Added new method AddCookie2 that is called when we have Set-Cookie2 as response
 - The common code in AddCookie and AddCookie2 is now in DoAdd
2001-Mar-24 Doychin Bondzhev
 - Added OnNewCookie event
   This event is called for every new cookie. Can be used to ask the user program do we have to store this
   cookie in the cookie collection
 - Added new method AddCookie
   This calls the OnNewCookie event and if the result is true it adds the new cookie in the collection
}

interface
                                                                         
Uses Classes, SysUtils, IdComponent, IdCookie, IdGlobal, IdURI;

Type
  TOnNewCookieEvent = procedure(ASender: TObject; ACookie: TIdCookieRFC2109; Var VAccept: Boolean) of object;

  TOnManagerEvent = procedure(ASender: TObject; ACookieCollection: TIdCookies) of object;
  TOnCreateEvent = TOnManagerEvent;
  TOnDestroyEvent = TOnManagerEvent;

  TIdCookieManager = class(TIdComponent)
  protected
    FOnCreate: TOnCreateEvent;
    FOnDestroy:  TOnDestroyEvent;
    FOnNewCookie: TOnNewCookieEvent;
    FCookieCollection: TIdCookies;

    procedure DoAdd(ACookie: TIdCookieRFC2109; ACookieText, AHost: String);
    procedure DoOnCreate; virtual;
    procedure DoOnDestroy; virtual;

    function DoOnNewCookie(ACookie: TIdCookieRFC2109): Boolean; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure AddCookie(ACookie, AHost: String);
    procedure AddCookie2(ACookie, AHost: String);
    function GenerateCookieList(URL: TIdURI; SecureConnection: Boolean = false): String;
    property CookieCollection: TIdCookies read FCookieCollection;
  published
    property OnCreate: TOnCreateEvent read FOnCreate write FOnCreate;
    property OnDestroy: TOnDestroyEvent read FOnDestroy write FOnDestroy;

    property OnNewCookie: TOnNewCookieEvent read FOnNewCookie write FOnNewCookie;
  end;

implementation

{ TIdCookieManager }

constructor TIdCookieManager.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCookieCollection := TIdCookies.Create(self);
  DoOnCreate;
end;

destructor TIdCookieManager.Destroy;
begin
  DoOnDestroy;
  FreeAndNil(FCookieCollection);
  inherited Destroy;
end;

function TIdCookieManager.GenerateCookieList(URL: TIdURI; SecureConnection: Boolean = false): String;
Var
  S1, S: String;
  i, j: Integer;
  LCookieList: TIdCookieList;
  LResultList: TIdCookieList;
  LCookiesByDomain: TIdCookieList;
begin
  j := 0;
  S := '';
  LCookiesByDomain := FCookieCollection.LockCookieListByDomain(caRead);
  try
    if LCookiesByDomain.Count > 0 then
    begin
      LResultList := TIdCookieList.Create;
      LResultList.Duplicates := dupAccept;
      LResultList.Sorted := true;

      try
        repeat
          // Fill cookie part in the request
          for i := j to LCookiesByDomain.Count - 1 do
          begin
            if Pos(LCookiesByDomain[i], URL.Host) > 0 then
              break;
          end;

          j := i + 1;

          if i = LCookiesByDomain.Count then
            break;

          LCookieList := LCookiesByDomain.Objects[i] as TIdCookieList;

          for i := LCookieList.Count - 1 downto 0 do
          begin
            if Pos(LCookieList.Cookies[i].Path, URL.Path) = 1 then
            begin
              S1 := LCookieList.Cookies[i].Expires;
              if (S1 <> '') and (GMTToLocalDateTime(S1) < Now) then
              begin
                // The Cookie has exiered. It has to be removed from the collection
                LCookieList.Cookies[i].Free;
                LCookieList.Delete(i);
              end else
                with LCookieList.Cookies[i] do
                begin
                  if ((Secure and SecureConnection) or (not Secure)) and (Value <> '') then
                  begin
                    LResultList.AddObject(Path, LCookieList.Cookies[i]);
                  end;
                end;
            end;
          end;
        until false;

        for i := LResultList.Count - 1 downto 0 do
        begin
          if Length(S) > 0  then S := S + '; ';
          S := S + LResultList.Cookies[i].CookieName + '=' + LResultList.Cookies[i].Value;
        end;
      finally
        LResultList.Free;
      end;
    end;
  finally
    FCookieCollection.UnlockCookieListByDomain(caRead);
  end;
  result := S;
end;

procedure TIdCookieManager.DoAdd(ACookie: TIdCookieRFC2109; ACookieText, AHost: String);
Var
  LDomain: String;
begin
  ACookie.CookieText := ACookieText;

  if Length(ACookie.Domain) = 0 then LDomain := AHost
  else LDomain := ACookie.Domain;

  ACookie.Domain := LDomain;
  
  if ACookie.IsValidCookie(AHost) then
  begin
    if DoOnNewCookie(ACookie) then
    begin
      FCookieCollection.AddCookie(ACookie);
    end
    else begin
      ACookie.Collection := nil;
      ACookie.Free;
    end;
  end
  else begin
    ACookie.Free;
  end;
end;

procedure TIdCookieManager.AddCookie(ACookie, AHost: String);
Var
  LCookie: TIdCookieRFC2109;
begin
  LCookie := FCookieCollection.Add;
  DoAdd(LCookie, ACookie, AHost);
end;

procedure TIdCookieManager.AddCookie2(ACookie, AHost: String);
Var
  LCookie: TIdCookieRFC2965;
begin
  LCookie := FCookieCollection.Add2;
  DoAdd(LCookie, ACookie, AHost);
end;

function TIdCookieManager.DoOnNewCookie(ACookie: TIdCookieRFC2109): Boolean;
begin
  result := true;
  if Assigned(FOnNewCookie) then
  begin
    OnNewCookie(self, ACookie, result);
  end;
end;

procedure TIdCookieManager.DoOnCreate;
begin
  if Assigned(FOnCreate) then
  begin
    OnCreate(self, FCookieCollection);
  end;
end;

procedure TIdCookieManager.DoOnDestroy;
begin
  if Assigned(FOnDestroy) then
  begin
    OnDestroy(self, FCookieCollection);
  end;
end;

end.
