{
  Implementation of the Basic authentication as specified in
  RFC 2616

  Copyright: (c) Chad Z. Hower and The Winshoes Working Group.

  Author: Doychin Bondzhev (doychin@dsoft-bg.com)

  Modified:

  2001-Sep-11 : DSiders
    Corrected spelling for EIdAlreadyRegisteredAuthenticationMethod
}

unit IdAuthentication;

interface

Uses
  Classes, IdHeaderList, IdGlobal, IdException;

Type
  TIdAuthenticationSchemes = (asBasic, asDigest, asNTLM, asUnknown);
  TIdAuthSchemeSet = set of TIdAuthenticationSchemes;

  TIdAuthWhatsNext = (wnAskTheProgram, wnDoRequest, wnFail);

  TIdAuthentication = class(TPersistent)
  protected
    FStatus: Integer;
    FParams: TIdHeaderList;
    FAuthParams: TIdHeaderList;

    function ReadAuthInfo(AuthName: String): String;
    function DoNext: TIdAuthWhatsNext; virtual; abstract;
    procedure SetAuthParams(AValue: TIdHeaderList);
    function GetPassword: String;
    function GetUserName: String;
    procedure SetPassword(const Value: String);
    procedure SetUserName(const Value: String);
  public
    constructor Create; virtual;
    destructor Destroy; override;

    procedure Reset; virtual;

    function Authentication: String; virtual; abstract;
    function KeepAlive: Boolean; virtual; abstract;
    function Next: TIdAuthWhatsNext;

    property AuthParams: TIdHeaderList read FAuthParams write SetAuthParams;
    property Params: TIdHeaderList read FParams;
    property Username: String read GetUserName write SetUserName;
    property Password: String read GetPassword write SetPassword;
  end;

  TIdAuthenticationClass = class of TIdAuthentication;

  TIdBasicAuthentication = class(TIdAuthentication)
  protected
    FRealm: String;
    function DoNext: TIdAuthWhatsNext; override;
  public
    constructor Create; override;
    function Authentication: String; override;
    function KeepAlive: Boolean; override;
    procedure Reset; override;

    property Realm: String read FRealm write FRealm;
  end;

  EIdAlreadyRegisteredAuthenticationMethod = class(EIdException);

  { Support functions }
  procedure RegisterAuthenticationMethod(MethodName: String; AuthClass: TIdAuthenticationClass);
  function FindAuthClass(AuthName: String): TIdAuthenticationClass;

implementation

Uses
  IdCoderMIME, SysUtils;

Type
  TAuthListObject = class(TObject)
    Auth: TIdAuthenticationClass;
  end;

Var
  AuthList: TStringList = nil;

procedure RegisterAuthenticationMethod(MethodName: String; AuthClass: TIdAuthenticationClass);
Var
  LAuthItem: TAuthListObject;
begin
  if not Assigned(AuthList) then begin
    AuthList := TStringList.Create;
  end;

  if AuthList.IndexOf(MethodName) < 0 then begin
    LAuthItem := TAuthListObject.Create;
    LAuthItem.Auth := AuthClass;
    AuthList.AddObject(MethodName, LAuthItem);
  end
  else begin
    raise EIdAlreadyRegisteredAuthenticationMethod.Create('This authentication method is already registered with class name' +
      TAuthListObject(AuthList.Objects[AuthList.IndexOf(MethodName)]).Auth.ClassName);
  end;
end;

function FindAuthClass(AuthName: String): TIdAuthenticationClass;
begin
  if AuthList.IndexOf(AuthName) = -1 then Abort;
  result := TAuthListObject(AuthList.Objects[AuthList.IndexOf(AuthName)]).Auth;
end;

{ TIdAuthentication }

constructor TIdAuthentication.Create;
begin
  inherited Create;
  FParams := TIdHeaderList.Create;

  FStatus := 0;
end;

destructor TIdAuthentication.Destroy;
begin
  FreeAndNil(FAuthParams);
  FreeAndNil(FParams);

  inherited Destroy;
end;

procedure TIdAuthentication.SetAuthParams(AValue: TIdHeaderList);
begin
  if not Assigned(FAuthParams) then begin
    FAuthParams := TIdHeaderList.Create;
  end;

  FAuthParams.Assign(AValue);
end;

function TIdAuthentication.ReadAuthInfo(AuthName: String): String;
Var
  i: Integer;
begin
  if Assigned(FAuthParams) then begin
    for i := 0 to FAuthParams.Count - 1 do begin
      if IndyPos(AuthName, FAuthParams[i]) = 1 then begin
        result := FAuthParams[i];
        exit;
      end;
    end;
  end
  else begin
    result := '';
  end;
end;

function TIdAuthentication.Next: TIdAuthWhatsNext;
begin
  result := DoNext;
end;

procedure TIdAuthentication.Reset;
begin
  // 
end;

function TIdAuthentication.GetPassword: String;
begin
  result := Params.Values['password'];
end;

function TIdAuthentication.GetUserName: String;
begin
  result := Params.Values['username'];
end;

procedure TIdAuthentication.SetPassword(const Value: String);
begin
  Params.Values['Password'] := Value;
end;

procedure TIdAuthentication.SetUserName(const Value: String);
begin
  Params.Values['Username'] := Value;
end;

{ TIdBasicAuthentication }

constructor TIdBasicAuthentication.Create;
begin
  inherited Create;
  FStatus := 0;
end;

function TIdBasicAuthentication.Authentication: String;
begin
  result := 'Basic ' {do not localize}
    + TIdEncoderMIME.EncodeString(Username + ':' + Password);
end;

function TIdBasicAuthentication.DoNext: TIdAuthWhatsNext;
Var
  S: String;
begin
  result := wnDoRequest;

  S := ReadAuthInfo('Basic');
  Fetch(S);

  while Length(S) > 0 do begin
    Params.Add(Fetch(S, ', '));
  end;

  FRealm := Copy(Params.Values['realm'], 2, Length(Params.Values['realm']) - 2);

  case FStatus of
    0: begin
      if (Length(Username) > 0) and (Length(Password) > 0) then begin
        result := wnDoRequest;
      end
      else begin
        result := wnAskTheProgram;
      end;
    end;
    1: begin
      result := wnFail;
    end;
  end;
end;

function TIdBasicAuthentication.KeepAlive: Boolean;
begin
  result := false;
end;

procedure TIdBasicAuthentication.Reset;
begin
  inherited Reset;
  FStatus := 0;
end;

initialization
  RegisterAuthenticationMethod('Basic', TIdBasicAuthentication);
finalization
  if Assigned(AuthList) then begin
    while AuthList.Count > 0 do begin
      AuthList.Objects[0].Free;
      AuthList.Delete(0);
    end;
    AuthList.Free;
  end;
end.
