unit IdAboutTest;

interface

uses
  {$IFDEF LINUX}
  QStdCtrls, QForms, QExtCtrls, QControls, QGraphics, types,QT,
  {$ELSE}
  Windows, Messages, StdCtrls, Buttons, ExtCtrls, Graphics, Controls, Forms,
  {$ENDIF}
  Classes, SysUtils;

type
  TScrollCredits = class;

  TformAbout = class(TForm)
    Panel1: TPanel;
    Panel2: TPanel;
    bvlDialogBorder: TBevel;
    lblCopyright: TLabel;
    lblVersion: TLabel;
    lblName: TLabel;
    lblPleaseVisitUs: TLabel;
    lblURL: TLabel;
    btnOk: TButton;
    tmrScroll: TTimer;
    procedure tmrScrollTimer(Sender: TObject);
    procedure FormShow(Sender: TObject);
    {$IFDEF LINUX}
    //This is necessary to prevent the About Box from being resized
    //which would cause an AV in the scroller and prevents other drawing problems
    procedure FormConstrainedResize(Sender: TObject; var MinWidth, MinHeight, MaxWidth
     , MaxHeight: Integer);
    {$ELSE}
    procedure lblURLClick(Sender: TObject);
    {$ENDIF}
  private
  protected
    scrCredits: TScrollCredits;
    procedure ScrollClick(Sender: TObject);
    {$IFNDEF LINUX}
    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd);  message WM_ERASEBKGND;
    {$ENDIF}
  public
    constructor Create(AOwner : TComponent); override;
  end;

  TCreditString = class(TCollectionItem)
  private
    FHeader: String;
    FBody: TStrings;
    FColCount: Integer;
    FTextExtent: TSize;
    procedure SetBody(const Value: TStrings);
    procedure SetHeader(const Value: String);
    procedure BodyChanged(sender: TObject);
    procedure SetColCount(const Value: Integer);
    function GetBodyItems(Index: Integer): String;
    procedure SetBodyItems(Index: Integer; const Value: String);
    {$HINTS OFF}
    property BodyItems[Index: Integer]: String read GetBodyItems write SetBodyItems; default;
    {$HINTS ON}
  protected
    function TextExtent(Canvas: TCanvas): TSize;
    procedure RenderText(const ATop, AWidth: Integer; Canvas: TCanvas);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property Header: String read FHeader write SetHeader;
    property Body: TStrings read FBody write SetBody;
    property ColCount: Integer read FColCount write SetColCount default 1;
  end;

  TCreditStrings = class(TOwnedCollection)
  private
    FOwner: TScrollCredits;
    FHeaderFont: TFont;
    FBodyFont: TFont;
    FColSpace: Integer;
    function GetItems(Index: Integer): TCreditString;
    procedure SetItems(Index: Integer; const Value: TCreditString);
    procedure SetBodyFont(const Value: TFont);
    procedure SetHeaderFont(const Value: TFont);
    procedure FontChange(Sender: TObject);
    procedure SetColSpace(const Value: Integer);
  protected
    procedure Update(Item: TCollectionItem); override;
  public
    function Add: TCreditString;
    constructor Create(AnOwner: TScrollCredits);
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    function TextExtent(Canvas: TCanvas): TSize;
    procedure RenderText(const ATop, AWidth: Integer; Canvas: TCanvas);
  public
    property ColSpace: Integer read FColSpace write SetColSpace;
    property Items[Index: Integer]: TCreditString read GetItems write SetItems; default;
    property HeaderFont: TFont read FHeaderFont write SetHeaderFont;
    property BodyFont: TFont read FBodyFont write SetBodyFont;
  end;

  {$IFDEF LINUX}
  TScrollCredits = class(TCustomControl)
  {$ELSE}
  TScrollCredits = class(TGraphicControl)
  {$ENDIF}
  private
    bmpCredits,
    bmpBack,
    bmpBuffer: TBitmap;
    FCredits: TCreditStrings;
    FCurrentLine: Integer;
    FBackground: TPicture;
    procedure SetCredits(const Value: TCreditStrings);
    procedure BackgroundChange(Sender: TObject);
    procedure CreditsChange;
    procedure SetCurrentLine(const Value: Integer);
    procedure SetBackground(const Value: TPicture);
    procedure TransposeCredits;
  protected
    procedure Paint; override;
    {$IFDEF LINUX}
    procedure FontChanged; override;
    // Set the widgetFlags for reducing the flikker on Linux
    function WidgetFlags: Integer; override;
    {$ELSE}
    procedure CMFontchanged(var Message: TMessage); message CM_FONTCHANGED;
    {$ENDIF}
    procedure InvalidateBitmap;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Background: TPicture read FBackground write SetBackground;
    property Credits: TCreditStrings read FCredits write SetCredits;
    property CurrentLine: Integer read FCurrentLine write SetCurrentLine;
    property OnClick;
  end;

// Procs
  procedure ShowAboutBox(const AProductName, AProductVersion: string);

implementation

{$R IdCreditsBitmap.res}

uses
  {$IFNDEF Linux}ShellApi,{$ENDIF}
  IdGlobal, IdResourceStrings;

Procedure ShowAboutBox(const AProductName, AProductVersion : String);
begin
  with TformAbout.Create(Application) do
  try
    lblName.Caption := AProductName;
    lblVersion.Caption := Format ( RSAAboutBoxVersion, [ AProductVersion ] );
    ShowModal;
  finally
    Free;
  end;
end;

{$IFNDEF LINUX}
procedure TformAbout.lblURLClick(Sender: TObject);
begin
  ShellAPI.shellExecute((Self as TControl).Handle,PChar('open'),PChar(lblURL.Caption),nil,nil, 0);
end;
{$ENDIF}

procedure TformAbout.ScrollClick(Sender: TObject);
begin
  (Sender as TScrollCredits).CurrentLine := 0;
end;


procedure TformAbout.tmrScrollTimer(Sender: TObject);
begin
  scrCredits.CurrentLine := Succ(scrCredits.CurrentLine);
end;

{$IFNDEF LINUX}
procedure TformAbout.WMEraseBkgnd(var Message: TWMEraseBkgnd);
begin
  Message.Result := 1;
end;
{$ENDIF}
type
  TRGB32 = packed Record
    b, g, r, a: byte;
  end;
  PRGB32Array = ^TRGB32Array;
  TRGB32Array = packed array[0..maxint div sizeof(TRGB32)-1] of TRGB32;
  PRGB24 = ^TRGB24;
  TRGB24 = packed Record
    b, g, r: byte;
  end;
  PRGB24Array = ^TRGB24Array;
  TRGB24Array = packed array[0..maxint div sizeof(TRGB24)-1] of TRGB24;

function AddSizeY(var size1: TSize; const size2: TSize): TSize;
begin
  with Result do
  begin
    cx := Max(size1.cx, size2.cx);
    cy := size1.cy + size2.cy;
  end;
  size1 := Result;
end;

function TextToStrings(const s: String): TStrings;
begin
  Result := TStringList.Create;
  Result.Text := s;
end;

{ TScrollCredits }

procedure TScrollCredits.BackgroundChange(Sender: TObject);
begin
  bmpBack.Assign(FBackground.Bitmap);
  bmpBack.PixelFormat := pf32bit;
  bmpBuffer.Width := bmpBack.Width;
  bmpBuffer.Height := bmpBack.Height;
  bmpBuffer.PixelFormat := pf32bit;
  bmpBuffer.Canvas.Draw(0, 0, bmpBack);
  Invalidate;
end;

{$IFDEF LINUX}
procedure TScrollCredits.FontChanged;
{$ELSE}
procedure TScrollCredits.CMFontchanged(var Message: TMessage);
{$ENDIF}
begin
  InvalidateBitmap;
  inherited;
end;

constructor TScrollCredits.Create(AOwner: TComponent);
begin
  inherited;
  FCredits := TCreditStrings.Create(self);
  bmpCredits := TBitmap.Create;
  bmpBack := TBitmap.Create;
  bmpBuffer := TBitmap.Create;
  FBackground := TPicture.Create;
  FBackground.OnChange := BackgroundChange;
  ControlStyle := ControlStyle + [csOpaque];
end;

procedure TScrollCredits.CreditsChange;
begin
  InvalidateBitmap;
  Invalidate;
end;

destructor TScrollCredits.Destroy;
begin
  bmpCredits.Free;
  bmpBack.Free;
  bmpBuffer.Free;
  FCredits.Free;
  FBackground.Free;
  inherited;
end;

{$IFDEF LINUX}
function TScrollCredits.WidgetFlags: Integer;
begin
// To reduce flickering on LINUX
  Result :=
    Inherited WidgetFlags
    or Integer(WidgetFlags_WRepaintNoErase)
    or Integer(WidgetFlags_WResizeNoErase);
end;
{$ENDIF}

procedure TScrollCredits.InvalidateBitmap;
var
  x, i: integer;
  sl: PRGB32Array;
  creditssize: TSize;
  bmpMeasure: TBitmap;
begin
  bmpMeasure := TBitmap.Create;
{$IFDEF LINUX}
  bmpMeasure.Width := 1;
  bmpMeasure.Height := 1;
{$ENDIF}
  creditssize := Credits.TextExtent(bmpMeasure.Canvas);
  bmpMeasure.Free;
  with bmpCredits do
  begin
    Width := creditssize.cx;
    Height := creditssize.cy+10;
    PixelFormat := pf32bit;
    // clear bitmap (and mark it as transparent (or in the case of Kylix: atleast try))
    for i := 0 to Height - 1 do
    begin
      sl := ScanLine[i];
      for x := 0 to Width - 1 do
      begin
        sl[x].r := $7f;
        sl[x].g := $7f;
        sl[x].b := $7f;
        {$IFDEF LINUX}
        sl[x].a := 0;
        {$ELSE}
        sl[x].a := $ff;
        {$ENDIF}
      end;
    end;
    Canvas.Brush.Style := bsClear;
    Credits.RenderText(0, Width, Canvas);
  end;
end;

procedure TScrollCredits.Paint;
begin
  bmpBuffer.Width := Canvas.ClipRect.Right - Canvas.ClipRect.Left; // Width;
  bmpBuffer.Height := Canvas.ClipRect.Bottom - Canvas.ClipRect.Top; //Height;
  TransposeCredits;
  Canvas.CopyRect(Canvas.ClipRect, bmpBuffer.Canvas, Canvas.ClipRect);
end;

procedure TScrollCredits.SetBackground(const Value: TPicture);
begin
  FBackground.Assign(Value);
end;

procedure TScrollCredits.SetCredits(const Value: TCreditStrings);
begin
  FCredits.Assign(Value);
end;

procedure TScrollCredits.SetCurrentLine(const Value: Integer);
begin
  if FCurrentLine > Value then
    bmpBuffer.Canvas.Draw(0, 0, bmpBack);
  if FCurrentLine <> Value then
  begin
    Invalidate;
    FCurrentLine := Value;
  end;
end;

procedure TScrollCredits.TransposeCredits;
var
  x, y,
  w, h, starty, maxh,
  startx: integer;
  transparency: integer;
  targetline, backdropline, sourceline: PRGB32Array;
begin
  h := bmpBuffer.Height;
  starty := Max(h-CurrentLine, 0);
  maxh := Min(h, bmpCredits.Height + h - CurrentLine);
  startx := Max(0, (bmpBuffer.Width - bmpCredits.Width) div 2);
  w := Min(bmpBuffer.Width, bmpCredits.Width + startx);
  for y := starty to maxh-1 do
  begin
    transparency := Round(100*(h - y) / h);
    targetline := bmpBuffer.ScanLine[y];
    sourceline := bmpCredits.ScanLine[y - (h - CurrentLine)];
    if( y >= bmpBack.Height ) then
    begin
     backdropline := bmpBack.Scanline[bmpBack.Height-1];
    end
    else
    begin
      backdropline := bmpBack.ScanLine[y];
    end;
    for x := startx to w-1 do
    {$IFDEF LINUX}
      if sourceline[x-startx].a > $7F then
    {$ELSE}
      if sourceline[x-startx].a < $7F then
    {$ENDIF}
        with backdropline[x] do
        begin
          targetline[x].b := Round((Transparency * b + (100 - Transparency) * sourceline[x-startx].b) * 0.01);
          targetline[x].g := Round((Transparency * g + (100 - Transparency) * sourceline[x-startx].g) * 0.01);
          targetline[x].r := Round((Transparency * r + (100 - Transparency) * sourceline[x-startx].r) * 0.01);
        end
      else
        targetline[x] := backdropline[x];
  end;
end;

{procedure TScrollCredits.TransposeCredits;
var
  x, y,
  w, h, starty, maxh,
  startx: integer;
  transparency: integer;
  targetline: PRGB24Array;
  sourceline: PRGB32Array;
begin
  h := bmpBuffer.Height;
  starty := Max(h-CurrentLine, 0);
  maxh := Min(h, bmpCredits.Height + h - CurrentLine);
  startx := Max(0, (bmpBuffer.Width - bmpCredits.Width) div 2);
  w := Min(bmpBuffer.Width, bmpCredits.Width + startx);
  for y := starty to maxh-1 do
  begin
    transparency := Round(100*(h - y) / h);
    targetline := bmpBuffer.ScanLine[y];
    sourceline := bmpCredits.ScanLine[y - (h - CurrentLine)];
    for x := startx to w-1 do
      if sourceline[x-startx].a < $7F then
        with targetline[x] do
        begin
          b := Round((Transparency * b + (100 - Transparency) * sourceline[x-startx].b) * 0.01);
          g := Round((Transparency * g + (100 - Transparency) * sourceline[x-startx].g) * 0.01);
          r := Round((Transparency * r + (100 - Transparency) * sourceline[x-startx].r) * 0.01);
        end;
  end;
end;}


{ TCreditString }

procedure TCreditString.Assign(Source: TPersistent);
begin
  if Source is TCreditString then
  begin
    FHeader := TCreditString(Source).Header;
    FBody := TCreditString(Source).Body;
  end
  else
    inherited;
end;

procedure TCreditString.BodyChanged(sender: TObject);
begin
  Changed(False);
end;

constructor TCreditString.Create(Collection: TCollection);
begin
  inherited;
  FColCount := 1;
  FBody := TStringList.Create;
  TStringList(FBody).OnChange := BodyChanged;
end;

destructor TCreditString.Destroy;
begin
  FBody.Free;
  inherited;
end;

function TCreditString.GetBodyItems(Index: Integer): String;
begin
  Result := Body[Index];
end;

procedure TCreditString.RenderText(const ATop, AWidth: Integer; Canvas: TCanvas);
var
  cw: Integer;

  procedure TextOut(const X, Y: Integer; const Text: String);
  var SaveColor: TColor;
  begin
    with Canvas do
    begin
      SaveColor := Font.Color;
      Font.Color := clBlack;
      TextOut(x + 2, y + 2, Text);
      Font.Color := clWhite;
      TextOut(x, y, Text);
      Font.Color := SaveColor;
      TextOut(x + 1, y + 1, Text);
    end;
  end;

  procedure CenterText(const X, Y, AWidth: Integer; const Text: String);
  var rx: Integer;
  begin
    rx := Round((AWidth - Canvas.TextWidth(Text)) / 2);
    TextOut(X + rx, Y, Text);
  end;

  procedure DrawColumn(const X, Y, Column: Integer);
  var
    i, li: Integer;
    currenty, lineheight : Integer;
  begin
    if Body.Count = 0 then Exit;
    lineheight := Canvas.TextHeight(Body[0]);
    currenty := Y;
    for i := 0 to Body.Count - 1 do
      if i mod ColCount = Column then
      begin
        with TextToStrings(Body[i]) do
        try
          for li := 0 to Count - 1 do
          begin
            CenterText(X, currenty, cw, Strings[li]);
            currenty := currenty + lineheight;
          end;
        finally
          Free;
        end;
      end;
  end;

  function ColumnWidth: integer;
  begin
    Result := (AWidth - TCreditStrings(Collection).ColSpace * (ColCount-1)) div ColCount;
  end;

var
  i,
  h,
  cs: Integer;
begin
  cw := ColumnWidth;
  cs := TCreditStrings(Collection).ColSpace;
  h := 0;
  if Header <> '' then
  begin
    Canvas.Font := TCreditStrings(Collection).HeaderFont;
    CenterText(0, ATop, AWidth, Header);
    h := Canvas.TextHeight(Header);
  end;
  Canvas.Font := TCreditStrings(Collection).BodyFont;
  for i := 0 to ColCount - 1 do
    DrawColumn((cw+cs)*i, ATop + h, i);
end;

procedure TCreditString.SetBody(const Value: TStrings);
begin
  FBody.Assign(Value);
end;

procedure TCreditString.SetBodyItems(Index: Integer; const Value: String);
begin
  Body[Index] := Value;
end;

procedure TCreditString.SetColCount(const Value: Integer);
begin
  if (FColCount <> Value) and (Value > 0) then
  begin
    FColCount := Value;
    Changed(False);
  end;
end;

procedure TCreditString.SetHeader(const Value: String);
begin
  if Value <> FHeader then
  begin
    FHeader := Value;
    Changed(False);
  end;
end;

function TCreditString.TextExtent(Canvas: TCanvas): TSize;

  function TextExtent(const s: string): TSize;
  var i: integer;
  begin
    FillChar(Result, SizeOf(TSize), 0);
    with TextToStrings(s) do
    try
      for i := 0 to Count - 1 do
        AddSizeY(Result, Canvas.TextExtent(Strings[i]));
    finally
      Free;
    end;
  end;

  function ColumnExtent(const Column: integer): TSize;
  var i: integer;
  begin
    FillChar(Result, SizeOf(TSize), 0);
    for i := 0 to Body.Count - 1 do
      if i mod ColCount = Column then
        AddSizeY(Result, TextExtent(Body[i]));
  end;

var
  i: integer;
  maxcolcx, maxcolcy: integer;
  csize: TSize;
begin
  if (FTextExtent.cx <> 0) or (FTextExtent.cy <> 0) then
    Result := FTextExtent
  else
  begin
    FillChar(Result, SizeOf(TSize), 0);
    maxcolcx := 0; maxcolcy := 0;
    if Header <> '' then
    begin
      Canvas.Font := (Collection as TCreditStrings).HeaderFont;
      Result := TextExtent(Header);
    end;
    Canvas.Font := (Collection as TCreditStrings).BodyFont;
    for i := 0 to ColCount - 1 do
    begin
      csize := ColumnExtent(i);
      maxcolcx := Max(csize.cx, maxcolcx);
      maxcolcy := Max(csize.cy, maxcolcy);
    end;
    i := TCreditStrings(Collection).ColSpace * (ColCount - 1);
    csize.cy := maxcolcy;
    csize.cx := maxcolcx*ColCount + i;
    AddSizeY(Result, csize);
    FTextExtent := Result;
  end;
end;

{ TCreditStrings }

function TCreditStrings.Add: TCreditString;
begin
  result := TCreditString(Inherited Add);
end;

procedure TCreditStrings.Assign(Source: TPersistent);
begin
  BeginUpdate;
  try
    if Source is TCreditStrings then
    begin
      ColSpace := TCreditStrings(Source).ColSpace;
      HeaderFont := TCreditStrings(Source).HeaderFont;
      BodyFont := TCreditStrings(Source).BodyFont;
    end;
    inherited;
  finally
    EndUpdate;
  end;
end;

constructor TCreditStrings.Create(AnOwner: TScrollCredits);
begin
  FOwner := AnOwner;
  inherited Create(AnOwner, TCreditString);
  FBodyFont := TFont.Create;
  FBodyFont.Assign(FOwner.Font);
  FBodyFont.OnChange := FontChange;
  FHeaderFont := TFont.Create;
  FHeaderFont.Assign(FOwner.Font);
  FHeaderFont.OnChange := FontChange;
end;

destructor TCreditStrings.Destroy;
begin
  FHeaderFont.Free;
  FBodyFont.Free;
  inherited;
end;

procedure TCreditStrings.FontChange(Sender: TObject);
begin
  Changed;
end;

function TCreditStrings.GetItems(Index: Integer): TCreditString;
begin
  Result := TCreditString(Inherited Items[Index]);
end;

procedure TCreditStrings.RenderText(const ATop, AWidth: Integer;
  Canvas: TCanvas);
var
  i,
  y : integer;
begin
  y := ATop;
  for i := 0 to Count - 1 do
  begin
    Items[i].RenderText(y, AWidth, Canvas);
    y := y + Items[i].TextExtent(Canvas).cy;
  end;
end;

procedure TCreditStrings.SetBodyFont(const Value: TFont);
begin
  FBodyFont.Assign(Value);
end;

procedure TCreditStrings.SetColSpace(const Value: Integer);
begin
  if FColSpace <> Value then
  begin
    FColSpace := Value;
    Changed;
  end;
end;

procedure TCreditStrings.SetHeaderFont(const Value: TFont);
begin
  FHeaderFont.Assign(Value);
end;

procedure TCreditStrings.SetItems(Index: Integer;
  const Value: TCreditString);
begin
  Inherited Items[Index] := Value;
end;

function TCreditStrings.TextExtent(Canvas: TCanvas): TSize;
var
  i : integer;
begin
  FillChar(Result, SizeOf(TSize), 0);
  for i := 0 to Count - 1 do
    AddSizeY(Result, Items[i].TextExtent(Canvas));
end;

procedure TCreditStrings.Update(Item: TCollectionItem);
var i: integer;
begin
  inherited;
  if Assigned(Item) then
    FillChar(TCreditString(Item).FTextExtent, SizeOf(TSize), 0)
  else
    for i := 0 to Count - 1 do
      FillChar(Items[i].FTextExtent, SizeOf(TSize), 0);
  FOwner.CreditsChange;
end;

procedure TformAbout.FormShow(Sender: TObject);
begin
  tmrScroll.enabled := true;
end;

constructor TformAbout.Create(AOwner: TComponent);
begin
  {We have use CreateNew to bypass Delphi's form streaming
  mechanism.  This is important because we do not keep form data
  in a DFM thus we would get a resource exception if we tried
  to use the Create constructor}
  inherited CreateNew(AOwner);
  Caption := RSAAboutFormCaption;
  {Create the subcontrols now so that the form size can be calculated}
  Panel1 := TPanel.Create(Self);
  Panel1.Parent := Self;
  Panel1.Left := 0;
  Panel1.Top := 412;
  Panel1.Width := 625;
  Panel1.Height := 41;
  Panel1.Align := alBottom;
  Panel1.TabOrder := 0;
  Panel2 := TPanel.Create(Self);
  Panel2.Parent := Self;
  Panel2.Left := 420;
  Panel2.Top := 0;
  Panel2.Width := 205;
  Panel2.Height := 412;
  Panel2.Align := alRight;
  Panel2.TabOrder := 1;

  {We have to create the scroller here to prevent
  problems}
  scrCredits := TScrollCredits.Create(self);
  with scrCredits do
  try
    Credits.BeginUpdate;
    Parent := Self;
    Align := alClient;
    Background.Bitmap.LoadFromResourceName(HInstance, 'TIDABOUTPICTURE');
    Self.Width := Background.Bitmap.Width + Panel2.Width;
    Self.Height := Background.Bitmap.Height + Panel1.Height;
    Self.Position := poScreenCenter;
    {Now for the credits}
    with Credits.HeaderFont do
    begin
      Name := 'Arial';
      Size := 17;
      Color := clRed;
      Style := [fsBold];
    end;
    with Credits.BodyFont do
    begin
      Name := 'Courier New';
      Size := 14;
      Color := clBlue;
      Style := [fsBold];
    end;
    with Credits.Add do
    begin
      {TODO:  Make the company names in a smaller font than the person's name
      and below the person's name}
      Header := RSAAboutCreditsCoordinator;
      Body.Add('Kudzu (Chad Z. Hower)');
      Body.Add('- Nevrona Designs -');
    end;
    with Credits.Add do
    begin
      //TODO: CoCordinator is missing an o
      Header := RSAAboutCreditsCoCordinator;
      Body.Add('Hadi Hariri');
      Body.Add('- Nevrona Designs -');
    end;
    with Credits.Add do
    begin
      ColCount := 2;
      Header := 'The Indy Pit Crew';  // YEAH!
      with Body do
      begin
        Add('Allen O''Neill');
        Add(' ');
        Add('Andrew Neillans');
        Add(' ');
        Add('Andrew Peter Mee');
        Add(' ');
        Add('Don Siders');
        Add(' ');
        Add('Doychin Bondzhev');
        Add('- dSoft-Bulgaria -');
        Add('Gregor Ibic');
        Add('- Intelicom d.o.o. -');
        Add('Idan Cohen');
        Add(' ');
        Add('J. Peter Mugaas');
        Add('- Nevrona Designs -');
        Add('Jan Pedersen');
        Add('- JPSoft DK -');
        Add('Jim Gunkel');
        Add('- Nevrona Designs -');
        Add('Mark Holmes');
        Add(' ');
        Add('Rune Moberg');
        Add(' ');
        Add('Stphane Grobty');
        Add(' ');
        Add('Sergio Perry');
        Add(' ');
      end;
    end;
    with Credits.Add do
    begin
      ColCount := 1;
      Header := 'Documentation Coordinator';
      Body.Add('Don Siders');
    end;
    with Credits.Add do
    begin
      ColCount := 1;
      Header := 'Demos Coordinator';
      Body.Add('Allen O''Neill');
    end;
    with Credits.Add do
    begin
      ColCount := 2;
      Header := 'Retired/Past Contributors';
      with Body do
      begin
         Add('Charles Stack');
      end;
    end;
    OnClick := ScrollClick;
  finally
    Credits.EndUpdate;
  end;
  {Back to form settings}
  Constraints.MaxHeight := Height;
  Constraints.MaxWidth := Width;
  Constraints.MinHeight := Height;
  Constraints.MinWidth := Width;
  {Back to main form settings}
  BorderIcons := [biSystemMenu];
  {$IFDEF  LINUX}

  {$ELSE}

  {$ENDIF}
  Color := clBtnFace;
  Font.Charset := DEFAULT_CHARSET;
  Font.Color := clBtnText;
  Font.Height := -11;
  {$IFNDEF LINUX}
  Font.Name := 'MS Sans Serif';
  BorderStyle := bsDialog;
  {$ELSE}
  Font.Name := 'helvetica';
  BorderStyle := fbsDialog;
  {$ENDIF}
  Font.Style := [];
  OnShow := FormShow;
  PixelsPerInch := 96;
  tmrScroll:= TTimer.Create(Self);
  tmrScroll.Enabled := False;
  tmrScroll.Interval := 25;
  tmrScroll.OnTimer := tmrScrollTimer;
  {Panel1 owned-controls}
  lblPleaseVisitUs:= TLabel.Create(Self);
  lblPleaseVisitUs.Parent := Panel1;
  lblPleaseVisitUs.Left := 8;
  lblPleaseVisitUs.Top := 4;
  lblPleaseVisitUs.Width := 494;
  lblPleaseVisitUs.Height := 17;
  lblPleaseVisitUs.Alignment := taCenter;
  lblPleaseVisitUs.Anchors := [akLeft, akTop, akRight];
  lblPleaseVisitUs.AutoSize := False;
  lblPleaseVisitUs.Font.Charset := DEFAULT_CHARSET;
  lblPleaseVisitUs.Font.Height := -11;
  lblPleaseVisitUs.Font.Style := [];
  lblPleaseVisitUs.ParentFont := False;
  lblPleaseVisitUs.Transparent := True;
  lblPleaseVisitUs.Caption := RSAAboutBoxPleaseVisit;
  lblURL:= TLabel.Create(Self);
  lblURL.Parent := Panel1;
  lblURL.AutoSize := False;
  lblURL.Left := 8;
  lblURL.Top := 20;
  lblURL.Width := 494;
  lblURL.Height := 13;
  {$IFNDEF LINUX}
  lblURL.Cursor := crHandPoint;
  lblURL.Font.Color := clBtnHighlight;
  lblURL.OnClick := lblURLClick;
  lblURL.Font.Style := [fsUnderline];
  {$ENDIF}
  lblURL.Alignment := taCenter;
  lblURL.Anchors := [akLeft, akTop, akRight];
  lblURL.AutoSize := False;
  lblURL.Font.Charset := DEFAULT_CHARSET ;
  lblURL.Font.Height := -11;
  lblURL.ParentFont := False;
  lblURL.Transparent := True;
  lblURL.Caption := RSAAboutBoxIndyWebsite;
  btnOk := TButton.Create(Self);
  btnOk.Parent := Panel1;
  btnOk.Left := 539;
  btnOk.Top := 8;
  btnOk.Width := 75;
  btnOk.Height := 25;
  btnOk.Anchors := [akTop, akRight];
  btnOk.Cancel := True;
  btnOk.Caption := RSAAboutOk;
  btnOk.Default := True;
  btnOk.ModalResult := 1;
  btnOk.TabOrder := 0;
  {Panel2 owned controls}
  bvlDialogBorder := TBevel.Create(AOwner);
  bvlDialogBorder.Parent := Panel2;
  bvlDialogBorder.Left := 1;
  bvlDialogBorder.Top := 1;
  bvlDialogBorder.Width := 203;
  bvlDialogBorder.Height := 410;
  bvlDialogBorder.Align := alClient;
  bvlDialogBorder.Style := bsRaised;
  lblCopyright:= TLabel.Create(Self);
  lblCopyright.Parent := Panel2;
  lblCopyright.Left := 6;
  lblCopyright.Top := 80;
  lblCopyright.Width := 193;
  lblCopyright.Height := 75;
  lblCopyright.Alignment := taCenter;
  lblCopyright.Anchors := [akLeft, akTop, akRight];
  lblCopyright.AutoSize := False;
  lblCopyright.Caption := RSAAboutBoxCopyright;
  lblCopyright.Font.Charset := DEFAULT_CHARSET;
  lblCopyright.Font.Color := clbtnText;
  lblCopyright.Font.Height := -11;
  lblCopyright.Font.Style := [];
  lblCopyright.ParentFont := False;
  lblCopyright.Transparent := True;
  lblCopyright.WordWrap := True;
  lblVersion := TLabel.Create(Self);
  lblVersion.Parent := Panel2;
  lblVersion.Left := 6;
  lblVersion.Top := 56;
  lblVersion.Width := 193;
  lblVersion.Height := 26;
  lblVersion.Alignment := taCenter;
  lblVersion.Anchors := [akLeft, akTop, akRight];
  lblVersion.AutoSize := False;
  lblVersion.Font.Charset := DEFAULT_CHARSET;
  lblVersion.Font.Color := clBlack;
  lblVersion.Font.Height := -13;
  lblVersion.Font.Name := 'Times New Roman';
  lblVersion.Font.Style := [fsBold];
  lblVersion.ParentFont := False;
  lblVersion.Transparent := True;
  lblName:= TLabel.Create(Self);
  lblName.Parent := Panel2;
  lblName.Left := 6;
  lblName.Top := 16;
  lblName.Width := 193;
  lblName.Height := 49;
  lblName.Alignment := taCenter;
  lblName.Anchors := [akLeft, akTop, akRight];
  lblName.AutoSize := False;
  lblName.Font.Charset := DEFAULT_CHARSET;
  lblName.Font.Color := clBtnText;
  lblName.Font.Height := -19;
  lblName.Font.Name := 'Times New Roman';
  lblName.Font.Style := [fsBold];
  lblName.ParentFont := False;
  lblName.Transparent := True;

  Brush.Style := bsClear;
end;

{$IFDEF LINUX}
procedure TformAbout.FormConstrainedResize(Sender: TObject; var MinWidth,
  MinHeight, MaxWidth, MaxHeight: Integer);
begin
  MinHeight := Self.Height;
  MinWidth := Self.Width;
  MaxHeight := Self.Height;
  MaxWidth := Self.Width;
end;
{$ENDIF}

end.
