unit IdMessageCoderMIME;

// for all 3 to 4s:
//// TODO: Predict output sizes and presize outputs, then use move on
// presized outputs when possible, or presize only and reposition if stream

interface

uses
  Classes,
  IdMessageCoder, IdMessage;

type
  TIdMessageDecoderMIME = class(TIdMessageDecoder)
  protected
    FFirstLine: string;
    FBodyEncoded: Boolean;
    FMIMEBoundary: string;
  public
    constructor Create(AOwner: TComponent); reintroduce; overload;
    constructor Create(AOwner: TComponent; ALine: string); reintroduce;
     overload;
    function ReadBody(ADestStream: TStream;
     var AMsgEnd: Boolean): TIdMessageDecoder; override;
    procedure ReadHeader; override;
    //
    property MIMEBoundary: string read FMIMEBoundary write FMIMEBoundary;
    property BodyEncoded: Boolean read FBodyEncoded write FBodyEncoded;
  end;

  TIdMessageDecoderInfoMIME = class(TIdMessageDecoderInfo)
  public
    function CheckForStart(ASender: TIdMessage; ALine: string): TIdMessageDecoder; override;
  end;

  TIdMessageEncoderMIME = class(TIdMessageEncoder)
  public
    procedure Encode(ASrc: TStream; ADest: TStream); override;
  end;

  TIdMessageEncoderInfoMIME = class(TIdMessageEncoderInfo)
  public
    constructor Create; override;
    procedure InitializeHeaders(AMsg: TIdMessage); override;
  end;

const
  IndyMIMEBoundary = '=_NextPart_2rfkindysadvnqw3nerasdf'; {do not localize}
  IndyMultiPartAlternativeBoundary = '=_NextPart_2altrfkindysadvnqw3nerasdf'; {do not localize}
  IndyMultiPartRelatedBoundary = '=_NextPart_2relrfksadvnqindyw3nerasdf'; {do not localize}
  MIMEGenericText = 'text/'; {do not localize}
  MIME7Bit = '7bit'; {do not localize}

implementation

uses
  IdCoder, IdCoderMIME, IdException, IdGlobal, IdResourceStrings, IdCoderQuotedPrintable,
  SysUtils;

{ TIdMessageDecoderInfoMIME }

function TIdMessageDecoderInfoMIME.CheckForStart(ASender: TIdMessage;
 ALine: string): TIdMessageDecoder;
begin
  if AnsiSameText(ALine, '--' + ASender.MIMEBoundary.Boundary) then begin
    Result := TIdMessageDecoderMIME.Create(ASender);
  end else if AnsiSameText(ASender.ContentTransferEncoding, 'base64') or
    AnsiSameText(ASender.ContentTransferEncoding, 'quoted-printable') then begin
      Result := TIdMessageDecoderMIME.Create(ASender, ALine);
  end else begin
    Result := nil;
  end;
end;

{ TIdCoderMIME }

constructor TIdMessageDecoderMIME.Create(AOwner: TComponent);
begin
  inherited;
  if AOwner is TIdMessage then begin
    if Length(TIdMessage(AOwner).ContentTransferEncoding) > 0 then begin
      FBodyEncoded := True;
    end else begin
      FMIMEBoundary := TIdMessage(AOwner).MIMEBoundary.Boundary;
      FBodyEncoded := False;
    end;
  end;
end;

constructor TIdMessageDecoderMIME.Create(AOwner: TComponent;
  ALine: string);
begin
  Create(AOwner);
  FFirstLine := ALine;
end;

function TIdMessageDecoderMIME.ReadBody(ADestStream: TStream; var AMsgEnd: Boolean): TIdMessageDecoder;
var
  s: string;
  LDecoder: TIdDecoder;
  LLine: string;
begin
  AMsgEnd := False;
  Result := nil;
  if FBodyEncoded then begin
    s := TIdMessage(Owner).ContentTransferEncoding;
  end else begin
    S := FHeaders.Values['Content-Transfer-Encoding'];
  end;
  if AnsiSameText(s, 'base64') then begin
    LDecoder := TIdDecoderMIME.Create(nil);
  end else if AnsiSameText(s, 'quoted-printable') then begin
    LDecoder := TIdDecoderQuotedPrintable.Create(nil);
  end else begin
    LDecoder := nil;
  end;
  try
    repeat
      if Length(FFirstLine) = 0 then begin // TODO: Improve this. Not very efficient
        LLine := ReadLn;
      end else begin
        LLine := FFirstLine;
        FFirstLine := '';
      end;
      if LLine = '.' then begin // ADELIM not needed since always ends with . (standard)
        AMsgEnd := True;
        break;
      end;
      // New boundary - end self and create new coder
      if AnsiSameText(LLine, '--' + FMIMEBoundary) then begin
        Result := TIdMessageDecoderMIME.Create(Owner);
        Break;
      // End of all coders (not quite ALL coders)
      end else if AnsiSameText(LLine, '--' + FMIMEBoundary + '--') then begin
        // POP the boundary
        if Owner is TIdMessage then begin
          TIdMessage(Owner).MIMEBoundary.Pop;
        end;
        Break;
      // Data to save, but not decode
      end else if LDecoder = nil then begin
        if (Length(LLine) > 0) and (LLine[1] = '.') then begin // Process . in front for no encoding
          Delete(LLine, 1, 1);
        end;
        LLine := LLine + EOL;
        ADestStream.WriteBuffer(LLine[1], Length(LLine));
      // Data to decode
      end
      else
      begin
        //for TIdDecoderQuotedPrintable, we have
        //to make sure all EOLs are intact
        if LDecoder is TIdDecoderQuotedPrintable then
        begin
          LDecoder.DecodeToStream(LLine+EOL,ADestStream);
        end
        else
        begin
          if Length(LLine) > 0 then begin
            LDecoder.DecodeToStream(LLine, ADestStream);
          end;
        end;
      end;
    until False;
  finally FreeAndNil(LDecoder); end;
end;

procedure TIdMessageDecoderMIME.ReadHeader;
var
  ABoundary,
  s: string;
  LLine: string;

  procedure CheckAndSetType(AContentType, AContentDisposition: string);
  var
    s: string;
  begin
    s := AContentDisposition;
    s := Fetch(s, ';');
    if (AnsiSameText(s, 'attachment')) or (IndyPos('NAME', UpperCase(AContentType)) > 0) then begin
      FPartType := ptAttachment;
      s := AContentDisposition;
      s := Copy(s, IndyPos('FILENAME', UpperCase(s)) + 9, Length(s));
      if Length(s) = 0 then begin
        // Get filename from Content-Type
        s := AContentType;
        s := Copy(s, IndyPos('NAME', UpperCase(s)) + 5, Length(s));
      end;
      if Length(s) > 0 then begin
        if s[1] = '"' then begin
          Fetch(s, '"');
          FFilename := Fetch(s, '"');
        end else begin
          FFilename := s;
        end;
      end;
    end else begin
      FPartType := ptText;
    end;
  end;

begin
  if FBodyEncoded then begin // Read header from the actual message since body parts don't exist
    CheckAndSetType(TIdMessage(Owner).ContentType, TIdMessage(OWner).ContentDisposition);
  end else begin
    // Read header
    repeat
      LLine := ReadLn;
      if Length(LLine) = 0 then begin
        Break;
      end;
      if LLine[1] in LWS then begin
        FHeaders[FHeaders.Count - 1] := FHeaders[FHeaders.Count - 1] + ' ' + Copy(LLine, 2, MaxInt);
      end else begin
        FHeaders.Add(StringReplace(LLine, ': ', '=', []));
      end;
    until False;
    s := FHeaders.Values['Content-Type'];
    ABoundary := TIdMIMEBoundary.FindBoundary(s);
    if Length(ABoundary) > 0 then begin
      if Owner is TIdMessage then begin
        TIdMessage(Owner).MIMEBoundary.Push(ABoundary);
        // Also update current boundary
        FMIMEBoundary := ABoundary;
      end;
    end;
    CheckAndSetType(FHeaders.Values['Content-Type']
     , FHeaders.Values['Content-Disposition']);
  end;
end;

{ TIdMessageEncoderInfoMIME }

constructor TIdMessageEncoderInfoMIME.Create;
begin
  inherited;
  FMessageEncoderClass := TIdMessageEncoderMIME;
end;

procedure TIdMessageEncoderInfoMIME.InitializeHeaders(AMsg: TIdMessage);
begin
  if AMsg.MessageParts.RelatedPartCount > 0 then begin
    AMsg.ContentType
     := 'multipart/related; type="multipart/alternative"; boundary="' + {do not localize}
     IndyMultiPartRelatedBoundary + '"';
  end else begin
    if AMsg.MessageParts.AttachmentCount > 0 then begin
      AMsg.ContentType := 'multipart/mixed; boundary="' {do not localize}
       + IndyMIMEBoundary + '"';
    end else begin
      if AMsg.MessageParts.TextPartCount > 0 then begin
        AMsg.ContentType :=
         'multipart/alternative; boundary="' {do not localize}
         + IndyMIMEBoundary + '"';
      end;
    end;
  end;
end;

{ TIdMessageEncoderMIME }

procedure TIdMessageEncoderMIME.Encode(ASrc, ADest: TStream);
var
  s: string;
  LEncoder: TIdEncoderMIME;
  LSPos, LSSize : Int64;

begin
  ASrc.Position := 0;
  LSPos := 0;
  LSSize := ASrc.Size;
  LEncoder := TIdEncoderMIME.Create(nil); try
    while LSPos < LSSize do begin
      s := LEncoder.Encode(ASrc, 57) + EOL;
      Inc(LSPos,57);
      ADest.WriteBuffer(s[1], Length(s));
    end;
  finally FreeAndNil(LEncoder); end;
end;

initialization
  TIdMessageDecoderList.RegisterDecoder('MIME'
   , TIdMessageDecoderInfoMIME.Create);
  TIdMessageEncoderList.RegisterEncoder('MIME'
   , TIdMessageEncoderInfoMIME.Create);
end.
